/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { LitElement, html } from 'lit';
import { property } from 'lit/decorators.js';
import { classMap } from 'lit/directives/class-map.js';
import CheckmarkFilled16 from '../../icons/checkmark--filled/16.js';
import ErrorFilled16 from '../../icons/error--filled/16.js';
import { prefix } from '../../globals/settings.js';
import LOADING_TYPE from '../loading/types.js';
import getLoadingIcon from '../loading/loading-icon.js';
import { INLINE_LOADING_STATE } from './defs.js';
export { INLINE_LOADING_STATE } from './defs.js';
import styles from './inline-loading.scss.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2023
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Lnline loading spinner.
 *
 * @element cds-inline-loading
 */
let CDSInlineLoading = class CDSInlineLoading extends LitElement {
    constructor() {
        super(...arguments);
        /**
         * The assistive text for the spinner icon.
         */
        this.assistiveText = 'Loading';
        /**
         * The loading status.
         */
        this.status = INLINE_LOADING_STATE.ACTIVE;
    }
    /**
     * @returns The template for the status icon.
     */
    _renderIcon() {
        const { assistiveText, status } = this;
        if (status === INLINE_LOADING_STATE.ERROR) {
            return ErrorFilled16({
                class: `${prefix}--inline-loading--error`,
            });
        }
        if (status === INLINE_LOADING_STATE.FINISHED) {
            return CheckmarkFilled16({
                class: `${prefix}--inline-loading__checkmark-container`,
            });
        }
        if (status === INLINE_LOADING_STATE.INACTIVE ||
            status === INLINE_LOADING_STATE.ACTIVE) {
            const classes = classMap({
                [`${prefix}--loading`]: true,
                [`${prefix}--loading--small`]: true,
                [`${prefix}--loading--stop`]: status === INLINE_LOADING_STATE.INACTIVE,
            });
            return html `
        <div class="${classes}">
          ${getLoadingIcon({ assistiveText, type: LOADING_TYPE.SMALL })}
        </div>
      `;
        }
        return undefined;
    }
    connectedCallback() {
        if (!this.hasAttribute('aria-live')) {
            this.setAttribute('aria-live', 'assertive');
        }
        super.connectedCallback();
    }
    render() {
        const statusIconResult = this._renderIcon();
        const statusIconWrapperResult = !statusIconResult
            ? undefined
            : html `
          <div class="${prefix}--inline-loading__animation">
            ${statusIconResult}
          </div>
        `;
        return html `
      ${statusIconWrapperResult}
      <p class="${prefix}--inline-loading__text"><slot></slot></p>
    `;
    }
};
CDSInlineLoading.styles = styles; // `styles` here is a `CSSResult` generated by custom Vite loader
__decorate([
    property({ attribute: 'assistive-text' })
], CDSInlineLoading.prototype, "assistiveText", void 0);
__decorate([
    property({ reflect: true })
], CDSInlineLoading.prototype, "status", void 0);
CDSInlineLoading = __decorate([
    carbonElement(`${prefix}-inline-loading`)
], CDSInlineLoading);
var CDSInlineLoading$1 = CDSInlineLoading;

export { CDSInlineLoading$1 as default };
//# sourceMappingURL=inline-loading.js.map
