/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { classMap } from 'lit/directives/class-map.js';
import { ifDefined } from 'lit/directives/if-defined.js';
import { LitElement, html } from 'lit';
import { query, property } from 'lit/decorators.js';
import { prefix } from '../../globals/settings.js';
import FocusMixin from '../../globals/mixins/focus.js';
import styles from './link.scss.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2023
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Link size.
 */
const LINK_SIZE = {
    MEDIUM: 'md',
    SMALL: 'sm',
    LARGE: 'lg',
};
/**
 * Link.
 *
 * @element cds-link
 * @csspart link The link.
 */
let CDSLink = class CDSLink extends FocusMixin(LitElement) {
    constructor() {
        super(...arguments);
        /**
         * `true` if there is an icon.
         */
        this._hasIcon = false;
        /**
         * `true` if the link should be disabled.
         */
        this.disabled = false;
        /**
         * `true` if the link should be inline.
         */
        this.inline = false;
        /**
         * Link size.
         */
        this.size = LINK_SIZE.MEDIUM;
        /**
         * `true` if the link has been visited.
         */
        this.visited = false;
    }
    /**
     * Handles `slotchange` event.
     */
    _handleSlotChange({ target }) {
        const { name } = target;
        const hasContent = target
            .assignedNodes()
            .some((node) => node.nodeType !== Node.TEXT_NODE || node.textContent.trim());
        this[name === 'icon' ? '_hasIcon' : ''] = hasContent;
        this.requestUpdate();
    }
    /**
     * The CSS class list for the link node.
     */
    get _classes() {
        const { disabled, size, inline, visited, _hasIcon } = this;
        return classMap({
            [`${prefix}--link`]: true,
            [`${prefix}--link--disabled`]: disabled,
            [`${prefix}--link--icon`]: _hasIcon,
            [`${prefix}--link--inline`]: inline,
            [`${prefix}--link--${size}`]: size,
            [`${prefix}--link--visited`]: visited,
        });
    }
    /**
     * Handles `click` event on the `<a>`.
     */
    // eslint-disable-next-line class-methods-use-this, @typescript-eslint/no-unused-vars
    _handleClick(_) { }
    /**
     * @returns The inner content.
     */
    // eslint-disable-next-line class-methods-use-this
    _renderInner() {
        const { _hasIcon: hasIcon, _handleSlotChange: handleSlotChange } = this;
        return html `
      <slot @slotchange="${handleSlotChange}"></slot>
      <div ?hidden="${!hasIcon}" class="${prefix}--link__icon">
        <slot name="icon" @slotchange="${handleSlotChange}"></slot>
      </div>
    `;
    }
    /**
     * @returns The disabled link content.
     */
    _renderDisabledLink() {
        const { _classes: classes } = this;
        return html `
      <p id="link" part="link" class="${classes}">${this._renderInner()}</p>
    `;
    }
    /**
     * @returns The link content.
     */
    _renderLink() {
        const { download, href, hreflang, linkRole, ping, rel, target, type, _classes: classes, _handleClick: handleClick, } = this;
        return html `
      <a
        tabindex="0"
        id="link"
        role="${ifDefined(linkRole)}"
        class="${classes}"
        part="link"
        download="${ifDefined(download)}"
        href="${ifDefined(href)}"
        hreflang="${ifDefined(hreflang)}"
        ping="${ifDefined(ping)}"
        rel="${ifDefined(rel)}"
        target="${ifDefined(target)}"
        type="${ifDefined(type)}"
        @click="${handleClick}">
        ${this._renderInner()}
      </a>
    `;
    }
    render() {
        const { disabled } = this;
        return disabled ? this._renderDisabledLink() : this._renderLink();
    }
};
CDSLink.shadowRootOptions = Object.assign(Object.assign({}, LitElement.shadowRootOptions), { delegatesFocus: true });
CDSLink.styles = styles; // `styles` here is a `CSSResult` generated by custom Vite loader
__decorate([
    query('#link')
], CDSLink.prototype, "_linkNode", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSLink.prototype, "disabled", void 0);
__decorate([
    property({ reflect: true })
], CDSLink.prototype, "download", void 0);
__decorate([
    property({ reflect: true })
], CDSLink.prototype, "href", void 0);
__decorate([
    property({ reflect: true })
], CDSLink.prototype, "hreflang", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSLink.prototype, "inline", void 0);
__decorate([
    property({ attribute: 'link-role' })
], CDSLink.prototype, "linkRole", void 0);
__decorate([
    property({ reflect: true })
], CDSLink.prototype, "ping", void 0);
__decorate([
    property({ reflect: true })
], CDSLink.prototype, "rel", void 0);
__decorate([
    property({ reflect: true })
], CDSLink.prototype, "size", void 0);
__decorate([
    property({ reflect: true })
], CDSLink.prototype, "target", void 0);
__decorate([
    property({ reflect: true })
], CDSLink.prototype, "type", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSLink.prototype, "visited", void 0);
CDSLink = __decorate([
    carbonElement(`${prefix}-link`)
], CDSLink);
var CDSLink$1 = CDSLink;

export { LINK_SIZE, CDSLink$1 as default };
//# sourceMappingURL=link.js.map
