/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { classMap } from 'lit/directives/class-map.js';
import { html, LitElement } from 'lit';
import { query, property } from 'lit/decorators.js';
import { prefix, selectorTabbable } from '../../globals/settings.js';
import HostListener from '../../globals/decorators/host-listener.js';
import HostListenerMixin from '../../globals/mixins/host-listener.js';
import { MODAL_SIZE } from './defs.js';
export { MODAL_SIZE } from './defs.js';
import styles from './modal.scss.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
// eslint-disable-next-line no-bitwise
const PRECEDING = Node.DOCUMENT_POSITION_PRECEDING | Node.DOCUMENT_POSITION_CONTAINS;
// eslint-disable-next-line no-bitwise
const FOLLOWING = Node.DOCUMENT_POSITION_FOLLOWING | Node.DOCUMENT_POSITION_CONTAINED_BY;
/**
 * Tries to focus on the given elements and bails out if one of them is successful.
 *
 * @param elems The elements.
 * @param reverse `true` to go through the list in reverse order.
 * @returns `true` if one of the attempts is successful, `false` otherwise.
 */
function tryFocusElems(elems, reverse = false) {
    if (!reverse) {
        for (let i = 0; i < elems.length; ++i) {
            const elem = elems[i];
            elem.focus();
            if (elem.ownerDocument.activeElement === elem) {
                return true;
            }
        }
    }
    else {
        for (let i = elems.length - 1; i >= 0; --i) {
            const elem = elems[i];
            elem.focus();
            if (elem.ownerDocument.activeElement === elem) {
                return true;
            }
        }
    }
    return false;
}
/**
 * Modal.
 *
 * @element cds-modal
 * @csspart dialog The dialog.
 * @fires cds-modal-beingclosed
 *   The custom event fired before this modal is being closed upon a user gesture.
 *   Cancellation of this event stops the user-initiated action of closing this modal.
 * @fires cds-modal-closed - The custom event fired after this modal is closed upon a user gesture.
 */
let CDSModal = class CDSModal extends HostListenerMixin(LitElement) {
    constructor() {
        super(...arguments);
        /**
         * The element that had focus before this modal gets open.
         */
        this._launcher = null;
        /**
         * Handles `click` event on this element.
         *
         * @param event The event.
         */
        this._handleClick = (event) => {
            if (event.composedPath().indexOf(this.shadowRoot) < 0 &&
                !this.preventCloseOnClickOutside) {
                this._handleUserInitiatedClose(event.target);
            }
        };
        /**
         * Handles `blur` event on this element.
         *
         * @param event The event.
         * @param event.target The event target.
         * @param event.relatedTarget The event relatedTarget.
         */
        this._handleBlur = async ({ target, relatedTarget }) => {
            var _a;
            const { open, _startSentinelNode: startSentinelNode, _endSentinelNode: endSentinelNode, } = this;
            const oldContains = target !== this && this.contains(target);
            const currentContains = relatedTarget !== this &&
                (this.contains(relatedTarget) ||
                    (((_a = this.shadowRoot) === null || _a === void 0 ? void 0 : _a.contains(relatedTarget)) &&
                        relatedTarget !== endSentinelNode));
            // Performs focus wrapping if _all_ of the following is met:
            // * This modal is open
            // * The viewport still has focus
            // * Modal body used to have focus but no longer has focus
            const { selectorTabbable: selectorTabbableForModal } = this
                .constructor;
            if (open && relatedTarget && oldContains && !currentContains) {
                const comparisonResult = target.compareDocumentPosition(relatedTarget);
                // eslint-disable-next-line no-bitwise
                if (relatedTarget === startSentinelNode || comparisonResult & PRECEDING) {
                    await this.constructor._delay();
                    if (!tryFocusElems(this.querySelectorAll(selectorTabbableForModal), true) &&
                        relatedTarget !== this) {
                        this.focus();
                    }
                }
                // eslint-disable-next-line no-bitwise
                else if (relatedTarget === endSentinelNode ||
                    comparisonResult & FOLLOWING) {
                    await this.constructor._delay();
                    if (!tryFocusElems(this.querySelectorAll(selectorTabbableForModal))) {
                        this.focus();
                    }
                }
            }
        };
        this._handleKeydown = ({ key, target }) => {
            if (key === 'Esc' || key === 'Escape') {
                this._handleUserInitiatedClose(target);
            }
        };
        /**
         * Specify whether the Modal is displaying an alert, error or warning.
         * Should go hand in hand with the danger prop.
         */
        this.alert = false;
        /**
         * Specify text for the accessibility label of the header
         */
        this.ariaLabel = '';
        /**
         * The additional CSS class names for the container <div> of the element.
         */
        this.containerClass = '';
        /**
         * Specify whether or not the Modal content should have any inner padding.
         */
        this.fullWidth = false;
        /**
         * Specify whether the modal contains scrolling content
         */
        this.hasScrollingContent = false;
        /**
         * `true` if the modal should be open.
         */
        this.open = false;
        /**
         * Modal size.
         */
        this.size = MODAL_SIZE.MEDIUM;
        /**
         * Prevent closing on click outside of modal
         */
        this.preventCloseOnClickOutside = false;
        /**
         * Prevent the modal from closing after clicking the close button
         */
        this.preventClose = false;
    }
    /**
     * Handles `click` event on the modal container.
     *
     * @param event The event.
     */
    _handleClickContainer(event) {
        if (event.target.matches(this.constructor.selectorCloseButton) &&
            !this.preventClose) {
            this._handleUserInitiatedClose(event.target);
        }
    }
    /**
     * Handles user-initiated close request of this modal.
     *
     * @param triggeredBy The element that triggered this close request.
     */
    _handleUserInitiatedClose(triggeredBy) {
        if (this.open) {
            const init = {
                bubbles: true,
                cancelable: true,
                composed: true,
                detail: {
                    triggeredBy,
                },
            };
            if (this.dispatchEvent(new CustomEvent(this.constructor.eventBeforeClose, init))) {
                this.open = false;
                this.dispatchEvent(new CustomEvent(this.constructor.eventClose, init));
            }
        }
    }
    /**
     * Handles `slotchange` event.
     */
    _handleSlotChange() {
        this.querySelector(`${prefix}-modal-footer`)
            ? this.setAttribute('has-footer', '')
            : this.removeAttribute('has-footer');
    }
    firstUpdated() {
        const body = this.querySelector(this.constructor.selectorModalBody);
        if (!body) {
            const bodyElement = document.createElement(this.constructor.selectorModalBody);
            this.appendChild(bodyElement);
        }
    }
    render() {
        const { alert, ariaLabel, size, hasScrollingContent } = this;
        const containerClass = this.containerClass
            .split(' ')
            .filter(Boolean)
            .reduce((acc, item) => (Object.assign(Object.assign({}, acc), { [item]: true })), {});
        const containerClasses = classMap(Object.assign({ [`${prefix}--modal-container`]: true, [`${prefix}--modal-container--${size}`]: size }, containerClass));
        return html `
      <a
        id="start-sentinel"
        class="${prefix}--visually-hidden"
        href="javascript:void 0"
        role="navigation"></a>
      <div
        aria-label=${ariaLabel}
        part="dialog"
        class=${containerClasses}
        role="${alert ? 'alert' : 'dialog'}"
        tabindex="-1"
        @click=${this._handleClickContainer}>
        <slot @slotchange="${this._handleSlotChange}"></slot>
        ${hasScrollingContent
            ? html ` <div class="cds--modal-content--overflow-indicator"></div> `
            : ``}
      </div>
      <a
        id="end-sentinel"
        class="${prefix}--visually-hidden"
        href="javascript:void 0"
        role="navigation"></a>
    `;
    }
    async updated(changedProperties) {
        if (changedProperties.has('open')) {
            if (this.open) {
                this._launcher = this.ownerDocument.activeElement;
                const primaryFocusNode = this.querySelector(this.constructor.selectorPrimaryFocus);
                await this.constructor._delay();
                if (primaryFocusNode) {
                    // For cases where a `carbon-web-components` component (e.g. `<cds-button>`) being `primaryFocusNode`,
                    // where its first update/render cycle that makes it focusable happens after `<cds-modal>`'s first update/render cycle
                    primaryFocusNode.focus();
                }
                else if (!tryFocusElems(this.querySelectorAll(this.constructor.selectorTabbable), true)) {
                    this.focus();
                }
            }
            else if (this._launcher &&
                typeof this._launcher.focus === 'function') {
                this._launcher.focus();
                this._launcher = null;
            }
        }
    }
    /**
     * @param ms The number of milliseconds.
     * @returns A promise that is resolves after the given milliseconds.
     */
    static _delay(ms = 0) {
        return new Promise((resolve) => {
            setTimeout(resolve, ms);
        });
    }
    /**
     * A selector selecting buttons that should close this modal.
     */
    static get selectorCloseButton() {
        return `[data-modal-close],${prefix}-modal-close-button`;
    }
    /**
     * A selector selecting tabbable nodes.
     */
    static get selectorTabbable() {
        return selectorTabbable;
    }
    /**
     * A selector selecting the nodes that should be focused when modal gets open.
     */
    static get selectorPrimaryFocus() {
        return `[data-modal-primary-focus],${prefix}-modal-footer ${prefix}-button[kind="primary"]`;
    }
    /**
     * A selector selecting the modal body component
     */
    static get selectorModalBody() {
        return `${prefix}-modal-body`;
    }
    /**
     * The name of the custom event fired before this modal is being closed upon a user gesture.
     * Cancellation of this event stops the user-initiated action of closing this modal.
     */
    static get eventBeforeClose() {
        return `${prefix}-modal-beingclosed`;
    }
    /**
     * The name of the custom event fired after this modal is closed upon a user gesture.
     */
    static get eventClose() {
        return `${prefix}-modal-closed`;
    }
};
CDSModal.styles = styles; // `styles` here is a `CSSResult` generated by custom Vite loader
__decorate([
    query('#start-sentinel')
], CDSModal.prototype, "_startSentinelNode", void 0);
__decorate([
    query('#end-sentinel')
], CDSModal.prototype, "_endSentinelNode", void 0);
__decorate([
    HostListener('click')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSModal.prototype, "_handleClick", void 0);
__decorate([
    HostListener('shadowRoot:focusout')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSModal.prototype, "_handleBlur", void 0);
__decorate([
    HostListener('document:keydown')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSModal.prototype, "_handleKeydown", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSModal.prototype, "alert", void 0);
__decorate([
    property({ attribute: 'aria-label' })
], CDSModal.prototype, "ariaLabel", void 0);
__decorate([
    property({ attribute: 'container-class' })
], CDSModal.prototype, "containerClass", void 0);
__decorate([
    property({ type: Boolean, reflect: true, attribute: 'full-width' })
], CDSModal.prototype, "fullWidth", void 0);
__decorate([
    property({
        type: Boolean,
        reflect: true,
        attribute: 'has-scrolling-content',
    })
], CDSModal.prototype, "hasScrollingContent", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSModal.prototype, "open", void 0);
__decorate([
    property({ reflect: true })
], CDSModal.prototype, "size", void 0);
__decorate([
    property({ type: Boolean, attribute: 'prevent-close-on-click-outside' })
], CDSModal.prototype, "preventCloseOnClickOutside", void 0);
__decorate([
    property({ type: Boolean, attribute: 'prevent-close' })
], CDSModal.prototype, "preventClose", void 0);
CDSModal = __decorate([
    carbonElement(`${prefix}-modal`)
], CDSModal);
var CDSModal$1 = CDSModal;

export { CDSModal$1 as default };
//# sourceMappingURL=modal.js.map
