/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import CheckmarkFilled20 from '../../icons/checkmark--filled/20.js';
import ErrorFilled20 from '../../icons/error--filled/20.js';
import InformationFilled20 from '../../icons/information--filled/20.js';
import InformationSquareFilled20 from '../../icons/information--square--filled/20.js';
import WarningFilled20 from '../../icons/warning--filled/20.js';
import WarningAltFilled20 from '../../icons/warning--alt--filled/20.js';
import { svg, html } from 'lit';
import { property } from 'lit/decorators.js';
import { prefix } from '../../globals/settings.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';
import { NOTIFICATION_KIND, NOTIFICATION_TYPE } from './defs.js';
import CDSInlineNotification from './inline-notification.js';
import styles from './actionable-notification.scss.js';
import HostListener from '../../globals/decorators/host-listener.js';
import HostListenerMixin from '../../globals/mixins/host-listener.js';

/**
 * The default icons, keyed by notification kind.
 */
const iconsForKinds = {
    [NOTIFICATION_KIND.SUCCESS]: CheckmarkFilled20,
    [NOTIFICATION_KIND.INFO]: InformationFilled20,
    [NOTIFICATION_KIND.INFO_SQUARE]: InformationSquareFilled20,
    [NOTIFICATION_KIND.WARNING]: WarningFilled20,
    [NOTIFICATION_KIND.WARNING_ALT]: WarningAltFilled20,
    [NOTIFICATION_KIND.ERROR]: ErrorFilled20,
};
/**
 * Actionable notification.
 *
 * @element cds-actionable-notification
 * @slot subtitle - The subtitle.
 * @slot title - The title.
 * @fires cds-notification-beingclosed
 *   The custom event fired before this notification is being closed upon a user gesture.
 *   Cancellation of this event stops the user-initiated action of closing this notification.
 * @fires cds-notification-closed - The custom event fired after this notification is closed upon a user gesture.
 */
let CDSActionableNotification = class CDSActionableNotification extends HostListenerMixin(CDSInlineNotification) {
    constructor() {
        super(...arguments);
        this._type = NOTIFICATION_TYPE.ACTIONABLE;
        /**
         * Inline notification type.
         */
        this.inline = false;
        /**
         * Pass in the action button label that will be rendered within the ActionableNotification.
         */
        this.actionButtonLabel = '';
        /**
         * Specify if pressing the escape key should close notifications
         */
        this.closeOnEscape = true;
        /**
         * Specify if focus should be moved to the component when the notification contains actions
         */
        this.hasFocus = true;
        /**
         * Handles `keydown` event on this event.
         * Escape will close the notification if `closeOnEscape` is true
         */
        this._handleKeyDown = async (event) => {
            const { key } = event;
            if (this.closeOnEscape && key === 'Escape') {
                this.open = false;
            }
        };
        /**
         * The caption.
         */
        this.caption = '';
    }
    connectedCallback() {
        if (!this.hasAttribute('role')) {
            this.setAttribute('role', 'alertdialog');
        }
        super.connectedCallback();
    }
    _renderIcon() {
        const { statusIconDescription, kind, inline } = this;
        const { [kind]: icon } = iconsForKinds;
        return !icon
            ? undefined
            : icon({
                class: `${prefix}--${inline ? 'inline' : 'toast'}-notification__icon`,
                children: !statusIconDescription
                    ? undefined
                    : svg `<title>${statusIconDescription}</title>`,
            });
    }
    _renderText() {
        const { subtitle, title, _type: type } = this;
        return html `
      <div class="${prefix}--${type}-notification__text-wrapper">
        <div class="${prefix}--${type}-notification__content">
          <div class="${prefix}--${type}-notification__title">
            ${title}<slot name="title"></slot>
          </div>
          <div class="${prefix}--${type}-notification__subtitle">
            ${subtitle}<slot name="subtitle"></slot>
          </div>
          <slot></slot>
        </div>
      </div>
    `;
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        const button = this.querySelector(this.constructor
            .selectorActionButton);
        if (changedProperties.has('inline')) {
            button === null || button === void 0 ? void 0 : button.setAttribute('kind', this.inline ? 'ghost' : 'tertiary');
        }
        if (changedProperties.has('lowContrast')) {
            if (this.lowContrast) {
                button === null || button === void 0 ? void 0 : button.setAttribute('low-contrast', 'true');
            }
            else {
                button === null || button === void 0 ? void 0 : button.removeAttribute('low-contrast');
            }
        }
        if (changedProperties.has('hideCloseButton')) {
            if (this.hideCloseButton) {
                button === null || button === void 0 ? void 0 : button.setAttribute('hide-close-button', 'true');
            }
            else {
                button === null || button === void 0 ? void 0 : button.removeAttribute('hide-close-button');
            }
        }
        if (changedProperties.has('hasFocus')) {
            if (this.hasFocus) {
                this.focus();
            }
        }
    }
    render() {
        const { _type: type } = this;
        return html `
      <div class="${prefix}--${type}-notification__details">
        ${this._renderIcon()} ${this._renderText()}
      </div>
      <slot name="action"></slot>
      ${this._renderButton()}
    `;
    }
    /**
     * A selector that will return the action button element
     */
    static get selectorActionButton() {
        return `${prefix}-actionable-notification-button`;
    }
};
CDSActionableNotification.styles = styles; // `styles` here is a `CSSResult` generated by custom Vite loader
__decorate([
    property({ type: Boolean, reflect: true })
], CDSActionableNotification.prototype, "inline", void 0);
__decorate([
    property({ type: String, reflect: true, attribute: 'action-button-label' })
], CDSActionableNotification.prototype, "actionButtonLabel", void 0);
__decorate([
    property({ type: Boolean, reflect: true, attribute: 'close-on-escape' })
], CDSActionableNotification.prototype, "closeOnEscape", void 0);
__decorate([
    property({ type: Boolean, reflect: true, attribute: 'has-focus' })
], CDSActionableNotification.prototype, "hasFocus", void 0);
__decorate([
    HostListener('keydown')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSActionableNotification.prototype, "_handleKeyDown", void 0);
__decorate([
    property()
], CDSActionableNotification.prototype, "caption", void 0);
CDSActionableNotification = __decorate([
    carbonElement(`${prefix}-actionable-notification`)
], CDSActionableNotification);
var CDSActionableNotification$1 = CDSActionableNotification;

export { CDSActionableNotification$1 as default };
//# sourceMappingURL=actionable-notification.js.map
