/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import CheckmarkFilled20 from '../../icons/checkmark--filled/20.js';
import Close16 from '../../icons/close/16.js';
import ErrorFilled20 from '../../icons/error--filled/20.js';
import InformationFilled20 from '../../icons/information--filled/20.js';
import InformationSquareFilled20 from '../../icons/information--square--filled/20.js';
import WarningFilled20 from '../../icons/warning--filled/20.js';
import WarningAltFilled20 from '../../icons/warning--alt--filled/20.js';
import { LitElement, html, svg } from 'lit';
import { property } from 'lit/decorators.js';
import { ifDefined } from 'lit/directives/if-defined.js';
import { prefix } from '../../globals/settings.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';
import FocusMixin from '../../globals/mixins/focus.js';
import { NOTIFICATION_KIND, NOTIFICATION_TYPE } from './defs.js';
export { NOTIFICATION_KIND, NOTIFICATION_TYPE } from './defs.js';
import styles from './inline-notification.scss.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * The default icons, keyed by notification kind.
 */
const iconsForKinds = {
    [NOTIFICATION_KIND.SUCCESS]: CheckmarkFilled20,
    [NOTIFICATION_KIND.INFO]: InformationFilled20,
    [NOTIFICATION_KIND.INFO_SQUARE]: InformationSquareFilled20,
    [NOTIFICATION_KIND.WARNING]: WarningFilled20,
    [NOTIFICATION_KIND.WARNING_ALT]: WarningAltFilled20,
    [NOTIFICATION_KIND.ERROR]: ErrorFilled20,
};
/**
 * Inline notification.
 *
 * @element cds-inline-notification
 * @slot subtitle - The subtitle.
 * @slot title - The title.
 * @fires cds-notification-beingclosed
 *   The custom event fired before this notification is being closed upon a user gesture.
 *   Cancellation of this event stops the user-initiated action of closing this notification.
 * @fires cds-notification-closed - The custom event fired after this notification is closed upon a user gesture.
 */
let CDSInlineNotification = class CDSInlineNotification extends FocusMixin(LitElement) {
    constructor() {
        super(...arguments);
        /**
         * Current timeout identifier
         */
        this._timeoutID = null;
        /**
         * Notification type.
         */
        this._type = NOTIFICATION_TYPE.INLINE;
        /**
         * `true` to hide the close button.
         */
        this.hideCloseButton = false;
        /**
         * Notification kind.
         */
        this.kind = NOTIFICATION_KIND.SUCCESS;
        /**
         * Low contrast mode
         */
        this.lowContrast = false;
        /**
         * `true` if the notification should be open.
         */
        this.open = true;
        /**
         * Specify an optional duration the notification should be closed in
         */
        this.timeout = null;
        /**
         * The subtitle.
         */
        this.subtitle = '';
        /**
         * The title.
         */
        this.title = '';
    }
    /**
     * Cancels the current timeout configured for the notification
     *
     * @param timeoutID current timeout's identifier
     */
    _cancelTimeout(timeoutID) {
        clearTimeout(timeoutID);
        this._timeoutID = null;
    }
    /**
     * Overrides (if exists) the timeout to close the notification
     *
     * @param timeout the time in ms
     */
    _initializeTimeout(timeout) {
        if (this._timeoutID) {
            this._cancelTimeout(this._timeoutID);
        }
        this._timeoutID = setTimeout(this._handleUserOrTimerInitiatedClose.bind(this, null), timeout);
    }
    /**
     * Handles `click` event on the close button.
     *
     * @param event The event.
     * @param event.target The event target.
     */
    _handleClickCloseButton({ target }) {
        this._handleUserOrTimerInitiatedClose(target);
    }
    /**
     * Handles user-initiated or through timer close request of this modal.
     *
     * @param triggeredBy The element that triggered this close request, if there is one.
     */
    _handleUserOrTimerInitiatedClose(triggeredBy) {
        if (this.open) {
            const init = {
                bubbles: true,
                cancelable: true,
                composed: true,
                detail: {
                    triggeredBy,
                },
            };
            if (this.dispatchEvent(new CustomEvent(this.constructor.eventBeforeClose, init))) {
                this.open = false;
                this.dispatchEvent(new CustomEvent(this.constructor.eventClose, init));
            }
        }
    }
    /**
     * @returns The template part for the close button.
     */
    _renderButton() {
        const { ariaLabel, _type: type, _handleClickCloseButton: handleClickCloseButton, } = this;
        return html `
      <button
        type="button"
        class="${prefix}--${type}-notification__close-button"
        aria-label=${ifDefined(ariaLabel)}
        title=${ifDefined(ariaLabel)}
        @click="${handleClickCloseButton}">
        ${Close16({
            class: `${prefix}--${type}-notification__close-icon`,
        })}
      </button>
    `;
    }
    /**
     * @returns The template part for the text contents.
     */
    _renderText() {
        const { subtitle, title, _type: type } = this;
        return html `
      <div class="${prefix}--${type}-notification__text-wrapper">
        <p class="${prefix}--${type}-notification__title">
          ${title}<slot name="title"></slot>
        </p>
        <div class="${prefix}--${type}-notification__subtitle">
          ${subtitle}<slot name="subtitle"></slot>
        </div>
        <slot></slot>
      </div>
    `;
    }
    /**
     * @returns The template part for the icon.
     */
    _renderIcon() {
        const { statusIconDescription, kind, _type: type } = this;
        const { [kind]: icon } = iconsForKinds;
        return !icon
            ? undefined
            : icon({
                class: `${prefix}--${type}-notification__icon`,
                children: !statusIconDescription
                    ? undefined
                    : svg `<title>${statusIconDescription}</title>`,
            });
    }
    connectedCallback() {
        if (!this.hasAttribute('role')) {
            this.setAttribute('role', 'alert');
        }
        super.connectedCallback();
    }
    updated(changedProperties) {
        const openChanged = changedProperties.has('open');
        const timeoutChanged = changedProperties.has('timeout');
        if (openChanged || timeoutChanged) {
            if (this.open && this.timeout) {
                this._initializeTimeout(this.timeout);
            }
            else if (!this.open && this._timeoutID) {
                this._cancelTimeout(this._timeoutID);
            }
        }
    }
    render() {
        const { _type: type } = this;
        return html `
      <div class="${prefix}--${type}-notification__details">
        ${this._renderIcon()}${this._renderText()}
      </div>
      ${this._renderButton()}
    `;
    }
    /**
     * The name of the custom event fired before this notification is being closed upon a user gesture.
     * Cancellation of this event stops the user-initiated action of closing this notification.
     */
    static get eventBeforeClose() {
        return `${prefix}-notification-beingclosed`;
    }
    /**
     * The name of the custom event fired after this notification is closed upon a user gesture.
     */
    static get eventClose() {
        return `${prefix}-notification-closed`;
    }
};
CDSInlineNotification.styles = styles; // `styles` here is a `CSSResult` generated by custom Vite loader
__decorate([
    property({ attribute: 'aria-label' })
], CDSInlineNotification.prototype, "ariaLabel", void 0);
__decorate([
    property({ type: Boolean, reflect: true, attribute: 'hide-close-button' })
], CDSInlineNotification.prototype, "hideCloseButton", void 0);
__decorate([
    property({ attribute: 'status-icon-description' })
], CDSInlineNotification.prototype, "statusIconDescription", void 0);
__decorate([
    property({ reflect: true })
], CDSInlineNotification.prototype, "kind", void 0);
__decorate([
    property({ type: Boolean, reflect: true, attribute: 'low-contrast' })
], CDSInlineNotification.prototype, "lowContrast", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSInlineNotification.prototype, "open", void 0);
__decorate([
    property({ type: Number, reflect: true })
], CDSInlineNotification.prototype, "timeout", void 0);
__decorate([
    property()
], CDSInlineNotification.prototype, "subtitle", void 0);
__decorate([
    property()
], CDSInlineNotification.prototype, "title", void 0);
CDSInlineNotification = __decorate([
    carbonElement(`${prefix}-inline-notification`)
], CDSInlineNotification);
var CDSInlineNotification$1 = CDSInlineNotification;

export { CDSInlineNotification$1 as default };
//# sourceMappingURL=inline-notification.js.map
