/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { LitElement, html } from 'lit';
import { query, property } from 'lit/decorators.js';
import { classMap } from 'lit/directives/class-map.js';
import { prefix } from '../../globals/settings.js';
import WarningFilled16 from '../../icons/warning--filled/16.js';
import WarningAltFilled16 from '../../icons/warning--alt--filled/16.js';
import Add16 from '../../icons/add/16.js';
import Subtract16 from '../../icons/subtract/16.js';
import ifNonEmpty from '../../globals/directives/if-non-empty.js';
export { NUMBER_INPUT_VALIDATION_STATUS } from './defs.js';
import styles from './number-input.scss.js';
import CDSTextInput from '../text-input/text-input.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';
import { INPUT_SIZE } from '../text-input/defs.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Number input.
 *
 * @element cds-number-input
 * @fires cds-number-input
 *   The name of the custom event fired after the value is changed upon a user gesture.
 * @slot helper-text - The helper text.
 * @slot label-text - The label text.
 * @slot validity-message - The validity message. If present and non-empty, this input shows the UI of its invalid state.
 */
let CDSNumberInput = class CDSNumberInput extends CDSTextInput {
    constructor() {
        super(...arguments);
        this._min = '';
        this._max = '';
        this._step = '1';
        /**
         * Aria text for the button that increments the value
         */
        this.incrementButtonAssistiveText = 'increase number input';
        /**
         * Aria text for the button that decrements the value
         */
        this.decrementButtonAssistiveText = 'decrease number input';
        /**
         * Specify whether you want the steppers to be hidden
         */
        this.hideSteppers = false;
        /**
         * `true` to allow empty string.
         */
        this.allowEmpty = false;
        /**
         * The input box size.
         */
        this.size = INPUT_SIZE.MEDIUM;
    }
    /**
     * Handles `input` event on the `<input>` in the shadow DOM.
     */
    _handleInput(event) {
        const { target } = event;
        const { value } = target;
        this.dispatchEvent(new CustomEvent(this.constructor.eventInput, {
            bubbles: true,
            composed: true,
            cancelable: false,
            detail: {
                value,
            },
        }));
        super._handleInput(event);
    }
    /**
     * Handles `click` event on the up button in the shadow DOM.
     */
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    _handleUserInitiatedStepDown(_) {
        const { _input: input } = this;
        this.stepDown();
        this.dispatchEvent(new CustomEvent(this.constructor.eventInput, {
            bubbles: true,
            composed: true,
            cancelable: false,
            detail: {
                value: input.value,
            },
        }));
    }
    /**
     * Handles `click` event on the down button in the shadow DOM.
     */
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    _handleUserInitiatedStepUp(_) {
        const { _input: input } = this;
        this.stepUp();
        this.dispatchEvent(new CustomEvent(this.constructor.eventInput, {
            bubbles: true,
            composed: true,
            cancelable: false,
            detail: {
                value: input.value,
            },
        }));
    }
    _getInputValidity() {
        var _a, _b;
        if (this.invalid) {
            return false;
        }
        if (((_a = this._input) === null || _a === void 0 ? void 0 : _a.valueAsNumber) > Number(this.max) ||
            ((_b = this._input) === null || _b === void 0 ? void 0 : _b.valueAsNumber) < Number(this.min)) {
            return false;
        }
        if (this.value === '') {
            return this.allowEmpty;
        }
        return true;
    }
    /**
     * The minimum value allowed in the input
     */
    get min() {
        return this._min.toString();
    }
    set min(value) {
        const oldValue = this.min;
        this._min = value;
        this.requestUpdate('min', oldValue);
    }
    /**
     * The maximum value allowed in the input
     */
    get max() {
        return this._max.toString();
    }
    set max(value) {
        const oldValue = this.max;
        this._max = value;
        this.requestUpdate('max', oldValue);
    }
    /**
     * The amount the value should increase or decrease by
     */
    get step() {
        return this._step.toString();
    }
    set step(value) {
        const oldValue = this.step;
        this._step = value;
        this.requestUpdate('step', oldValue);
    }
    /**
     * Handles incrementing the value in the input
     */
    stepUp() {
        this._input.stepUp();
    }
    /**
     * Handles decrementing the value in the input
     */
    stepDown() {
        this._input.stepDown();
    }
    render() {
        const { _handleInput: handleInput, _handleUserInitiatedStepDown: handleUserInitiatedStepDown, _handleUserInitiatedStepUp: handleUserInitiatedStepUp, } = this;
        const isValid = this._getInputValidity();
        const invalidIcon = WarningFilled16({
            class: `${prefix}--number__invalid`,
        });
        const warnIcon = WarningAltFilled16({
            class: `${prefix}--number__invalid ${prefix}--number__invalid--warning`,
        });
        const normalizedProps = {
            disabled: !this.readonly && this.disabled,
            invalid: !this.readonly && !isValid,
            warn: !this.readonly && isValid && this.warn,
            'slot-name': '',
            'slot-text': '',
            icon: null,
        };
        const wrapperClasses = classMap({
            [`${prefix}--number`]: true,
            [`${prefix}--number--${this.size}`]: this.size,
            [`${prefix}--number--nosteppers`]: this.hideSteppers,
            [`${prefix}--number--readonly`]: this.readonly,
        });
        const inputWrapperClasses = classMap({
            [`${prefix}--number__input-wrapper`]: true,
            [`${prefix}--number__input-wrapper--warning`]: normalizedProps.warn,
            [`${prefix}--number__input-wrapper--slug`]: this._hasAILabel,
        });
        const labelClasses = classMap({
            [`${prefix}--label`]: true,
            [`${prefix}--label--disabled`]: normalizedProps.disabled,
            [`${prefix}--visually-hidden`]: this.hideLabel,
        });
        const helperTextClasses = classMap({
            [`${prefix}--form__helper-text`]: true,
            [`${prefix}--form__helper-text--disabled`]: normalizedProps.disabled,
        });
        const incrementButton = html `
      <button
        class="${prefix}--number__control-btn up-icon"
        aria-label="${this.incrementButtonAssistiveText}"
        aria-live="polite"
        aria-atomic="true"
        type="button"
        ?disabled=${normalizedProps.disabled}
        @click=${handleUserInitiatedStepUp}>
        ${Add16()}
      </button>
      <div class="${prefix}--number__rule-divider"></div>
    `;
        const decrementButton = html `
      <button
        class="${prefix}--number__control-btn down-icon"
        aria-label="${this.decrementButtonAssistiveText}"
        aria-live="polite"
        aria-atomic="true"
        type="button"
        ?disabled=${normalizedProps.disabled}
        @click=${handleUserInitiatedStepDown}>
        ${Subtract16()}
      </button>
      <div class="${prefix}--number__rule-divider"></div>
    `;
        const input = html `
      <input
        autocomplete="${this.autocomplete}"
        ?autofocus="${this.autofocus}"
        ?data-invalid="${normalizedProps.invalid}"
        ?disabled="${normalizedProps.disabled}"
        id="input"
        name="${ifNonEmpty(this.name)}"
        pattern="${ifNonEmpty(this.pattern)}"
        ?readonly="${this.readonly}"
        ?required="${this.required}"
        type="number"
        .value="${this._value}"
        @input="${handleInput}"
        min="${ifNonEmpty(this.min)}"
        max="${ifNonEmpty(this.max)}"
        step="${ifNonEmpty(this.step)}"
        role="alert"
        aria-atomic="true" />
    `;
        if (normalizedProps.invalid) {
            normalizedProps.icon = invalidIcon;
            normalizedProps['slot-name'] = 'invalid-text';
            normalizedProps['slot-text'] = this.invalidText;
        }
        else if (normalizedProps.warn) {
            normalizedProps.icon = warnIcon;
            normalizedProps['slot-name'] = 'warn-text';
            normalizedProps['slot-text'] = this.warnText;
        }
        return html `
      <div class="${wrapperClasses}" ?data-invalid=${normalizedProps.invalid}>
        <label class="${labelClasses}" for="input">
          <slot name="label-text"> ${this.label} </slot>
        </label>
        <div class="${inputWrapperClasses}">
          ${normalizedProps.icon} ${input}
          <slot name="ai-label" @slotchange="${this._handleSlotChange}"></slot>
          <slot name="slug" @slotchange="${this._handleSlotChange}"></slot>
          <div class="${prefix}--number__controls">
            ${!this.hideSteppers
            ? html `${decrementButton} ${incrementButton}`
            : null}
          </div>
        </div>
        <div
          class="${helperTextClasses}"
          ?hidden="${normalizedProps.invalid || normalizedProps.warn}">
          <slot name="helper-text"> ${this.helperText} </slot>
        </div>
        <div
          class="${prefix}--form-requirement"
          ?hidden="${!normalizedProps.invalid && !normalizedProps.warn}">
          <slot name="${normalizedProps['slot-name']}">
            ${normalizedProps['slot-text']}
          </slot>
        </div>
      </div>
    `;
    }
    /**
     * The name of the custom event fired after the value is changed upon a user gesture.
     */
    static get eventInput() {
        return `${prefix}-number-input`;
    }
};
CDSNumberInput.shadowRootOptions = Object.assign(Object.assign({}, LitElement.shadowRootOptions), { delegatesFocus: true });
CDSNumberInput.styles = styles; // `styles` here is a `CSSResult` generated by custom Vite loader
__decorate([
    query('input')
], CDSNumberInput.prototype, "_input", void 0);
__decorate([
    property({ reflect: true })
], CDSNumberInput.prototype, "min", null);
__decorate([
    property({ reflect: true })
], CDSNumberInput.prototype, "max", null);
__decorate([
    property({ reflect: true })
], CDSNumberInput.prototype, "step", null);
__decorate([
    property({ attribute: 'increment-button-assistive-text' })
], CDSNumberInput.prototype, "incrementButtonAssistiveText", void 0);
__decorate([
    property({ attribute: 'decrement-button-assistive-text' })
], CDSNumberInput.prototype, "decrementButtonAssistiveText", void 0);
__decorate([
    property({ type: Boolean, attribute: 'hide-steppers', reflect: true })
], CDSNumberInput.prototype, "hideSteppers", void 0);
__decorate([
    property({ type: Boolean, attribute: 'allow-empty', reflect: true })
], CDSNumberInput.prototype, "allowEmpty", void 0);
__decorate([
    property({ reflect: true })
], CDSNumberInput.prototype, "size", void 0);
CDSNumberInput = __decorate([
    carbonElement(`${prefix}-number-input`)
], CDSNumberInput);
var CDSNumberInput$1 = CDSNumberInput;

export { CDSNumberInput$1 as default };
//# sourceMappingURL=number-input.js.map
