/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { html } from 'lit';
import { property } from 'lit/decorators.js';
import { prefix } from '../../globals/settings.js';
import HostListener from '../../globals/decorators/host-listener.js';
import FocusMixin from '../../globals/mixins/focus.js';
import HostListenerMixin from '../../globals/mixins/host-listener.js';
import { find } from '../../globals/internal/collection-helpers.js';
import { OVERFLOW_MENU_SIZE } from './defs.js';
export { OVERFLOW_MENU_SIZE } from './defs.js';
import styles from './overflow-menu.scss.js';
import CDSIconButton from '../icon-button/icon-button.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2023
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Overflow menu.
 *
 * @element cds-overflow-menu
 * @slot icon - The icon for the trigger button.
 */
let CDSOverflowMenu = class CDSOverflowMenu extends HostListenerMixin(FocusMixin(CDSIconButton)) {
    constructor() {
        super(...arguments);
        /**
         * The menu body.
         */
        this._menuBody = null;
        /**
         * Handles `click` event on the trigger button.
         */
        this._handleClickTrigger = async () => {
            this._handleUserInitiatedToggle();
        };
        /**
         * Handles `keydown` event on the trigger button.
         */
        this._handleKeydownTrigger = async (event) => {
            if (event.key === ' ' || event.key === 'Enter') {
                this._handleUserInitiatedToggle();
                event.preventDefault();
            }
        };
        /**
         * `true` if this tooltip is in a data table row
         */
        this.dataTable = false;
        /**
         * `true` if this overflow menu should be disabled.
         */
        this.disabled = false;
        /**
         * `true` if this overflow menu body should be flipped.
         */
        this.flipped = false;
        /**
         * `true` if the dropdown should be open.
         */
        this.open = false;
        /**
         * Index (starting at 1) of overflow menu item to focus on open.
         */
        this.index = 1;
        /**
         * Overflow menu size.
         */
        this.size = OVERFLOW_MENU_SIZE.MEDIUM;
        /**
         * `true` if this menu is a toolbar action
         */
        this.toolbarAction = false;
    }
    /**
     * Handles user-initiated toggling of the menu.
     */
    async _handleUserInitiatedToggle() {
        this.open = !this.open;
        const { index, open, updateComplete } = this;
        if (open) {
            await updateComplete;
            const { _menuBody: menuBody } = this;
            const menuItem = menuBody === null || menuBody === void 0 ? void 0 : menuBody.querySelector(`${prefix}-overflow-menu-item:nth-of-type(${index})`);
            menuItem === null || menuItem === void 0 ? void 0 : menuItem.focus();
        }
    }
    /**
     * @returns The position of the trigger button in the viewport.
     */
    get triggerPosition() {
        return this.getBoundingClientRect();
    }
    connectedCallback() {
        if (!this.hasAttribute('aria-haspopup')) {
            this.setAttribute('aria-haspopup', 'true');
        }
        if (!this.shadowRoot) {
            this.attachShadow({ mode: 'open' });
        }
        super.connectedCallback();
    }
    updated(changedProperties) {
        var _a, _b, _c, _d, _e, _f, _g;
        const button = (_b = (_a = this.shadowRoot) === null || _a === void 0 ? void 0 : _a.querySelector(`${prefix}-tooltip`)) === null || _b === void 0 ? void 0 : _b.querySelector('button');
        button === null || button === void 0 ? void 0 : button.classList.add(`${prefix}--btn--icon-only`, `${prefix}--overflow-menu`);
        if (changedProperties.has('open')) {
            const { open } = this;
            if (open && !this._menuBody) {
                this._menuBody = find(this.childNodes, (elem) => elem.constructor.FLOATING_MENU);
            }
            const { _menuBody: menuBody } = this;
            if (menuBody) {
                menuBody.open = open;
                const tooltipContent = (_c = this.querySelector('[slot=tooltip-content]')) === null || _c === void 0 ? void 0 : _c.textContent;
                button === null || button === void 0 ? void 0 : button.setAttribute('aria-expanded', String(Boolean(open)));
                button === null || button === void 0 ? void 0 : button.setAttribute('aria-label', String(tooltipContent));
            }
        }
        if (changedProperties.has('dataTable')) {
            const tooltip = (_d = this.shadowRoot) === null || _d === void 0 ? void 0 : _d.querySelector(`${prefix}-tooltip`);
            tooltip === null || tooltip === void 0 ? void 0 : tooltip.setAttribute('data-table', '');
        }
        if (changedProperties.has('size')) {
            const { size } = this;
            const { _menuBody: menuBody } = this;
            if (menuBody) {
                menuBody.size = size;
            }
            button === null || button === void 0 ? void 0 : button.classList.forEach((item) => {
                if (item.startsWith(`${prefix}--overflow-menu--`)) {
                    button === null || button === void 0 ? void 0 : button.classList.remove(item);
                }
            });
            button === null || button === void 0 ? void 0 : button.classList.add(`${prefix}--overflow-menu--${this.size}`);
            const tooltip = (_e = this.shadowRoot) === null || _e === void 0 ? void 0 : _e.querySelector(`${prefix}-tooltip`);
            tooltip === null || tooltip === void 0 ? void 0 : tooltip.setAttribute('size', this.size);
        }
        if (changedProperties.has('toolbarAction') && this.toolbarAction) {
            (_g = (_f = this.shadowRoot) === null || _f === void 0 ? void 0 : _f.querySelector(`${prefix}-tooltip`)) === null || _g === void 0 ? void 0 : _g.setAttribute('toolbar-action', '');
        }
        super.updated(changedProperties);
    }
    render() {
        return html ` ${super.render()} `;
    }
};
CDSOverflowMenu.styles = styles; // `styles` here is a `CSSResult` generated by custom Vite loader
__decorate([
    HostListener('click')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSOverflowMenu.prototype, "_handleClickTrigger", void 0);
__decorate([
    HostListener('keydown')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSOverflowMenu.prototype, "_handleKeydownTrigger", void 0);
__decorate([
    property({ type: Boolean, reflect: true, attribute: 'data-table' })
], CDSOverflowMenu.prototype, "dataTable", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSOverflowMenu.prototype, "disabled", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSOverflowMenu.prototype, "flipped", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSOverflowMenu.prototype, "open", void 0);
__decorate([
    property()
], CDSOverflowMenu.prototype, "index", void 0);
__decorate([
    property({ reflect: true })
], CDSOverflowMenu.prototype, "size", void 0);
__decorate([
    property({ type: Boolean, attribute: 'toolbar-action', reflect: true })
], CDSOverflowMenu.prototype, "toolbarAction", void 0);
CDSOverflowMenu = __decorate([
    carbonElement(`${prefix}-overflow-menu`)
], CDSOverflowMenu);
var CDSOverflowMenu$1 = CDSOverflowMenu;

export { CDSOverflowMenu$1 as default };
//# sourceMappingURL=overflow-menu.js.map
