/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { html, LitElement } from 'lit';
import { query, property } from 'lit/decorators.js';
import CaretLeft16 from '../../icons/caret--left/16.js';
import CaretRight16 from '../../icons/caret--right/16.js';
import { prefix } from '../../globals/settings.js';
import FocusMixin from '../../globals/mixins/focus.js';
import HostListenerMixin from '../../globals/mixins/host-listener.js';
import HostListener from '../../globals/decorators/host-listener.js';
import styles from './pagination.scss.js';
import { PAGINATION_SIZE } from './defs.js';
import '../button/button.js';
import '../button/button-set.js';
import '../button/button-skeleton.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Pagination UI.
 *
 * @element cds-pagination
 * @slot page-sizes-select - Where to put in the `<page-sizes-select>`.
 * @fires cds-pagination-changed-current - The custom event fired after the current page is changed from `<cds-pages-select>`.
 * @fires cds-page-sizes-select-changed - The custom event fired after the number of rows per page is changed from `<cds-page-sizes-select>`.
 */
let CDSPagination = class CDSPagination extends FocusMixin(HostListenerMixin(LitElement)) {
    constructor() {
        super(...arguments);
        /**
         * The assistive text for the button to go to previous page.
         */
        this.backwardText = 'Previous page';
        /**
         * The current page
         */
        this.page = 1;
        /**
         * The formatter for the assistive text for screen readers to announce.
         * Should be changed upon the locale the UI is rendered with.
         */
        this.formatLabelText = ({ count }) => `Page number, of ${count} page${count <= 1 ? '' : 's'}`;
        /**
         * The formatter, used with determinate the total pages. Should be changed upon the locale the UI is rendered with.
         */
        this.formatStatusWithDeterminateTotal = ({ start, end, count }) => `${start}–${end} of ${count} item${count <= 1 ? '' : 's'}`;
        /**
         * The formatter, used with indeterminate the total pages. Should be changed upon the locale the UI is rendered with.
         */
        this.formatStatusWithIndeterminateTotal = ({ start, end, count }) => end == null
            ? `Item ${start}–`
            : `${start}–${end} item${count <= 1 ? '' : 's'}`;
        /**
         * The formatter for the text next to the select box. Should be changed upon the locale the UI is rendered with.
         */
        this.formatSupplementalText = ({ count }) => this.pagesUnknown || !this.totalItems
            ? `page`
            : `of ${count} page${count <= 1 ? '' : 's'}`;
        /**
         * The translatable text indicating the number of items per page.
         */
        this.itemsPerPageText = 'Items per page:';
        /**
         * `true` if the pagination UI should be disabled.
         */
        this.disabled = false;
        /**
         * The assistive text for the button to go to next page.
         */
        this.forwardText = 'Next page';
        /**
         * true if the select box to change the page should be disabled.
         */
        this.pageInputDisabled = false;
        /**
         * Number of items per page.
         */
        this.pageSize = 10;
        /**
         * true if the total number of items is unknown.
         */
        this.pagesUnknown = false;
        /**
         * Specify the size of the Pagination.
         */
        this.size = PAGINATION_SIZE.MEDIUM;
        /**
         * The row number where current page start with, index that starts with zero.
         */
        this.start = 0;
        /**
         * The number of total pages.
         */
        this.totalPages = 1;
    }
    _handleSlotChange({ target }) {
        const content = target
            .assignedNodes()
            .filter((node) => node.nodeType !== Node.TEXT_NODE || node.textContent.trim());
        content.forEach((item) => {
            this._pageSizeSelect.appendChild(item);
        });
    }
    /**
     * @returns Page status text.
     */
    _renderStatusText() {
        const { start, pageSize, totalItems, pagesUnknown, formatStatusWithDeterminateTotal, formatStatusWithIndeterminateTotal, } = this;
        // * Regular: `1-10 of 100 items`
        // * Indeterminate total: `Item 1-10` (`Item 11-` at the last page)
        const end = Math.min(start + pageSize, totalItems == null ? Infinity : totalItems);
        const format = totalItems == null || pagesUnknown
            ? formatStatusWithIndeterminateTotal
            : formatStatusWithDeterminateTotal;
        // `start`/`end` properties starts with zero, whereas we want to show number starting with 1
        return format({ start: start + 1, end, count: totalItems });
    }
    /**
     * Handles user-initiated change in the row number the current page starts with.
     *
     * @param start The new current row number, index that starts with zero.
     */
    _handleUserInitiatedChangeStart(start) {
        this.start = start;
        this.dispatchEvent(new CustomEvent(this.constructor.eventChangeCurrent, {
            bubbles: true,
            composed: true,
            detail: {
                page: this.page,
                pageSize: this.pageSize,
            },
        }));
    }
    /**
     * Handles user-initiated change in the size of the page.
     */
    _handleUserInitiatedPageSizeChange() {
        this.dispatchEvent(new CustomEvent(this.constructor.eventPageSizeChanged, {
            bubbles: true,
            composed: true,
            detail: {
                page: this.page,
                pageSize: this.pageSize,
            },
        }));
    }
    /**
     * Handles `click` event on the previous button.
     */
    _handleClickPrevButton() {
        var _a;
        const { start: oldStart, pageSize } = this;
        this.page--;
        const newStart = Math.max(oldStart - pageSize, 0);
        if (newStart !== oldStart) {
            this._handleUserInitiatedChangeStart(newStart);
        }
        // reset focus to forward button if it reaches the beginning
        if (this.page === 1) {
            const { selectorForwardButton } = this
                .constructor;
            ((_a = this.shadowRoot) === null || _a === void 0 ? void 0 : _a.querySelector(`[button-class-name*=${selectorForwardButton}]`)).focus();
        }
    }
    /**
     * Handles `click` event on the next button.
     */
    _handleClickNextButton() {
        var _a;
        const { start: oldStart, pageSize, totalItems } = this;
        this.page++;
        const newStart = oldStart + pageSize;
        if (newStart < (totalItems == null ? Infinity : totalItems)) {
            this._handleUserInitiatedChangeStart(newStart);
        }
        // reset focus to previous button if it reaches the end
        if (this.page === this.totalPages) {
            const { selectorPreviousButton } = this
                .constructor;
            ((_a = this.shadowRoot) === null || _a === void 0 ? void 0 : _a.querySelector(`[button-class-name*=${selectorPreviousButton}]`)).focus();
        }
    }
    /**
     * Handles user-initiated change in number of rows per page.
     *
     * @param event The event.
     */
    _handleChangeSelector(event) {
        const { value } = event.detail;
        const { totalItems, pageSize } = this;
        if (event.composedPath()[0] === this._pageSizeSelect) {
            this.pageSize = parseInt(value);
            // Default pageSize to effectively be 1 when we have a value of 0 to avoid
            // division by 0.
            this.totalPages =
                pageSize > 0 ? Math.ceil(totalItems / pageSize) : totalItems;
            this.page = 1;
            this.start = 0;
            this._handleUserInitiatedPageSizeChange();
        }
        else {
            this.page = value;
            this._handleUserInitiatedChangeStart((value - 1) * pageSize);
        }
    }
    updated(changedProperties) {
        const { page, pageSize, totalItems } = this;
        const { selectorPageSizesSelect, selectorPagesSelect } = this
            .constructor;
        if (changedProperties.has('pageSize')) {
            this.shadowRoot.querySelector(selectorPageSizesSelect).value =
                pageSize;
        }
        if (changedProperties.has('pageSize') || changedProperties.has('start')) {
            // Default pageSize to effectively be 1 when we have a value of 0 to avoid
            // division by 0.
            this.totalPages =
                pageSize > 0 ? Math.ceil(totalItems / pageSize) : totalItems;
            this.shadowRoot.querySelector(selectorPagesSelect).value =
                this.page.toString();
        }
        if (changedProperties.has('page')) {
            this._handleUserInitiatedChangeStart((page - 1) * pageSize);
        }
    }
    render() {
        const { page, disabled, forwardText, backwardText, itemsPerPageText, pageInputDisabled, pageSize, pageSizeInputDisabled, pagesUnknown, size, start, totalItems, totalPages, _handleClickPrevButton: handleClickPrevButton, _handleClickNextButton: handleClickNextButton, _handleSlotChange: handleSlotChange, formatLabelText, formatSupplementalText, } = this;
        const { isLastPage = start + pageSize >= totalItems } = this;
        const prevButtonDisabled = disabled || start === 0;
        const nextButtonDisabled = disabled || isLastPage;
        const prevButtonClassMap = {
            [`${prefix}--btn`]: true,
            [`${prefix}--btn--icon-only`]: true,
            [`${prefix}--pagination__button`]: true,
            [`${prefix}--pagination__button--backward`]: true,
            [`${prefix}--pagination__button--no-index`]: prevButtonDisabled,
            [`${prefix}--btn--${size}`]: true,
            [`${prefix}--btn--ghost`]: true,
        };
        const nextButtonClassMap = {
            [`${prefix}--btn`]: true,
            [`${prefix}--btn--icon-only`]: true,
            [`${prefix}--pagination__button`]: true,
            [`${prefix}--pagination__button--forward`]: true,
            [`${prefix}--pagination__button--no-index`]: nextButtonDisabled,
            [`${prefix}--btn--${size}`]: true,
            [`${prefix}--btn--ghost`]: true,
        };
        const prevButtonClasses = Object.entries(prevButtonClassMap)
            .filter(([, value]) => value === true)
            .map(([key]) => key)
            .join(' ');
        const nextButtonClasses = Object.entries(nextButtonClassMap)
            .filter(([, value]) => value === true)
            .map(([key]) => key)
            .join(' ');
        return html `
      <div class="${prefix}--pagination__left">
        <label for="select" class="${prefix}--pagination__text"
          ><slot name="label-text">${itemsPerPageText}</slot></label
        >
        <cds-select
          ?disabled=${disabled || pageSizeInputDisabled}
          id="page-size-select"
          left-select
          pagination
          size="${size}"
          inline
          value="${pageSize}">
          <slot @slotchange=${handleSlotChange}></slot>
        </cds-select>
        <span
          class="${prefix}--pagination__text ${prefix}--pagination__items-count"
          >${this._renderStatusText()}</span
        >
      </div>
      <div class="${prefix}--pagination__right">
        <label for="select" class="${prefix}--label ${prefix}--visually-hidden">
          ${formatLabelText({ count: totalPages })}
        </label>
        ${pagesUnknown || !totalItems
            ? html `
              <span
                class="${prefix}--pagination__text ${prefix}--pagination__page-text"
                >${formatSupplementalText({ count: totalPages })}</span
              >

              <cds-select
                ?disabled=${disabled || pageInputDisabled}
                id="pages-select"
                pagination
                size="${size}"
                inline
                value="${page}">
                ${Array.from(new Array(totalPages)).map((_item, index) => html `
                    <cds-select-item value="${index + 1}">
                      ${index + 1}
                    </cds-select-item>
                  `)}
              </cds-select>
            `
            : html `
              <cds-select
                ?disabled=${disabled || pageInputDisabled}
                id="pages-select"
                pagination
                size="${size}"
                inline
                value="${page}">
                ${Array.from(new Array(totalPages)).map((_item, index) => html `
                    <cds-select-item value="${index + 1}">
                      ${index + 1}
                    </cds-select-item>
                  `)}
              </cds-select>
              <span class="${prefix}--pagination__text"
                >${formatSupplementalText({ count: totalPages })}</span
              >
            `}

        <div class="${prefix}--pagination__control-buttons">
          <cds-button
            pagination
            size="${size}"
            ?disabled="${prevButtonDisabled}"
            button-class-name="${prevButtonClasses}"
            tooltip-text="${backwardText}"
            @click="${handleClickPrevButton}">
            ${CaretLeft16({ slot: 'icon' })}
          </cds-button>
          <cds-button
            tooltip-position="top-right"
            pagination
            size="${size}"
            ?disabled="${nextButtonDisabled}"
            button-class-name="${nextButtonClasses}"
            tooltip-text="${forwardText}"
            @click="${handleClickNextButton}">
            ${CaretRight16({ slot: 'icon' })}
          </cds-button>
        </div>
      </div>
    `;
    }
    /**
     * A selector that will return the select box for the current page.
     */
    static get selectorPagesSelect() {
        return `${prefix}-select#pages-select`;
    }
    /**
     * A selector that will return the select box for page sizes.
     */
    static get selectorPageSizesSelect() {
        return `${prefix}-select`;
    }
    /**
     * A selector that will return the previous button.
     */
    static get selectorPreviousButton() {
        return `${prefix}--pagination__button--backward`;
    }
    /**
     * A selector that will return the forward button.
     */
    static get selectorForwardButton() {
        return `${prefix}--pagination__button--forward`;
    }
    /**
     * The name of the custom event fired after the current row number is changed.
     */
    static get eventChangeCurrent() {
        return `${prefix}-pagination-changed-current`;
    }
    /**
     * The name of the custom event fired after the number of rows per page is changed from `<cds-page-sizes-select>`.
     */
    static get eventPageSizeChanged() {
        return `${prefix}-page-sizes-select-changed`;
    }
};
CDSPagination.shadowRootOptions = Object.assign(Object.assign({}, LitElement.shadowRootOptions), { delegatesFocus: true });
CDSPagination.styles = styles; // `styles` here is a `CSSResult` generated by custom Vite loader
__decorate([
    query(`${prefix}-select`)
], CDSPagination.prototype, "_pageSizeSelect", void 0);
__decorate([
    HostListener(`${prefix}-select-selected`)
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSPagination.prototype, "_handleChangeSelector", null);
__decorate([
    property({ attribute: 'backward-text' })
], CDSPagination.prototype, "backwardText", void 0);
__decorate([
    property({ type: Number, reflect: true })
], CDSPagination.prototype, "page", void 0);
__decorate([
    property({ attribute: false })
], CDSPagination.prototype, "formatLabelText", void 0);
__decorate([
    property({ attribute: false })
], CDSPagination.prototype, "formatStatusWithDeterminateTotal", void 0);
__decorate([
    property({ attribute: false })
], CDSPagination.prototype, "formatStatusWithIndeterminateTotal", void 0);
__decorate([
    property({ attribute: false })
], CDSPagination.prototype, "formatSupplementalText", void 0);
__decorate([
    property({ type: Boolean, attribute: 'is-last-page' })
], CDSPagination.prototype, "isLastPage", void 0);
__decorate([
    property({ attribute: 'items-per-page-text' })
], CDSPagination.prototype, "itemsPerPageText", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSPagination.prototype, "disabled", void 0);
__decorate([
    property({ attribute: 'forward-text' })
], CDSPagination.prototype, "forwardText", void 0);
__decorate([
    property({ type: Boolean, attribute: 'page-input-disabled' })
], CDSPagination.prototype, "pageInputDisabled", void 0);
__decorate([
    property({ type: Number, attribute: 'page-size', reflect: true })
], CDSPagination.prototype, "pageSize", void 0);
__decorate([
    property({ type: Boolean, attribute: 'page-size-input-disabled' })
], CDSPagination.prototype, "pageSizeInputDisabled", void 0);
__decorate([
    property({ attribute: 'page-size-label-text' })
], CDSPagination.prototype, "pageSizeLabelText", void 0);
__decorate([
    property({ type: Boolean, reflect: true, attribute: 'pages-unknown' })
], CDSPagination.prototype, "pagesUnknown", void 0);
__decorate([
    property({ reflect: true })
], CDSPagination.prototype, "size", void 0);
__decorate([
    property({ type: Number })
], CDSPagination.prototype, "start", void 0);
__decorate([
    property({ type: Number, attribute: 'total-items' })
], CDSPagination.prototype, "totalItems", void 0);
__decorate([
    property({ type: Number })
], CDSPagination.prototype, "totalPages", void 0);
CDSPagination = __decorate([
    carbonElement(`${prefix}-pagination`)
], CDSPagination);
var CDSPagination$1 = CDSPagination;

export { CDSPagination$1 as default };
//# sourceMappingURL=pagination.js.map
