/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { classMap } from 'lit/directives/class-map.js';
import { LitElement, html } from 'lit';
import { query, property } from 'lit/decorators.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';
import { prefix } from '../../globals/settings.js';
import styles from './popover.scss.js';
import PopoverController from '../../globals/controllers/popover-controller.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
var CDSPopover_1;
/**
 * Popover.
 *
 * @element cds-popover
 */
let CDSPopover = CDSPopover_1 = class CDSPopover extends LitElement {
    constructor() {
        super(...arguments);
        /**
         * Create popover controller instance
         */
        this.popoverController = new PopoverController(this);
        /**
         * Specify direction of alignment
         */
        this.align = '';
        /**
         * Specify whether a auto align functionality should be applied
         */
        this.autoalign = false;
        /**
         * Specify whether a caret should be rendered
         */
        this.caret = true;
        /**
         * Specify whether a dropShadow should be rendered
         */
        this.dropShadow = true;
        /**
         * Render the component using the high-contrast variant
         */
        this.highContrast = false;
        /**
         * Specify whether the component is currently open or closed
         */
        this.open = false;
        /**
         * Render the component using the tab tip variant
         */
        this.tabTip = false;
    }
    /**
     * Handles `slotchange` event.
     */
    _handleSlotChange({ target }) {
        if (this.tabTip) {
            const component = target
                .assignedNodes()
                .filter((node) => node.nodeType !== Node.TEXT_NODE || node.textContent.trim());
            component[0].classList.add(`${prefix}--popover--tab-tip__button`);
        }
        this.requestUpdate();
    }
    updated(changedProperties) {
        var _a, _b, _c;
        const { selectorPopoverContent } = this.constructor;
        ['open', 'align', 'autoalign', 'caret', 'dropShadow', 'tabTip'].forEach((name) => {
            if (changedProperties.has(name)) {
                const { [name]: value } = this;
                if (this.querySelector(selectorPopoverContent)) {
                    this.querySelector(selectorPopoverContent)[name] = value;
                }
            }
        });
        if (this.autoalign && this.open) {
            // auto align functionality with @floating-ui/dom library
            const button = this._triggerSlotNode.assignedElements()[0];
            const content = this._contentSlotNode.assignedElements()[0];
            const tooltip = (_a = content === null || content === void 0 ? void 0 : content.shadowRoot) === null || _a === void 0 ? void 0 : _a.querySelector(CDSPopover_1.selectorPopoverContentClass);
            const arrowElement = (_b = content === null || content === void 0 ? void 0 : content.shadowRoot) === null || _b === void 0 ? void 0 : _b.querySelector(CDSPopover_1.selectorPopoverCaret);
            if (button && tooltip) {
                (_c = this.popoverController) === null || _c === void 0 ? void 0 : _c.setPlacement({
                    trigger: button,
                    target: tooltip,
                    arrowElement: this.caret && arrowElement
                        ? arrowElement
                        : undefined,
                    caret: this.caret,
                    flip: true,
                    alignment: this.align,
                });
            }
        }
    }
    render() {
        const { dropShadow, highContrast, open, tabTip, _handleSlotChange: handleSlotChange, } = this;
        if (tabTip) {
            this.caret = tabTip ? false : true;
        }
        this.align = this.align ? this.align : tabTip ? 'bottom-left' : 'bottom';
        const classes = classMap({
            [`${prefix}--popover-container`]: true,
            [`${prefix}--popover--caret`]: this.caret,
            [`${prefix}--popover--drop-shadow`]: dropShadow,
            [`${prefix}--popover--high-contrast`]: highContrast,
            [`${prefix}--popover--open`]: open,
            [`${prefix}--popover--${this.align}`]: true,
            [`${prefix}--popover--tab-tip`]: tabTip,
        });
        return html `
    <span class="${classes}">
      <slot @slotchange="${handleSlotChange}"></slot>
      <slot name="content"><slot>
    </span>
    `;
    }
    /**
     * A selector that will return popover content element within
     * CDSPopoverContent's shadowRoot.
     */
    static get selectorPopoverContentClass() {
        return `.${prefix}--popover-content`;
    }
    /**
     * A selector that will return popover caret element within
     * CDSPopoverContent's shadowRoot.
     */
    static get selectorPopoverCaret() {
        return `.${prefix}--popover-caret`;
    }
    /**
     * A selector that will return the CDSPopoverContent.
     */
    static get selectorPopoverContent() {
        return `${prefix}-popover-content`;
    }
};
CDSPopover.styles = styles; // `styles` here is a `CSSResult` generated by custom Vite loader
__decorate([
    query('slot')
], CDSPopover.prototype, "_triggerSlotNode", void 0);
__decorate([
    query('slot[name="content"]')
], CDSPopover.prototype, "_contentSlotNode", void 0);
__decorate([
    property({ reflect: true, type: String })
], CDSPopover.prototype, "align", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSPopover.prototype, "autoalign", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSPopover.prototype, "caret", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSPopover.prototype, "dropShadow", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSPopover.prototype, "highContrast", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSPopover.prototype, "open", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSPopover.prototype, "tabTip", void 0);
CDSPopover = CDSPopover_1 = __decorate([
    carbonElement(`${prefix}-popover`)
], CDSPopover);
var CDSPopover$1 = CDSPopover;

export { CDSPopover$1 as default };
//# sourceMappingURL=popover.js.map
