/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { LitElement, html, svg } from 'lit';
import { property } from 'lit/decorators.js';
import CheckmarkOutline16 from '../../icons/checkmark--outline/16.js';
import CircleDash16 from '../../icons/circle-dash/16.js';
import Incomplete16 from '../../icons/incomplete/16.js';
import Warning16 from '../../icons/warning/16.js';
import { prefix } from '../../globals/settings.js';
import FocusMixin from '../../globals/mixins/focus.js';
import { PROGRESS_STEP_STAT } from './defs.js';
export { PROGRESS_STEP_STAT } from './defs.js';
import styles from './progress-indicator.scss.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Icons, keyed by state.
 */
const icons = {
    [PROGRESS_STEP_STAT.COMPLETE]: CheckmarkOutline16,
    [PROGRESS_STEP_STAT.INCOMPLETE]: CircleDash16,
    [PROGRESS_STEP_STAT.INVALID]: Warning16,
    [PROGRESS_STEP_STAT.CURRENT]: Incomplete16,
};
/**
 * Progress step.
 *
 * @element cds-progress-step
 * @slot secondary-label-text - The secondary progress label.
 */
let CDSProgressStep = class CDSProgressStep extends FocusMixin(LitElement) {
    constructor() {
        super(...arguments);
        /**
         * `true` if the progress step should be disabled.
         */
        this.disabled = false;
        /**
         * The progress state.
         */
        this.state = PROGRESS_STEP_STAT.INCOMPLETE;
        /**
         * `true` if the progress step should be vertical.
         *
         * @private
         */
        this.vertical = false;
        /**
         * `true` if the progress step should be spaced equally.
         *
         * @private
         */
        this.spaceEqually = false;
    }
    connectedCallback() {
        if (!this.hasAttribute('role')) {
            this.setAttribute('role', 'listitem');
        }
        super.connectedCallback();
    }
    updated(changedProperties) {
        if (changedProperties.has('disabled')) {
            this.setAttribute('aria-disabled', String(Boolean(this.disabled)));
        }
    }
    render() {
        const { description, iconLabel, label, secondaryLabelText, secondaryLabel, state, } = this;
        const svgLabel = iconLabel || description;
        const optionalLabel = secondaryLabelText || secondaryLabel;
        return html `
      <div class="${prefix}--progress-step-button" tabindex="0">
        ${icons[state]({
            class: {
                [PROGRESS_STEP_STAT.INVALID]: `${prefix}--progress__warning`,
            }[state],
            children: svgLabel ? svg `<title>${svgLabel}</title>` : undefined,
        })}
        <slot name="label-text">
          <p
            role="button"
            class="${prefix}--progress-label"
            aria-describedby="label-tooltip"
            title="${label}">
            ${label}
          </p>
        </slot>
        <slot name="secondary-label-text">
          ${!optionalLabel
            ? undefined
            : html `<p class="${prefix}--progress-optional">
                ${optionalLabel}
              </p>`}
        </slot>
        <span class="${prefix}--progress-line"></span>
      </div>
    `;
    }
};
CDSProgressStep.shadowRootOptions = Object.assign(Object.assign({}, LitElement.shadowRootOptions), { delegatesFocus: true });
CDSProgressStep.styles = styles;
__decorate([
    property({ type: Boolean, reflect: true })
], CDSProgressStep.prototype, "disabled", void 0);
__decorate([
    property({ attribute: 'icon-label' })
], CDSProgressStep.prototype, "iconLabel", void 0);
__decorate([
    property({ reflect: true })
], CDSProgressStep.prototype, "description", void 0);
__decorate([
    property({ attribute: 'label-text' })
], CDSProgressStep.prototype, "labelText", void 0);
__decorate([
    property()
], CDSProgressStep.prototype, "label", void 0);
__decorate([
    property({ attribute: 'secondary-label-text' })
], CDSProgressStep.prototype, "secondaryLabelText", void 0);
__decorate([
    property({ attribute: 'secondary-label' })
], CDSProgressStep.prototype, "secondaryLabel", void 0);
__decorate([
    property()
], CDSProgressStep.prototype, "state", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSProgressStep.prototype, "vertical", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSProgressStep.prototype, "spaceEqually", void 0);
CDSProgressStep = __decorate([
    carbonElement(`${prefix}-progress-step`)
], CDSProgressStep);
var CDSProgressStep$1 = CDSProgressStep;

export { CDSProgressStep$1 as default };
//# sourceMappingURL=progress-step.js.map
