/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { html, LitElement } from 'lit';
import { property } from 'lit/decorators.js';
import { classMap } from 'lit/directives/class-map.js';
import { prefix } from '../../globals/settings.js';
import FormMixin from '../../globals/mixins/form.js';
import HostListenerMixin from '../../globals/mixins/host-listener.js';
import HostListener from '../../globals/decorators/host-listener.js';
import { find, forEach } from '../../globals/internal/collection-helpers.js';
import { RADIO_BUTTON_LABEL_POSITION, RADIO_BUTTON_ORIENTATION } from './defs.js';
export { RADIO_BUTTON_ORIENTATION } from './defs.js';
import WarningFilled16 from '../../icons/warning--filled/16.js';
import WarningAltFilled16 from '../../icons/warning--alt--filled/16.js';
import styles from './radio-button.scss.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Radio button group.
 *
 * @element cds-radio-button-group
 * @fires cds-radio-button-group-changed - The custom event fired after this radio button group changes its selected item.
 * @fires cds-radio-button-changed
 *   The name of the custom event fired after a radio button changes its checked state.
 */
let CDSRadioButtonGroup = class CDSRadioButtonGroup extends FormMixin(HostListenerMixin(LitElement)) {
    constructor() {
        super(...arguments);
        /**
         * Handles user-initiated change in selected radio button.
         */
        this._handleAfterChangeRadioButton = () => {
            const { selectorRadioButton } = this
                .constructor;
            const selected = find(this.querySelectorAll(selectorRadioButton), (elem) => elem.checked);
            const oldValue = this.value;
            this.value = selected && selected.value;
            if (oldValue !== this.value) {
                const { eventChange } = this.constructor;
                this.dispatchEvent(new CustomEvent(eventChange, {
                    bubbles: true,
                    composed: true,
                    detail: {
                        value: this.value,
                    },
                }));
            }
        };
        /**
         * `true` if there is an AI Label.
         */
        this._hasAILabel = false;
        /**
         * `true` if the radio button group should be disabled.
         */
        this.disabled = false;
        /**
         * The label position.
         */
        this.labelPosition = RADIO_BUTTON_LABEL_POSITION.RIGHT;
        /**
         * The label position.
         */
        this.legendText = '';
        /**
         * Specify whether the control is currently in warning state
         */
        this.warn = false;
        /**
         * Provide the text that is displayed when the control is in warning state
         */
        this.warnText = '';
        /**
         * Specify if the currently value is invalid.
         */
        this.invalid = false;
        /**
         * Message which is displayed if the value is invalid.
         */
        this.invalidText = '';
        /**
         * The orientation to lay out radio buttons.
         */
        this.orientation = RADIO_BUTTON_ORIENTATION.HORIZONTAL;
        /**
         * Controls the readonly state of the radio button group.
         */
        this.readOnly = false;
    }
    _handleFormdata(event) {
        const { formData } = event; // TODO: Wait for `FormDataEvent` being available in `lib.dom.d.ts`
        const { disabled, name, value } = this;
        if (!disabled &&
            typeof name !== 'undefined' &&
            typeof value !== 'undefined') {
            formData.append(name, value);
        }
    }
    /**
     * Handles `slotchange` event.
     */
    _handleSlotChange({ target }) {
        const hasContent = target
            .assignedNodes()
            .filter((elem) => elem.matches !== undefined
            ? elem.matches(this.constructor.aiLabelItem) ||
                // remove reference to slug in v12
                elem.matches(this.constructor.slugItem)
            : false);
        this._hasAILabel = Boolean(hasContent);
        hasContent[0].setAttribute('size', 'mini');
        this.requestUpdate();
    }
    updated(changedProperties) {
        const { selectorRadioButton } = this
            .constructor;
        ['disabled', 'labelPosition', 'orientation', 'readOnly', 'name'].forEach((name) => {
            if (changedProperties.has(name)) {
                const { [name]: value } = this;
                // Propagate the property to descendants until `:host-context()` gets supported in all major browsers
                forEach(this.querySelectorAll(selectorRadioButton), (elem) => {
                    elem[name] = value;
                });
            }
        });
        if (changedProperties.has('value')) {
            const { value } = this;
            forEach(this.querySelectorAll(selectorRadioButton), (elem) => {
                elem.checked =
                    value === elem.value;
            });
        }
        if (changedProperties.has('invalid')) {
            forEach(this.querySelectorAll(selectorRadioButton), (elem) => {
                elem.invalid = this.invalid;
            });
        }
    }
    render() {
        const { readOnly, invalid, invalidText, warn, warnText, disabled, orientation, legendText, helperText, _hasAILabel: hasAILabel, _handleSlotChange: handleSlotChange, } = this;
        const showWarning = !readOnly && !invalid && warn;
        const showHelper = !invalid && !disabled && !warn;
        const invalidIcon = WarningFilled16({
            class: `${prefix}--radio-button__invalid-icon`,
        });
        const warnIcon = WarningAltFilled16({
            class: `${prefix}--radio-button__invalid-icon ${prefix}--radio-button__invalid-icon--warning`,
        });
        const helper = helperText
            ? html `<div class="${prefix}--form__helper-text">${helperText}</div>`
            : null;
        const fieldsetClasses = classMap({
            [`${prefix}--radio-button-group`]: true,
            [`${prefix}--radio-button-group--readonly`]: readOnly,
            [`${prefix}--radio-button-group--${orientation}`]: orientation === 'vertical',
            [`${prefix}--radio-button-group--slug`]: hasAILabel,
        });
        return html ` <fieldset
        class="${fieldsetClasses}"
        ?disabled="${disabled}"
        aria-readonly="${readOnly}">
        ${legendText
            ? html ` <legend class="${prefix}--label">
              ${legendText}
              <slot name="ai-label" @slotchange="${handleSlotChange}"></slot>
              <slot name="slug" @slotchange="${handleSlotChange}"></slot>
            </legend>`
            : ``}
        <slot></slot>
      </fieldset>
      <div class="${prefix}--radio-button__validation-msg">
        ${!readOnly && invalid
            ? html `
              ${invalidIcon}
              <div class="${prefix}--form-requirement">${invalidText}</div>
            `
            : null}
        ${showWarning
            ? html `${warnIcon}
              <div class="${prefix}--form-requirement">${warnText}</div>`
            : null}
      </div>
      ${showHelper ? helper : null}`;
    }
    /**
     * A selector that will return the radio buttons.
     */
    static get selectorRadioButton() {
        return `${prefix}-radio-button`;
    }
    /**
     * A selector that will return the slug item.
     *
     * remove in v12
     */
    static get slugItem() {
        return `${prefix}-slug`;
    }
    /**
     * A selector that will return the AI Label item.
     */
    static get aiLabelItem() {
        return `${prefix}-ai-label`;
    }
    /**
     * The name of the custom event fired after this radio button group changes its selected item.
     */
    static get eventChange() {
        return `${prefix}-radio-button-group-changed`;
    }
    /**
     * The name of the custom event fired after a radio button changes its checked state.
     */
    static get eventChangeRadioButton() {
        return `${prefix}-radio-button-changed`;
    }
};
CDSRadioButtonGroup.styles = styles;
__decorate([
    HostListener('eventChangeRadioButton')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSRadioButtonGroup.prototype, "_handleAfterChangeRadioButton", void 0);
__decorate([
    property()
], CDSRadioButtonGroup.prototype, "defaultSelected", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSRadioButtonGroup.prototype, "disabled", void 0);
__decorate([
    property({ reflect: true, attribute: 'label-position' })
], CDSRadioButtonGroup.prototype, "labelPosition", void 0);
__decorate([
    property({ reflect: true, attribute: 'legend-text' })
], CDSRadioButtonGroup.prototype, "legendText", void 0);
__decorate([
    property({ attribute: 'helper-text' })
], CDSRadioButtonGroup.prototype, "helperText", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSRadioButtonGroup.prototype, "warn", void 0);
__decorate([
    property({ attribute: 'warn-text' })
], CDSRadioButtonGroup.prototype, "warnText", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSRadioButtonGroup.prototype, "invalid", void 0);
__decorate([
    property({ attribute: 'invalid-text' })
], CDSRadioButtonGroup.prototype, "invalidText", void 0);
__decorate([
    property()
], CDSRadioButtonGroup.prototype, "name", void 0);
__decorate([
    property({ reflect: true })
], CDSRadioButtonGroup.prototype, "orientation", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSRadioButtonGroup.prototype, "readOnly", void 0);
__decorate([
    property()
], CDSRadioButtonGroup.prototype, "value", void 0);
CDSRadioButtonGroup = __decorate([
    carbonElement(`${prefix}-radio-button-group`)
], CDSRadioButtonGroup);
var CDSRadioButtonGroup$1 = CDSRadioButtonGroup;

export { CDSRadioButtonGroup$1 as default };
//# sourceMappingURL=radio-button-group.js.map
