/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { LitElement, html } from 'lit';
import { query, property } from 'lit/decorators.js';
import { classMap } from 'lit/directives/class-map.js';
import ChevronDown16 from '../../icons/chevron--down/16.js';
import WarningFilled16 from '../../icons/warning--filled/16.js';
import WarningAltFilled16 from '../../icons/warning--alt--filled/16.js';
import { prefix } from '../../globals/settings.js';
import { ifDefined } from 'lit/directives/if-defined.js';
import FormMixin from '../../globals/mixins/form.js';
import { filter } from '../../globals/internal/collection-helpers.js';
import '../text-input/text-input.js';
import styles from './select.scss.js';
import ifNonEmpty from '../../globals/directives/if-non-empty.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';
import { INPUT_SIZE } from '../text-input/defs.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2020, 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Select box.
 *
 * @element cds-select
 * @fires cds-select-selected
 *   The name of the custom event fired after an item is selected.
 * @slot helper-text - The helper text.
 * @slot label-text - The label text.
 * @slot validity-message - The validity message. If present and non-empty, this input shows the UI of its invalid state.
 */
let CDSSelect = class CDSSelect extends FormMixin(LitElement) {
    constructor() {
        super(...arguments);
        /**
         * `true` if there is an AI Label.
         */
        this._hasAILabel = false;
        /**
         * The mutation observer DOM mutation.
         */
        this._observerMutation = null;
        /**
         * The `value` for placeholder `<option>`.
         */
        this._placeholderItemValue = `__${prefix}-select-placeholder_${Math.random()
            .toString(36)
            .slice(2)}`;
        /**
         * Handles DOM mutation of `<cds-select-item>` or `<cds-select-item-group>` put in `<cds-select>`, or their changes.
         * In such event, `<cds-select>` creates the corresponding `<option>` and `<optgroup>`, respectively, into shadow DOM,
         * with `._renderItems()`.
         * Doing so allows the shadow DOM style of `<cds-select>` to control the style of the `<option>` and `<optgroup>`,
         * notably the disabled ones.
         */
        this._handleMutation = () => {
            this.requestUpdate();
        };
        /**
         * Sets the select to be focussed automatically on page load. Defaults to false
         */
        this.autofocus = false;
        /**
         * Controls the disabled state of the select
         */
        this.disabled = false;
        /**
         * The helper text.
         */
        this.helperText = '';
        /**
         * Specify whether the label should be hidden, or not
         */
        this.hideLabel = false;
        /**
         * ID to link the `label` and `select`
         */
        this.id = '';
        /**
         * Specify if the currently value is invalid.
         */
        this.invalid = false;
        /**
         * Message which is displayed if the value is invalid.
         */
        this.invalidText = '';
        /**
         * Specify if the currently value is warn.
         */
        this.warn = false;
        /**
         * Message which is displayed if the value is warn.
         */
        this.warnText = '';
        /**
         * The label text.
         */
        this.labelText = '';
        /**
         * Specify whether you want the inline version of this control
         */
        this.inline = false;
        /**
         * Name for the select in the `FormData`
         */
        this.name = '';
        /**
         * Pattern to validate the select against for HTML validity checking
         */
        this.pattern = '';
        /**
         * Value to display when the select has an empty `value`
         */
        this.placeholder = '';
        /**
         * Controls the readonly state of the select
         */
        this.readonly = false;
        /**
         * Boolean property to set the required status
         */
        this.required = false;
        /**
         * The special validity message for `required`.
         */
        this.requiredValidityMessage = 'Please fill out this field.';
        /**
         * The input box size.
         */
        this.size = INPUT_SIZE.MEDIUM;
        /**
         * The value of the text area.
         */
        this.value = '';
    }
    /**
     * Handles `oninput` event on the `<input>`.
     *
     * @param event The event.
     * @param event.target The event target.
     */
    _handleInput({ target }) {
        const { value } = target;
        this.value = value;
        const { eventSelect } = this.constructor;
        this.dispatchEvent(new CustomEvent(eventSelect, {
            bubbles: true,
            composed: true,
            detail: {
                value,
            },
        }));
    }
    /**
     * @param element The parent element containing pseudo `<optgroup>`/`<option>`.
     * @returns The template containing child `<optgroup>`/`<option>` that will be rendered to shadow DOM.
     */
    _renderItems(element) {
        const { selectorItem, selectorLeafItem } = this
            .constructor;
        // Harvests attributes from `<cds-select-item>` and `<cds-select-item-group>`.
        // Does not use properties to avoid delay in attribute to property mapping, which runs in custom element reaction cycle:
        // https://html.spec.whatwg.org/multipage/custom-elements.html#custom-element-reactions
        return html `
      ${filter(element.childNodes, (item) => item.nodeType === Node.ELEMENT_NODE &&
            item.matches(selectorItem)).map((item) => {
            const disabled = item.hasAttribute('disabled');
            const label = item.getAttribute('label');
            const selected = item.hasAttribute('selected');
            const value = item.getAttribute('value');
            const { textContent } = item;
            return item.matches(selectorLeafItem)
                ? html `
              <option
                class="${prefix}--select-option"
                ?disabled="${disabled}"
                label="${ifNonEmpty(label)}"
                ?selected="${selected}"
                value="${ifDefined(value)}">
                ${textContent}
              </option>
            `
                : html `
              <optgroup
                class="${prefix}--select-optgroup"
                ?disabled="${disabled}"
                label="${ifDefined(label)}">
                ${this._renderItems(item)}
              </optgroup>
            `;
        })}
    `;
    }
    _handleFormdata(event) {
        const { formData } = event; // TODO: Wait for `FormDataEvent` being available in `lib.dom.d.ts`
        const { disabled, name, value } = this;
        if (!disabled) {
            formData.append(name, value);
        }
    }
    /**
     * Handles `slotchange` event.
     */
    _handleAILabelSlotChange({ target }) {
        const hasContent = target
            .assignedNodes()
            .filter((elem) => elem.matches !== undefined
            ? elem.matches(this.constructor.aiLabelItem) ||
                // remove reference to slug in v12
                elem.matches(this.constructor.slugItem)
            : false);
        this._hasAILabel = Boolean(hasContent);
        hasContent[0].setAttribute('size', 'mini');
        this.requestUpdate();
    }
    /**
     * The count of child `<option>`s.
     * If the placeholder is in effect, it includes the `<option>` for the placeholder.
     */
    get length() {
        return this._selectNode.length;
    }
    /**
     * The child `<option>`s.
     */
    get options() {
        return this._selectNode.options;
    }
    /**
     * This form control's type.
     */
    get type() {
        return this._selectNode.type;
    }
    /**
     * `true` to enable multiple selection.
     */
    get multiple() {
        return false;
    }
    /**
     * The selected index.
     */
    get selectedIndex() {
        var _a;
        return (_a = this._selectNode) === null || _a === void 0 ? void 0 : _a.selectedIndex;
    }
    set selectedIndex(value) {
        this._selectNode.selectedIndex = value;
        this.value = this._selectNode.value;
    }
    connectedCallback() {
        super.connectedCallback();
        this._observerMutation = new MutationObserver(this._handleMutation);
        this._observerMutation.observe(this, {
            attributes: true,
            childList: true,
            subtree: true,
        });
    }
    disconnectedCallback() {
        if (this._observerMutation) {
            this._observerMutation.disconnect();
            this._observerMutation = null;
        }
        super.disconnectedCallback();
    }
    updated(changedProperties) {
        var _a, _b, _c, _d, _e;
        if (changedProperties.has('value')) {
            const { value, _placeholderItemValue: placeholderItemValue } = this;
            // Ensures setting the `value` after rendering child `<option>`s/`<optgroup>`s when there is a change in `value`,
            // given reflecting `value` requires child `<option>`s/`<optgroup>`s being there beforehand
            this._selectNode.value = !value ? placeholderItemValue : value;
        }
        const label = (_a = this.shadowRoot) === null || _a === void 0 ? void 0 : _a.querySelector("slot[name='ai-label']");
        if (label) {
            label === null || label === void 0 ? void 0 : label.classList.toggle(`${prefix}--slug--revert`, (_b = this.querySelector(`${prefix}-ai-label`)) === null || _b === void 0 ? void 0 : _b.hasAttribute('revert-active'));
        }
        else {
            (_d = (_c = this.shadowRoot) === null || _c === void 0 ? void 0 : _c.querySelector("slot[name='slug']")) === null || _d === void 0 ? void 0 : _d.classList.toggle(`${prefix}--slug--revert`, (_e = this.querySelector(`${prefix}-slug`)) === null || _e === void 0 ? void 0 : _e.hasAttribute('revert-active'));
        }
    }
    render() {
        const { disabled, helperText, hideLabel, inline, invalid, invalidText, labelText, placeholder, readonly, size, warn, warnText, value, _placeholderItemValue: placeholderItemValue, _handleInput: handleInput, _handleAILabelSlotChange: handleAILabelSlotChange, _hasAILabel: hasAILabel, } = this;
        const selectClasses = classMap({
            [`${prefix}--select`]: true,
            [`${prefix}--select--inline`]: inline,
            [`${prefix}--select--invalid`]: invalid,
            [`${prefix}--select--warning`]: warn,
            [`${prefix}--select--disabled`]: disabled,
            [`${prefix}--select--readonly`]: readonly,
            [`${prefix}--select--slug`]: hasAILabel,
        });
        const inputClasses = classMap({
            [`${prefix}--select-input`]: true,
            [`${prefix}--select-input--${size}`]: size,
        });
        const labelClasses = classMap({
            [`${prefix}--label`]: true,
            [`${prefix}--label--disabled`]: disabled,
        });
        const helperTextClasses = classMap({
            [`${prefix}--form__helper-text`]: true,
            [`${prefix}--form__helper-text--disabled`]: disabled,
        });
        const supplementalText = helperText
            ? html `
          <div class="${helperTextClasses}">
            <slot name="helper-text"> ${helperText} </slot>
          </div>
        `
            : null;
        const errorText = invalid || warn
            ? html ` <div class="${prefix}--form-requirement">
            ${invalid ? invalidText : warnText}
          </div>`
            : null;
        const input = html `
      <select
        id="input"
        class="${inputClasses}"
        ?disabled="${disabled}"
        aria-readonly="${String(Boolean(readonly))}"
        aria-invalid="${String(Boolean(invalid))}"
        aria-describedby="${ifDefined(!invalid ? undefined : 'invalid-text')}"
        @input="${handleInput}">
        ${!placeholder || value
            ? undefined
            : html `
              <option
                disabled
                hidden
                class="${prefix}--select-option"
                value="${placeholderItemValue}"
                selected>
                ${placeholder}
              </option>
            `}
        ${this._renderItems(this)}
      </select>
      ${ChevronDown16({ class: `${prefix}--select__arrow` })}
      <slot name="ai-label" @slotchange=${handleAILabelSlotChange}></slot>
      <slot name="slug" @slotchange=${handleAILabelSlotChange}></slot>
      ${!invalid
            ? undefined
            : WarningFilled16({ class: `${prefix}--select__invalid-icon` })}
      ${!invalid && warn
            ? WarningAltFilled16({
                class: `${prefix}--select__invalid-icon ${prefix}--select__invalid-icon--warning`,
            })
            : null}
    `;
        return html `
      <div class="${selectClasses}">
        ${!hideLabel
            ? html `<label class="${labelClasses}" for="input">
              <slot name="label-text"> ${labelText} </slot>
            </label>`
            : null}
        ${inline
            ? html `<div class="${prefix}--select-input--inline__wrapper">
              <div
                class="${prefix}--select-input__wrapper"
                ?data-invalid="${invalid}">
                ${input}
              </div>
              ${errorText}
            </div>`
            : html `<div
              class="${prefix}--select-input__wrapper"
              ?data-invalid="${invalid}">
              ${input}
            </div> `}
        ${!inline && errorText ? errorText : supplementalText}
      </div>
    `;
    }
    /**
     * A selector selecting child pseudo `<optgroup>`/`<option>`.
     */
    static get selectorItem() {
        return `${prefix}-select-item-group,${prefix}-select-item`;
    }
    /**
     * A selector selecting child pseudo `<option>`.
     */
    static get selectorLeafItem() {
        return `${prefix}-select-item`;
    }
    /**
     * A selector that will return the slug item.
     *
     * remove in v12
     */
    static get slugItem() {
        return `${prefix}-slug`;
    }
    /**
     * A selector that will return the AI Label item.
     */
    static get aiLabelItem() {
        return `${prefix}-ai-label`;
    }
    /**
     * The name of the custom event fired after item is selected.
     */
    static get eventSelect() {
        return `${prefix}-select-selected`;
    }
};
CDSSelect.shadowRootOptions = Object.assign(Object.assign({}, LitElement.shadowRootOptions), { delegatesFocus: true });
CDSSelect.styles = styles; // `styles` here is a `CSSResult` generated by custom Vite loader
__decorate([
    query('select')
], CDSSelect.prototype, "_selectNode", void 0);
__decorate([
    property({ type: Boolean })
], CDSSelect.prototype, "autofocus", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSSelect.prototype, "disabled", void 0);
__decorate([
    property({ attribute: 'helper-text' })
], CDSSelect.prototype, "helperText", void 0);
__decorate([
    property({ type: Boolean, attribute: 'hide-label' })
], CDSSelect.prototype, "hideLabel", void 0);
__decorate([
    property()
], CDSSelect.prototype, "id", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSSelect.prototype, "invalid", void 0);
__decorate([
    property({ attribute: 'invalid-text' })
], CDSSelect.prototype, "invalidText", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSSelect.prototype, "warn", void 0);
__decorate([
    property({ attribute: 'warn-text' })
], CDSSelect.prototype, "warnText", void 0);
__decorate([
    property({ attribute: 'label-text' })
], CDSSelect.prototype, "labelText", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSSelect.prototype, "inline", void 0);
__decorate([
    property({ type: Boolean })
    // eslint-disable-next-line class-methods-use-this
], CDSSelect.prototype, "multiple", null);
__decorate([
    property()
], CDSSelect.prototype, "name", void 0);
__decorate([
    property()
], CDSSelect.prototype, "pattern", void 0);
__decorate([
    property({ reflect: true })
], CDSSelect.prototype, "placeholder", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSSelect.prototype, "readonly", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSSelect.prototype, "required", void 0);
__decorate([
    property({ attribute: 'required-validity-message' })
], CDSSelect.prototype, "requiredValidityMessage", void 0);
__decorate([
    property({ type: Number })
], CDSSelect.prototype, "selectedIndex", null);
__decorate([
    property({ reflect: true })
], CDSSelect.prototype, "size", void 0);
__decorate([
    property({ reflect: true })
], CDSSelect.prototype, "value", void 0);
CDSSelect = __decorate([
    carbonElement(`${prefix}-select`)
], CDSSelect);
var CDSSelect$1 = CDSSelect;

export { CDSSelect$1 as default };
//# sourceMappingURL=select.js.map
