/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { classMap } from 'lit/directives/class-map.js';
import { LitElement, html } from 'lit';
import { property } from 'lit/decorators.js';
import { ifDefined } from 'lit/directives/if-defined.js';
import { prefix } from '../../globals/settings.js';
import WarningFilled16 from '../../icons/warning--filled/16.js';
import WarningAltFilled16 from '../../icons/warning--alt--filled/16.js';
import FocusMixin from '../../globals/mixins/focus.js';
import styles from './slider.scss.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2023
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * The `<input>` box for slider.
 *
 * @element cds-slider-input
 * @fires cds-slider-input-changed - The custom event fired after the value is changed by user gesture.
 */
let CDSSliderInput = class CDSSliderInput extends FocusMixin(LitElement) {
    constructor() {
        super(...arguments);
        /**
         * The internal value of `max` property.
         */
        this._max = '100';
        /**
         * The internal value of `min` property.
         */
        this._min = '0';
        /**
         * The internal value of `step` property.
         */
        this._step = '1';
        /**
         * `true` if the input should be disabled.
         */
        this.disabled = false;
        /**
         * true to specify if the control is invalid.
         */
        this.invalid = false;
        /**
         * true to specify if the control should display warn icon and text.
         */
        this.warn = false;
        /**
         * The type of the `<input>`.
         */
        this.type = 'number';
        /**
         * true` if the input should be readonly.
         */
        this.readonly = false;
    }
    /**
     * Handles `change` event to fire a normalized custom event.
     */
    _handleChange({ target }) {
        this.dispatchEvent(new CustomEvent(this.constructor.eventChange, {
            bubbles: true,
            composed: true,
            detail: {
                value: Number(target.value),
            },
        }));
    }
    /**
     * Handles `input` event to fire a normalized custom event.
     */
    _handleInput({ target }) {
        this.dispatchEvent(new CustomEvent(this.constructor.eventChange, {
            bubbles: true,
            composed: true,
            detail: {
                value: Number(target.value),
                intermediate: true,
            },
        }));
    }
    /**
     * The maximum value.
     */
    get max() {
        return this._max.toString();
    }
    set max(value) {
        const { max: oldMax } = this;
        this._max = value;
        this.requestUpdate('max', oldMax);
    }
    /**
     * The minimum value.
     */
    get min() {
        return this._min.toString();
    }
    set min(value) {
        const { min: oldMin } = this;
        this._min = value;
        this.requestUpdate('min', oldMin);
    }
    /**
     * The snapping step of the value.
     */
    get step() {
        return this._step.toString();
    }
    set step(value) {
        const { step: oldStep } = this;
        this._step = value;
        this.requestUpdate('step', oldStep);
    }
    render() {
        const { disabled, max, min, readonly, step, type, value, invalid, warn, _handleChange: handleChange, _handleInput: handleInput, } = this;
        const classes = classMap({
            [`${prefix}--text-input`]: true,
            [`${prefix}--slider-text-input`]: true,
            [`${prefix}--text-input--invalid`]: invalid,
            [`${prefix}--slider-text-input--warn`]: warn,
        });
        const invalidIcon = WarningFilled16({
            class: `${prefix}--slider__invalid-icon`,
        });
        const warnIcon = WarningAltFilled16({
            class: `${prefix}--slider__invalid-icon ${prefix}--slider__invalid-icon--warning`,
        });
        return html `
      <input
        ?disabled="${disabled}"
        ?data-invalid="${invalid}"
        type="${ifDefined(type)}"
        class="${classes}"
        max="${max}"
        min="${min}"
        ?readonly="${ifDefined(readonly)}"
        step="${step}"
        .value="${value}"
        @change="${handleChange}"
        @input="${handleInput}" />
      ${invalid ? html `${invalidIcon}` : null}
      ${warn ? html `${warnIcon}` : null}
    `;
    }
    /**
     * A selector that will return the parent slider.
     */
    static get selectorParent() {
        return `${prefix}-slider`;
    }
    /**
     * The name of the custom event fired after the value is changed by user gesture.
     */
    static get eventChange() {
        return `${prefix}-slider-input-changed`;
    }
};
CDSSliderInput.shadowRootOptions = Object.assign(Object.assign({}, LitElement.shadowRootOptions), { delegatesFocus: true });
CDSSliderInput.styles = styles;
__decorate([
    property({ type: Boolean, reflect: true })
], CDSSliderInput.prototype, "disabled", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSSliderInput.prototype, "invalid", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSSliderInput.prototype, "warn", void 0);
__decorate([
    property({ type: Number, reflect: true })
], CDSSliderInput.prototype, "max", null);
__decorate([
    property({ type: Number, reflect: true })
], CDSSliderInput.prototype, "min", null);
__decorate([
    property({ type: Number, reflect: true })
], CDSSliderInput.prototype, "step", null);
__decorate([
    property()
], CDSSliderInput.prototype, "type", void 0);
__decorate([
    property({ type: Number })
], CDSSliderInput.prototype, "value", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSSliderInput.prototype, "readonly", void 0);
CDSSliderInput = __decorate([
    carbonElement(`${prefix}-slider-input`)
], CDSSliderInput);
var CDSSliderInput$1 = CDSSliderInput;

export { CDSSliderInput$1 as default };
//# sourceMappingURL=slider-input.js.map
