/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { html, LitElement } from 'lit';
import { query, property } from 'lit/decorators.js';
import { classMap } from 'lit/directives/class-map.js';
import throttle from 'lodash-es/throttle';
import { prefix } from '../../globals/settings.js';
import FocusMixin from '../../globals/mixins/focus.js';
import FormMixin from '../../globals/mixins/form.js';
import HostListenerMixin from '../../globals/mixins/host-listener.js';
import HostListener from '../../globals/decorators/host-listener.js';
import styles from './slider.scss.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * The direction to move the thumb, associated with key symbols.
 */
const THUMB_DIRECTION = {
    Left: -1,
    ArrowLeft: -1,
    Up: -1,
    ArrowUp: -1,
    Right: 1,
    ArrowRight: 1,
    Down: 1,
    ArrowDown: 1,
};
/**
 * Slider.
 *
 * @element cds-slider
 * @fires cds-slider-input-changed
 *   The name of the custom event fired after the value is changed in `<cds-slider-input>` by user gesture.
 * @slot label-text - The label text.
 * @slot max-text - The text for maximum value.
 * @slot min-text - The text for minimum value.
 * @fires cds-slider-changed - The custom event fired after the value is changed by user gesture.
 */
let CDSSlider = class CDSSlider extends HostListenerMixin(FormMixin(FocusMixin(LitElement))) {
    constructor() {
        super(...arguments);
        /**
         * The internal value of `max` property.
         */
        this._max = '100';
        /**
         * The internal value of `min` property.
         */
        this._min = '0';
        /**
         * The internal value of `step` property.
         */
        this._step = '1';
        /**
         * The internal value of `stepMultiplier` property.
         */
        this._stepMultiplier = '4';
        /**
         * The handle for the throttled listener of `pointermove` event.
         */
        this._throttledHandlePointermoveImpl = null;
        /**
         * `true` if dragging of thumb is in progress.
         */
        this._dragging = false;
        /**
         * Handles `pointermove` to update the thumb position and the value as necessary.
         *
         * @param event The event.
         */
        this._handlePointermove = (event) => {
            const { disabled, _dragging: dragging } = this;
            if (!disabled && dragging) {
                this._throttledHandlePointermoveImpl(event);
            }
        };
        /**
         * Handles `pointerup` and `pointerleave` event to finishing dragging.
         */
        this._endDrag = () => {
            if (this._dragging) {
                this.dispatchEvent(new CustomEvent(this.constructor.eventChange, {
                    bubbles: true,
                    composed: true,
                    detail: {
                        value: this.value,
                    },
                }));
                this._dragging = false;
                this._thumbNode.style.touchAction = '';
            }
        };
        /**
         * Handles `${prefix}-slider-input-changed` event to update the value.
         */
        this._handleChangeInput = ({ detail }) => {
            const { intermediate, value } = detail;
            this.value = value;
            this.dispatchEvent(new CustomEvent(this.constructor.eventChange, {
                bubbles: true,
                composed: true,
                detail: {
                    value,
                    intermediate,
                },
            }));
        };
        /**
         * `true` if the check box should be disabled.
         */
        this.disabled = false;
        /**
         * 	true to specify if the control is required.
         */
        this.required = false;
        /**
         * 	Whether the slider should be read-only
         */
        this.readonly = false;
        /**
         * The label text.
         */
        this.labelText = '';
        /**
         * The label associated with the maximum value.
         */
        this.maxLabel = '';
        /**
         * The label associated with the minimum value.
         */
        this.minLabel = '';
        /**
         * The formatter for the text for maximum value.
         * Should be changed upon the locale the UI is rendered with.
         */
        this.formatMaxText = (max, maxLabel) => `${max}${maxLabel}`;
        /**
         * The formatter for the text for min/max value.
         * Should be changed upon the locale the UI is rendered with.
         */
        this.formatMinText = (min, minLabel) => `${min}${minLabel}`;
        /**
         * true to specify if the control is invalid.
         */
        this.invalid = false;
        /**
         * Message which is displayed if the value is invalid.
         */
        this.invalidText = '';
        /**
         * true to specify if the control should display warn icon and text.
         */
        this.warn = false;
        /**
         * Provide the text that is displayed when the control is in warning state
         */
        this.warnText = '';
    }
    /**
     * The rate of the thumb position in the track.
     * When we try to set a new value, we adjust the value considering `step` property.
     */
    get _rate() {
        const { max, min, value } = this;
        // Copes with out-of-range value coming programmatically or from `<cds-slider-input>`
        return ((Math.min(Number(max), Math.max(Number(min), value)) - Number(min)) /
            (Number(max) - Number(min)));
    }
    set _rate(rate) {
        const { max, min, step } = this;
        this.value =
            Number(min) +
                Math.round(((Number(max) - Number(min)) * Math.min(1, Math.max(0, rate))) /
                    Number(step)) *
                    Number(step);
    }
    /**
     * Handles `click` event on the `<label>` to focus on the thumb.
     */
    _handleClickLabel() {
        var _a;
        (_a = this._thumbNode) === null || _a === void 0 ? void 0 : _a.focus();
    }
    _handleFormdata(event) {
        const { formData } = event; // TODO: Wait for `FormDataEvent` being available in `lib.dom.d.ts`
        const { disabled, name, value } = this;
        if (!disabled) {
            formData.append(name, String(value));
        }
    }
    /**
     * Handles `keydown` event on the thumb to increase/decrease the value.
     */
    _handleKeydown({ key, shiftKey }) {
        if (!this.disabled) {
            if (key in THUMB_DIRECTION) {
                const { max: rawMax, min: rawMin, step: rawStep, stepMultiplier: rawstepMultiplier, value, } = this;
                const max = Number(rawMax);
                const min = Number(rawMin);
                const step = Number(rawStep);
                const stepMultiplier = Number(rawstepMultiplier);
                const diff = (!shiftKey ? step : (max - min) / stepMultiplier) *
                    THUMB_DIRECTION[key];
                const stepCount = (value + diff) / step;
                // Snaps to next
                this.value = Math.min(max, Math.max(min, (diff >= 0 ? Math.floor(stepCount) : Math.ceil(stepCount)) * step));
                this.dispatchEvent(new CustomEvent(this.constructor.eventChange, {
                    bubbles: true,
                    composed: true,
                    detail: {
                        value: this.value,
                        intermediate: false,
                    },
                }));
            }
        }
    }
    /**
     * Handles `pointerdown` event on the thumb to start dragging.
     */
    _startDrag() {
        this._dragging = true;
        this._thumbNode.style.touchAction = 'none';
    }
    /**
     * Handles `pointerdown` event on the track to update the thumb position and the value as necessary.
     */
    _handleClick(event) {
        if (!this.disabled) {
            const { _trackNode: trackNode } = this;
            const isRtl = trackNode
                .ownerDocument.defaultView.getComputedStyle(trackNode)
                .getPropertyValue('direction') === 'rtl';
            const thumbPosition = event.clientX;
            const { left: trackLeft, width: trackWidth } = trackNode.getBoundingClientRect();
            this._rate =
                (isRtl
                    ? trackLeft + trackWidth - thumbPosition
                    : thumbPosition - trackLeft) / trackWidth;
            this.dispatchEvent(new CustomEvent(this.constructor.eventChange, {
                bubbles: true,
                composed: true,
                detail: {
                    value: this.value,
                },
            }));
        }
    }
    /**
     * Updates thumb position and value upon user's `pointermove` gesture.
     *
     * @param event The event.
     */
    _handlePointermoveImpl(event) {
        const { disabled, _dragging: dragging, _trackNode: trackNode } = this;
        if (!disabled && dragging) {
            const isRtl = trackNode
                .ownerDocument.defaultView.getComputedStyle(trackNode)
                .getPropertyValue('direction') === 'rtl';
            const thumbPosition = event.clientX;
            const { left: trackLeft, width: trackWidth } = this._trackNode.getBoundingClientRect();
            this._rate =
                (isRtl
                    ? trackLeft + trackWidth - thumbPosition
                    : thumbPosition - trackLeft) / trackWidth;
            this.dispatchEvent(new CustomEvent(this.constructor.eventChange, {
                bubbles: true,
                composed: true,
                detail: {
                    value: this.value,
                    intermediate: true,
                },
            }));
        }
    }
    /**
     * The maximum value.
     */
    get max() {
        return this._max.toString();
    }
    set max(value) {
        const { max: oldMax } = this;
        this._max = value;
        this.requestUpdate('max', oldMax);
    }
    /**
     * The minimum value.
     */
    get min() {
        return this._min.toString();
    }
    set min(value) {
        const { min: oldMin } = this;
        this._min = value;
        this.requestUpdate('min', oldMin);
    }
    /**
     * The snapping step of the value.
     */
    get step() {
        return this._step.toString();
    }
    set step(value) {
        const { step: oldStep } = this;
        this._step = value;
        this.requestUpdate('step', oldStep);
    }
    /**
     * A value determining how much the value should increase/decrease by Shift+arrow keys,
     * which will be `(max - min) / stepMultiplier`.
     */
    get stepMultiplier() {
        return this._stepMultiplier.toString();
    }
    set stepMultiplier(value) {
        const { stepMultiplier: oldstepMultiplier } = this;
        this._stepMultiplier = value;
        this.requestUpdate('stepMultiplier', oldstepMultiplier);
    }
    _getInputValidity(input) {
        if (this.invalid) {
            return false;
        }
        if ((input === null || input === void 0 ? void 0 : input.valueAsNumber) > Number(this.max) ||
            (input === null || input === void 0 ? void 0 : input.valueAsNumber) < Number(this.min)) {
            return false;
        }
        return true;
    }
    connectedCallback() {
        super.connectedCallback();
        if (!this._throttledHandlePointermoveImpl) {
            this._throttledHandlePointermoveImpl = throttle(this._handlePointermoveImpl, 10);
        }
    }
    disconnectedCallback() {
        if (this._throttledHandlePointermoveImpl) {
            this._throttledHandlePointermoveImpl.cancel();
            this._throttledHandlePointermoveImpl = null;
        }
        super.disconnectedCallback();
    }
    shouldUpdate(changedProperties) {
        var _a;
        const input = this.querySelector(this.constructor.selectorInput);
        if (changedProperties.has('disabled')) {
            if (input) {
                input.disabled = this.disabled;
            }
            if (this.disabled) {
                this._dragging = false;
            }
        }
        if (changedProperties.has('readonly')) {
            if (input) {
                input.readonly = this.readonly;
            }
            if (this.readonly) {
                this._dragging = false;
            }
        }
        if (input) {
            ['max', 'min', 'step', 'value'].forEach((name) => {
                if (changedProperties.has(name)) {
                    input[name] = this[name];
                }
            });
            if (changedProperties.has('value') ||
                changedProperties.has('invalid') ||
                changedProperties.has('warn') ||
                changedProperties.has('readonly')) {
                const innerInput = (_a = input === null || input === void 0 ? void 0 : input.shadowRoot) === null || _a === void 0 ? void 0 : _a.querySelector('input');
                this.isValid = this._getInputValidity(innerInput);
                if (!this.readonly && !this.isValid) {
                    input.invalid = true;
                }
                else {
                    input.invalid = false;
                }
                if (!this.readonly && !this.invalid && this.warn && this.isValid) {
                    input.warn = true;
                }
                else {
                    input.warn = false;
                }
            }
        }
        return true;
    }
    render() {
        const { disabled, formatMaxText, formatMinText, labelText, max, min, maxLabel, minLabel, readonly, invalidText, isValid, warn, warnText, value, _rate: rate, _handleClickLabel: handleClickLabel, _handleKeydown: handleKeydown, _handleClick: handleClick, _startDrag: startDrag, _endDrag: endDrag, } = this;
        const labelClasses = classMap({
            [`${prefix}--label`]: true,
            [`${prefix}--label--disabled`]: disabled,
        });
        const sliderClasses = classMap({
            [`${prefix}--slider`]: true,
            [`${prefix}--slider--disabled`]: disabled,
            [`${prefix}--slider--readonly`]: readonly,
        });
        return html `
      <label class="${labelClasses}" @click="${handleClickLabel}">
        <slot name="label-text">${labelText}</slot>
      </label>
      <div class="${prefix}--slider-container">
        <span class="${prefix}--slider__range-label">
          <slot name="min-text">${formatMinText(min, minLabel)}</slot>
        </span>
        <div
          @keydown="${handleKeydown}"
          @click="${handleClick}"
          @pointerup="${endDrag}"
          @pointerleave="${endDrag}"
          class="${sliderClasses}"
          tabindex="-1"
          role="presentation">
          <div
            id="thumb"
            class="${prefix}--slider__thumb"
            role="slider"
            tabindex="${!readonly ? 0 : -1}"
            aria-valuemax="${max}"
            aria-valuemin="${min}"
            aria-valuenow="${value}"
            style="left: ${rate * 100}%"
            @pointerdown="${startDrag}"></div>
          <div id="track" class="${prefix}--slider__track"></div>
          <div class="${prefix}-ce--slider__filled-track-container">
            <div
              class="${prefix}--slider__filled-track"
              style="transform: translate(0%, -50%) scaleX(${rate})"></div>
          </div>
        </div>
        <span class="${prefix}--slider__range-label">
          <slot name="max-text">${formatMaxText(max, maxLabel)}</slot>
        </span>
        <slot></slot>
      </div>

      ${!readonly && !isValid
            ? html `
            <div
              class="${prefix}--slider__validation-msg ${prefix}--slider__validation-msg--invalid ${prefix}--form-requirement">
              ${invalidText}
            </div>
          `
            : null}
      ${!readonly && warn && isValid
            ? html `<div
            class="${prefix}--slider__validation-msg ${prefix}--form-requirement">
            ${warnText}
          </div>`
            : null}
    `;
    }
    /**
     * A selector that will return the `<input>` box got entering the value directly.
     */
    static get selectorInput() {
        return `${prefix}-slider-input`;
    }
    /**
     * The name of the custom event fired after the value is changed by user gesture.
     */
    static get eventChange() {
        return `${prefix}-slider-changed`;
    }
    /**
     * The name of the custom event fired after the value is changed in `<cds-slider-input>` by user gesture.
     */
    static get eventChangeInput() {
        return `${prefix}-slider-input-changed`;
    }
};
CDSSlider.shadowRootOptions = Object.assign(Object.assign({}, LitElement.shadowRootOptions), { delegatesFocus: true });
CDSSlider.styles = styles;
__decorate([
    query('#thumb')
], CDSSlider.prototype, "_thumbNode", void 0);
__decorate([
    query('#track')
], CDSSlider.prototype, "_trackNode", void 0);
__decorate([
    HostListener('document:pointermove')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSSlider.prototype, "_handlePointermove", void 0);
__decorate([
    HostListener('eventChangeInput')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSSlider.prototype, "_handleChangeInput", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSSlider.prototype, "disabled", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSSlider.prototype, "required", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSSlider.prototype, "readonly", void 0);
__decorate([
    property({ attribute: 'label-text' })
], CDSSlider.prototype, "labelText", void 0);
__decorate([
    property({ attribute: 'max-label', reflect: true })
], CDSSlider.prototype, "maxLabel", void 0);
__decorate([
    property({ attribute: 'min-label', reflect: true })
], CDSSlider.prototype, "minLabel", void 0);
__decorate([
    property({ attribute: false })
], CDSSlider.prototype, "formatMaxText", void 0);
__decorate([
    property({ attribute: false })
], CDSSlider.prototype, "formatMinText", void 0);
__decorate([
    property({ type: Number, reflect: true })
], CDSSlider.prototype, "max", null);
__decorate([
    property({ type: Number, reflect: true })
], CDSSlider.prototype, "min", null);
__decorate([
    property()
], CDSSlider.prototype, "name", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSSlider.prototype, "invalid", void 0);
__decorate([
    property({ attribute: 'invalid-text' })
], CDSSlider.prototype, "invalidText", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSSlider.prototype, "warn", void 0);
__decorate([
    property({ attribute: 'warn-text' })
], CDSSlider.prototype, "warnText", void 0);
__decorate([
    property({ type: Number, reflect: true })
], CDSSlider.prototype, "step", null);
__decorate([
    property({ type: Number, reflect: true, attribute: 'step-multiplier' })
], CDSSlider.prototype, "stepMultiplier", null);
__decorate([
    property({ type: Number })
], CDSSlider.prototype, "value", void 0);
__decorate([
    property({ type: Boolean })
], CDSSlider.prototype, "isValid", void 0);
CDSSlider = __decorate([
    carbonElement(`${prefix}-slider`)
], CDSSlider);
var CDSSlider$1 = CDSSlider;

export { CDSSlider$1 as default };
//# sourceMappingURL=slider.js.map
