/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { LitElement, html } from 'lit';
import { property } from 'lit/decorators.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';
import { prefix } from '../../globals/settings.js';
import styles from './stack.scss.js';
import { SPACING_STEPS, STACK_ORIENTATION } from './defs.js';
export { SPACING_STEPS, STACK_ORIENTATION } from './defs.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2023, 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * The Stack component is a useful layout utility in a component-based model.
 * This allows components to not use margin and instead delegate the
 * responsibility of positioning and layout to parent components.
 *
 * In the case of the Stack component, it uses the spacing scale from the
 * Design Language in order to determine how much space there should be between
 * items rendered by the Stack component. It also supports a custom `gap` prop
 * which will allow a user to provide a custom value for the gap of the layout.
 *
 * This component supports both horizontal and vertical orientations.
 *
 * @element cds-stack
 */
let CDSStack = class CDSStack extends LitElement {
    constructor() {
        super(...arguments);
        /**
         * Specify the orientation of them items in the Stack
         */
        this.orientation = STACK_ORIENTATION.VERTICAL;
        /**
         * Turn on when passing in custom value to 'gap' attribute (ie. gap="2rem")
         */
        this.useCustomGapValue = false;
    }
    updated(changedProperties) {
        var _a, _b, _c, _d;
        if (changedProperties.has('gap')) {
            if (this.useCustomGapValue) {
                this.style.setProperty(`--${prefix}-stack-gap`, `${this.gap}`);
            }
            else {
                const oldGapValue = changedProperties.get('gap');
                (_b = (_a = this.shadowRoot) === null || _a === void 0 ? void 0 : _a.querySelector('div')) === null || _b === void 0 ? void 0 : _b.classList.remove(`${prefix}--stack-scale-${oldGapValue}`);
                (_d = (_c = this.shadowRoot) === null || _c === void 0 ? void 0 : _c.querySelector('div')) === null || _d === void 0 ? void 0 : _d.classList.add(`${prefix}--stack-scale-${this.gap}`);
            }
        }
    }
    render() {
        return html `<div><slot></slot></div>`;
    }
};
CDSStack.styles = styles; // `styles` here is a `CSSResult` generated by custom Vite loader
__decorate([
    property({ type: String, reflect: true })
], CDSStack.prototype, "orientation", void 0);
__decorate([
    property({ type: SPACING_STEPS || String, reflect: true })
], CDSStack.prototype, "gap", void 0);
__decorate([
    property({ type: Boolean, attribute: 'use-custom-gap-value' })
], CDSStack.prototype, "useCustomGapValue", void 0);
CDSStack = __decorate([
    carbonElement(`${prefix}-stack`)
], CDSStack);
var CDSStack$1 = CDSStack;

export { CDSStack$1 as default };
//# sourceMappingURL=stack.js.map
