/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { html, LitElement } from 'lit';
import { query, property } from 'lit/decorators.js';
import { ifDefined } from 'lit/directives/if-defined.js';
import CheckmarkFilled16 from '../../icons/checkmark--filled/16.js';
import { prefix } from '../../globals/settings.js';
import HostListener from '../../globals/decorators/host-listener.js';
import HostListenerMixin from '../../globals/mixins/host-listener.js';
import RadioGroupManager, { NAVIGATION_DIRECTION } from '../../globals/internal/radio-group-manager.js';
import styles from './structured-list.scss.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2023
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Map of navigation direction by key.
 */
const navigationDirectionForKey = {
    ArrowUp: NAVIGATION_DIRECTION.BACKWARD,
    Up: NAVIGATION_DIRECTION.BACKWARD,
    ArrowDown: NAVIGATION_DIRECTION.FORWARD,
    Down: NAVIGATION_DIRECTION.FORWARD, // IE
};
/**
 * The interface for `RadioGroupManager` for structured list row.
 */
class StructuredListRowRadioButtonDelegate {
    constructor(row) {
        this._row = row;
    }
    get checked() {
        return this._row.selected;
    }
    set checked(checked) {
        this._row.selected = checked;
        this._row.tabIndex = checked ? 0 : -1;
    }
    get tabIndex() {
        return this._row.tabIndex;
    }
    set tabIndex(tabIndex) {
        this._row.tabIndex = tabIndex;
    }
    get name() {
        return this._row.selectionName;
    }
    compareDocumentPosition(other) {
        return this._row.compareDocumentPosition(other._row);
    }
    focus() {
        this._row.focus();
    }
}
/**
 * Structured list row.
 *
 * @element cds-structured-list-row
 */
let CDSStructuredListRow = class CDSStructuredListRow extends HostListenerMixin(LitElement) {
    constructor() {
        super(...arguments);
        /**
         * The radio group manager associated with the radio button.
         */
        this._manager = null;
        /**
         * The interface for `RadioGroupManager` for structured list row.
         */
        this._radioButtonDelegate = new StructuredListRowRadioButtonDelegate(this);
        /**
         * Handles `click` event on this element.
         */
        this._handleClick = () => {
            const { _inputNode: inputNode } = this;
            if (inputNode) {
                this.selected = true;
                if (this._manager) {
                    this._manager.select(this._radioButtonDelegate);
                }
            }
        };
        /**
         * Handles `keydown` event on this element.
         */
        this._handleKeydown = (event) => {
            const { _inputNode: inputNode } = this;
            const manager = this._manager;
            if (inputNode && manager) {
                const navigationDirection = navigationDirectionForKey[event.key];
                if (navigationDirection) {
                    manager.select(manager.navigate(this._radioButtonDelegate, navigationDirection));
                }
                if (event.key === ' ' || event.key === 'Enter') {
                    manager.select(this._radioButtonDelegate);
                }
            }
        };
        /**
         * `true` if this structured list row should be selectable and selected.
         */
        this.selected = false;
        /**
         * The `name` attribute for the `<input>` for selection.
         * If present, this structured list row will be a selectable one.
         */
        this.selectionName = '';
        /**
         * The `value` attribute for the `<input>` for selection.
         */
        this.selectionValue = '';
        /**
         * The content to put into the `<title>` attribute of the selection icon.
         */
        this.selectionIconTitle = '';
    }
    connectedCallback() {
        var _a;
        if (!this.hasAttribute('role')) {
            this.setAttribute('role', 'row');
        }
        super.connectedCallback();
        if (!this._manager) {
            this._manager = RadioGroupManager.get(this.getRootNode({ composed: true }));
            const { selectionName } = this;
            if (selectionName) {
                (_a = this._manager) === null || _a === void 0 ? void 0 : _a.add(this._radioButtonDelegate);
            }
        }
    }
    disconnectedCallback() {
        if (this._manager) {
            this._manager.delete(this._radioButtonDelegate);
        }
        super.disconnectedCallback();
    }
    updated(changedProperties) {
        const { _manager: manager, selectionName } = this;
        if (changedProperties.has('selectionName')) {
            if (manager) {
                manager.delete(this._radioButtonDelegate, changedProperties.get('selectionName'));
                if (selectionName) {
                    manager.add(this._radioButtonDelegate);
                }
            }
            this.setAttribute('tabindex', !selectionName ||
                !manager ||
                !manager.shouldBeFocusable(this._radioButtonDelegate)
                ? '-1'
                : '0');
        }
    }
    render() {
        const { selected, selectionName, selectionValue, selectionIconTitle } = this;
        if (selectionName) {
            // "Selected" style with `.${prefix}--structured-list-td` does not work somehow - Need investigation
            return html `
        <slot></slot>
        <input
          id="input"
          type="radio"
          class="${prefix}--structured-list-input ${prefix}--visually-hidden"
          .checked=${selected}
          name=${selectionName}
          value=${ifDefined(selectionValue)} />
        <div
          class="${prefix}--structured-list-td ${prefix}--structured-list-cell">
          ${CheckmarkFilled16({
                class: `${prefix}--structured-list-svg`,
                title: selectionIconTitle,
            })}
        </div>
      `;
        }
        return html ` <slot></slot> `;
    }
};
CDSStructuredListRow.styles = styles;
__decorate([
    query('#input')
], CDSStructuredListRow.prototype, "_inputNode", void 0);
__decorate([
    HostListener('click')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSStructuredListRow.prototype, "_handleClick", void 0);
__decorate([
    HostListener('keydown')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSStructuredListRow.prototype, "_handleKeydown", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSStructuredListRow.prototype, "selected", void 0);
__decorate([
    property({ attribute: 'selection-name' })
], CDSStructuredListRow.prototype, "selectionName", void 0);
__decorate([
    property({ attribute: 'selection-value' })
], CDSStructuredListRow.prototype, "selectionValue", void 0);
__decorate([
    property({ attribute: 'selection-icon-title' })
], CDSStructuredListRow.prototype, "selectionIconTitle", void 0);
CDSStructuredListRow = __decorate([
    carbonElement(`${prefix}-structured-list-row`)
], CDSStructuredListRow);
var CDSStructuredListRow$1 = CDSStructuredListRow;

export { CDSStructuredListRow$1 as default };
//# sourceMappingURL=structured-list-row.js.map
