/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { classMap } from 'lit/directives/class-map.js';
import { LitElement, html } from 'lit';
import { property } from 'lit/decorators.js';
import { prefix } from '../../globals/settings.js';
import { forEach } from '../../globals/internal/collection-helpers.js';
import FocusMixin from '../../globals/mixins/focus.js';
import styles from './structured-list.scss.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2023
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Structured list wrapper.
 *
 * @element cds-structured-list
 */
let CDSStructuredList = class CDSStructuredList extends FocusMixin(LitElement) {
    constructor() {
        super(...arguments);
        /**
         * The `name` attribute for the `<input>` for selection.
         * If present, this structured list will be a selectable one.
         */
        this.selectionName = '';
        /**
         * Specify if structured list is condensed, default is false
         */
        this.condensed = false;
        /**
         * Specify if structured list is flush, default is false
         */
        this.flush = false;
    }
    connectedCallback() {
        if (!this.hasAttribute('role')) {
            this.setAttribute('role', 'table');
        }
        super.connectedCallback();
    }
    shouldUpdate(changedProperties) {
        if (changedProperties.has('selectionName')) {
            // Propagate `selectionName` attribute to descendants until `:host-context()` gets supported in all major browsers
            forEach(this.querySelectorAll(this.constructor.selectorRowsWithHeader), (elem) => {
                elem.selectionName = this.selectionName;
            });
        }
        return true;
    }
    updated(changedProperties) {
        const attributes = ['condensed', 'flush'];
        attributes.forEach((attr) => {
            if (changedProperties.has(attr)) {
                // Propagate watched attribute to descendants until `:host-context()` gets supported in all major browsers
                forEach(this.querySelectorAll(this.constructor
                    .selectorRowsWithHeader), (elem) => {
                    this[`${attr}`]
                        ? elem.setAttribute(attr, '')
                        : elem.removeAttribute(attr);
                });
            }
        });
    }
    render() {
        const { condensed, flush, selectionName } = this;
        const classes = classMap({
            [`${prefix}--structured-list`]: true,
            [`${prefix}--structured-list--selection`]: Boolean(selectionName),
            [`${prefix}--structured-list--condensed`]: condensed,
            [`${prefix}--structured-list--flush`]: flush,
        });
        return html `
      <section id="section" class=${classes}><slot></slot></section>
    `;
    }
};
/**
 * The CSS selector to find the rows, including header rows.
 */
CDSStructuredList.selectorRowsWithHeader = `${prefix}-structured-list-row,${prefix}-structured-list-header-row`;
CDSStructuredList.shadowRootOptions = Object.assign(Object.assign({}, LitElement.shadowRootOptions), { delegatesFocus: true });
CDSStructuredList.styles = styles;
__decorate([
    property({ attribute: 'selection-name' })
], CDSStructuredList.prototype, "selectionName", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSStructuredList.prototype, "condensed", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSStructuredList.prototype, "flush", void 0);
CDSStructuredList = __decorate([
    carbonElement(`${prefix}-structured-list`)
], CDSStructuredList);
var CDSStructuredList$1 = CDSStructuredList;

export { CDSStructuredList$1 as default };
//# sourceMappingURL=structured-list.js.map
