/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { html } from 'lit';
import { query, state, property } from 'lit/decorators.js';
import { classMap } from 'lit/directives/class-map.js';
import { prefix } from '../../globals/settings.js';
import HostListenerMixin from '../../globals/mixins/host-listener.js';
import HostListener from '../../globals/decorators/host-listener.js';
import { forEach } from '../../globals/internal/collection-helpers.js';
import ChevronRight16 from '../../icons/chevron--right/16.js';
import ChevronLeft16 from '../../icons/chevron--left/16.js';
import CDSContentSwitcher from '../content-switcher/content-switcher.js';
import { TABS_TYPE, TABS_KEYBOARD_ACTION } from './defs.js';
export { TABS_KEYBOARD_ACTION, TABS_TYPE } from './defs.js';
import styles from './tabs.scss.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';
import { NAVIGATION_DIRECTION } from '../content-switcher/defs.js';
export { NAVIGATION_DIRECTION } from '../content-switcher/defs.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Tabs.
 *
 * @element cds-tabs
 * @fires cds-tabs-beingselected
 *   The custom event fired before a tab is selected upon a user gesture.
 *   Cancellation of this event stops changing the user-initiated selection.
 * @fires cds-tabs-selected - The custom event fired after a a tab is selected upon a user gesture.
 */
let CDSTabs = class CDSTabs extends HostListenerMixin(CDSContentSwitcher) {
    constructor() {
        super(...arguments);
        /**
         * The currently selected index
         */
        // @ts-ignore: TS thinks this method is not referred to
        this._currentIndex = 0;
        /**
         * Total number of tabs in the component
         */
        this._totalTabs = 0;
        /**
         * `true` if the tablist is scrollable
         */
        // @ts-ignore: TS thinks this method is not referred to
        this._isScrollable = false;
        /**
         * The DOM element for the tablist.
         */
        this.tablist = null;
        /**
         * The width of the overflow scroll buttons.
         */
        this.BUTTON_WIDTH = 44;
        /**
         * The current scroll position.
         */
        this._currentScrollPosition = 0;
        /**
         * An assistive text for screen reader to announce, telling the open state.
         */
        this.selectingItemsAssistiveText = 'Selecting items. Use up and down arrow keys to navigate.';
        /**
         * An assistive text for screen reader to announce, telling that an item is selected.
         */
        this.selectedItemAssistiveText = 'Selected an item.';
        /**
         * The content of the trigger button for narrow mode.
         */
        this.triggerContent = '';
        /**
         * Tabs type.
         */
        this.type = TABS_TYPE.REGULAR;
        /**
         * `true` if left-hand scroll intersection sentinel intersects with the host element.
         * In this condition, the left-hand paginator button should be hidden.
         */
        this._isIntersectionLeftTrackerInContent = true;
        /**
         * `true` if right-hand scroll intersection sentinel intersects with the host element.
         * In this condition, the right-hand paginator button should be hidden.
         */
        this._isIntersectionRightTrackerInContent = true;
        /**
         * The observer for the intersection of left-side content edge.
         */
        this._observerIntersection = null;
        /**
         * The intersection observer callback for the scrolling container.
         *
         * @param records The intersection observer records.
         */
        this._observeIntersectionContainer = (records) => {
            const { _intersectionLeftSentinelNode: intersectionLeftSentinelNode, _intersectionRightSentinelNode: intersectionRightSentinelNode, } = this;
            records.forEach(({ isIntersecting, target }) => {
                if (target === intersectionLeftSentinelNode) {
                    this._isIntersectionLeftTrackerInContent = isIntersecting;
                }
                if (target === intersectionRightSentinelNode) {
                    this._isIntersectionRightTrackerInContent = isIntersecting;
                }
            });
        };
    }
    /**
     * Navigates through tabs.
     *
     * @param direction `-1` to navigate backward, `1` to navigate forward.
     * @param [options] The options.
     * @param [options.immediate]
     *   Defaults to `true`
     *   `true` to make it "immediate selection change" mode, which does:
     *
     *   Starts with the selected item
     *   Going prev/next item immediately changes the selection
     */
    _navigate(direction, { immediate = true } = {}) {
        const { selectorItem, selectorItemHighlighted, selectorItemSelected } = this
            .constructor;
        const nextItem = this._getNextItem(this.querySelector(immediate ? selectorItemSelected : selectorItemHighlighted), direction);
        if (!nextItem) {
            return;
        }
        if (immediate) {
            this._handleUserInitiatedSelectItem(nextItem);
        }
        else {
            forEach(this.querySelectorAll(selectorItem), (item) => {
                item[immediate ? 'selected' : 'highlighted'] =
                    nextItem === item;
            });
        }
        // Using `{ block: 'nearest' }` to prevent scrolling unless scrolling is absolutely necessary.
        // `scrollIntoViewOptions` seems to work in latest Safari despite of MDN/caniuse table.
        // IE falls back to the old behavior.
        nextItem.scrollIntoView({ block: 'nearest', inline: 'nearest' });
        const nextItemText = nextItem.textContent;
        if (nextItemText) {
            this._assistiveStatusText = nextItemText;
        }
        this._currentIndex += direction;
        this.requestUpdate();
    }
    _handleClick(event) {
        super._handleClick(event);
    }
    _handleKeydown(event) {
        const { key } = event;
        const action = this.constructor.getAction(key);
        const enabledTabs = this.querySelectorAll(`${prefix}-tab:not([disabled])`);
        switch (action) {
            case TABS_KEYBOARD_ACTION.HOME:
                {
                    const [firstEnabledTab] = enabledTabs;
                    firstEnabledTab.scrollIntoView({
                        block: 'nearest',
                        inline: 'nearest',
                    });
                    this._handleUserInitiatedSelectItem(firstEnabledTab);
                    this.requestUpdate();
                }
                break;
            case TABS_KEYBOARD_ACTION.END:
                {
                    const lastEnabledTab = enabledTabs[enabledTabs.length - 1];
                    lastEnabledTab.scrollIntoView({
                        block: 'nearest',
                        inline: 'nearest',
                    });
                    this._handleUserInitiatedSelectItem(lastEnabledTab);
                    this.requestUpdate();
                }
                break;
            case TABS_KEYBOARD_ACTION.NAVIGATING:
                {
                    const direction = NAVIGATION_DIRECTION[key];
                    if (direction) {
                        this._navigate(direction);
                    }
                }
                break;
        }
    }
    /**
     * Handles click on overflow scroll buttons.
     *
     * @param _ Event object
     * @param [options] The options.
     * @param [options.direction] `-1` to scroll forward, `1` to scroll backward.
     */
    _handleScrollButtonClick(_, { direction }) {
        if (!this.tablist) {
            return;
        }
        const { scrollLeft, clientWidth, scrollWidth } = this._contentContainerNode;
        switch (direction) {
            case -1:
                this._contentContainerNode.scrollLeft = Math.max(scrollLeft - (scrollWidth / this._totalTabs) * 1.5, 0);
                break;
            case 1:
                this._contentContainerNode.scrollLeft =
                    Math.min(scrollLeft + (scrollWidth / this._totalTabs) * 1.5, scrollWidth - clientWidth) + 1;
                break;
        }
    }
    _handleSlotchange() {
        const { selectorItemSelected } = this.constructor;
        const selectedItem = this.querySelector(selectorItemSelected);
        const nextItem = this._getNextItem(selectedItem, 1);
        // Specifies child `<cds-tab>` to hide its divider instead of using CSS,
        // until `:host-context()` gets supported in all major browsers
        nextItem.hideDivider = true;
    }
    _selectionDidChange(itemToSelect) {
        super._selectionDidChange(itemToSelect);
        this._assistiveStatusText = this.selectedItemAssistiveText;
    }
    /**
     * Cleans-up and creates the intersection observer for the scrolling container.
     *
     * @param [options] The options.
     * @param [options.create] `true` to create the new intersection observer.
     */
    _cleanAndCreateIntersectionObserverContainer({ create, } = {}) {
        const { _intersectionLeftSentinelNode: intersectionLeftSentinelNode, _intersectionRightSentinelNode: intersectionRightSentinelNode, } = this;
        if (this._observerIntersection) {
            this._observerIntersection.disconnect();
            this._observerIntersection = null;
        }
        if (create) {
            this._observerIntersection = new IntersectionObserver(this._observeIntersectionContainer, {
                root: this,
                threshold: 0,
            });
            if (intersectionLeftSentinelNode) {
                this._observerIntersection.observe(intersectionLeftSentinelNode);
            }
            if (intersectionRightSentinelNode) {
                this._observerIntersection.observe(intersectionRightSentinelNode);
            }
        }
    }
    disconnectedCallback() {
        this._cleanAndCreateIntersectionObserverContainer();
        super.disconnectedCallback();
    }
    shouldUpdate(changedProperties) {
        super.shouldUpdate(changedProperties);
        if (this.tablist) {
            const { clientWidth, scrollWidth } = this.tablist;
            this._isScrollable = scrollWidth > clientWidth;
        }
        const { selectorItem } = this.constructor;
        if (changedProperties.has('type')) {
            forEach(this.querySelectorAll(selectorItem), (elem) => {
                this._totalTabs++;
                elem.type = this.type;
            });
        }
        return true;
    }
    firstUpdated() {
        const { selectorTablist } = this.constructor;
        const tablist = this.shadowRoot.querySelector(selectorTablist);
        this.tablist = tablist;
        this._cleanAndCreateIntersectionObserverContainer({ create: true });
    }
    updated(changedProperties) {
        var _a, _b;
        if (changedProperties.has('value')) {
            const tab = this.querySelector(`${prefix}-tab[value="${this.value}"]`);
            if (tab) {
                const { width: tabWidth } = (_a = tab === null || tab === void 0 ? void 0 : tab.getBoundingClientRect()) !== null && _a !== void 0 ? _a : {};
                const start = (_b = tab.offsetLeft) !== null && _b !== void 0 ? _b : 0;
                const end = tab.offsetLeft + tabWidth;
                // The start and end of the visible area of the tablist
                const visibleStart = this.tablist.scrollLeft + this.BUTTON_WIDTH;
                const visibleEnd = this.tablist.scrollLeft +
                    this.tablist.clientWidth -
                    this.BUTTON_WIDTH;
                // The beginning of the tab is clipped and not visible
                if (start < visibleStart) {
                    this.tablist.scrollLeft = start - this.BUTTON_WIDTH;
                }
                // The end of the tab is clipped and not visible
                if (end > visibleEnd) {
                    this.tablist.scrollLeft =
                        end + this.BUTTON_WIDTH - this.tablist.clientWidth;
                }
            }
        }
        if (changedProperties.has('_currentScrollPosition')) {
            if (this._contentNode) {
                this._contentNode.style.insetInlineStart = `-${this._currentScrollPosition}px`;
            }
        }
    }
    /**
     * Render the previous button if tablist is wider than container.
     */
    renderPreviousButton() {
        const { _isIntersectionLeftTrackerInContent: isIntersectionLeftTrackerInContent, } = this;
        const previousButtonClasses = classMap({
            [`${prefix}--tab--overflow-nav-button`]: true,
            [`${prefix}--tabs__nav-caret-left`]: true,
            [`${prefix}--tab--overflow-nav-button--previous`]: true,
            [`${prefix}--tab--overflow-nav-button--hidden`]: isIntersectionLeftTrackerInContent,
        });
        return html `
      <button
        part="prev-button"
        tabindex="-1"
        aria-hidden="true"
        class="${previousButtonClasses}"
        @click=${(_) => this._handleScrollButtonClick(_, {
            direction: NAVIGATION_DIRECTION.Left,
        })}>
        ${ChevronLeft16()}
      </button>
    `;
    }
    /**
     * Render the next button if tablist is wider than container.
     */
    renderNextButton() {
        const { _isIntersectionRightTrackerInContent: isIntersectionRightTrackerInContent, } = this;
        const nextButtonClasses = classMap({
            [`${prefix}--tab--overflow-nav-button`]: true,
            [`${prefix}--tabs__nav-caret-right`]: true,
            [`${prefix}--tab--overflow-nav-button--next`]: true,
            [`${prefix}--tab--overflow-nav-button--hidden`]: isIntersectionRightTrackerInContent,
        });
        return html `
      <button
        part="next-button"
        tabindex="-1"
        aria-hidden="true"
        class="${nextButtonClasses}"
        @click=${(_) => this._handleScrollButtonClick(_, {
            direction: NAVIGATION_DIRECTION.Right,
        })}>
        ${ChevronRight16()}
      </button>
    `;
    }
    render() {
        const { _assistiveStatusText: assistiveStatusText, _handleSlotchange: handleSlotchange, } = this;
        return html `
      ${this.renderPreviousButton()}
      <div class="${prefix}--tabs-nav-content-container">
        <div class="${prefix}--tabs-nav-content">
          <div class="${prefix}--tabs-nav">
            <div id="tablist" role="tablist" class="${prefix}--tab--list">
              <div class="${prefix}--sub-content-left"></div>
              <slot @slotchange=${handleSlotchange}></slot>
              <div class="${prefix}--sub-content-right"></div>
            </div>
          </div>
        </div>
      </div>
      ${this.renderNextButton()}
      <div
        class="${prefix}--assistive-text"
        role="status"
        aria-live="assertive"
        aria-relevant="additions text">
        ${assistiveStatusText}
      </div>
    `;
    }
    /**
     * A selector that will return tabs.
     */
    static get selectorItem() {
        return `${prefix}-tab`;
    }
    /**
     * A selector that will return enabled tabs.
     */
    static get selectorItemEnabled() {
        return `${prefix}-tab:not([disabled])`;
    }
    /**
     * A selector that will return highlighted tabs.
     */
    static get selectorItemHighlighted() {
        return `${prefix}-tab[highlighted]`;
    }
    /**
     * A selector that will return selected tabs.
     */
    static get selectorItemSelected() {
        return `${prefix}-tab[selected]`;
    }
    /**
     * A selector that returns the tablist
     */
    static get selectorTablist() {
        return `.${prefix}--tab--list`;
    }
    /**
     * The name of the custom event fired before a tab is selected upon a user gesture.
     * Cancellation of this event stops changing the user-initiated selection.
     */
    static get eventBeforeSelect() {
        return `${prefix}-tabs-beingselected`;
    }
    /**
     * The name of the custom event fired after a a tab is selected upon a user gesture.
     */
    static get eventSelect() {
        return `${prefix}-tabs-selected`;
    }
    /**
     * @param key The key symbol.
     * @returns A action for dropdown for the given key symbol.
     */
    static getAction(key) {
        if (key === 'Home') {
            return TABS_KEYBOARD_ACTION.HOME;
        }
        if (key === 'End') {
            return TABS_KEYBOARD_ACTION.END;
        }
        if (key in NAVIGATION_DIRECTION) {
            return TABS_KEYBOARD_ACTION.NAVIGATING;
        }
        return TABS_KEYBOARD_ACTION.NONE;
    }
};
/**
 * Symbols of keys that triggers opening/closing menu and selecting/deselecting menu item.
 */
CDSTabs.TRIGGER_KEYS = new Set([' ', 'Enter']);
CDSTabs.styles = styles;
__decorate([
    HostListener('click')
], CDSTabs.prototype, "_handleClick", null);
__decorate([
    HostListener('keydown')
], CDSTabs.prototype, "_handleKeydown", null);
__decorate([
    query(`.${prefix}--tabs-nav-content-container`)
], CDSTabs.prototype, "_contentContainerNode", void 0);
__decorate([
    query(`.${prefix}--tabs-nav-content`)
], CDSTabs.prototype, "_contentNode", void 0);
__decorate([
    state()
], CDSTabs.prototype, "_currentScrollPosition", void 0);
__decorate([
    query(`.${prefix}--sub-content-left`)
], CDSTabs.prototype, "_intersectionLeftSentinelNode", void 0);
__decorate([
    query(`.${prefix}--sub-content-right`)
], CDSTabs.prototype, "_intersectionRightSentinelNode", void 0);
__decorate([
    property({ attribute: 'selecting-items-assistive-text' })
], CDSTabs.prototype, "selectingItemsAssistiveText", void 0);
__decorate([
    property({ attribute: 'selected-item-assistive-text' })
], CDSTabs.prototype, "selectedItemAssistiveText", void 0);
__decorate([
    property({ attribute: 'trigger-content' })
], CDSTabs.prototype, "triggerContent", void 0);
__decorate([
    property({ reflect: true })
], CDSTabs.prototype, "type", void 0);
__decorate([
    state()
], CDSTabs.prototype, "_isIntersectionLeftTrackerInContent", void 0);
__decorate([
    state()
], CDSTabs.prototype, "_isIntersectionRightTrackerInContent", void 0);
CDSTabs = __decorate([
    carbonElement(`${prefix}-tabs`)
], CDSTabs);
var CDSTabs$1 = CDSTabs;

export { CDSTabs$1 as default };
//# sourceMappingURL=tabs.js.map
