/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { LitElement, html } from 'lit';
import { query, property } from 'lit/decorators.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';
import { classMap } from 'lit/directives/class-map.js';
import { prefix } from '../../globals/settings.js';
import View16 from '../../icons/view/16.js';
import ViewOff16 from '../../icons/view--off/16.js';
import WarningFilled16 from '../../icons/warning--filled/16.js';
import WarningAltFilled16 from '../../icons/warning--alt--filled/16.js';
import ifNonEmpty from '../../globals/directives/if-non-empty.js';
import FormMixin from '../../globals/mixins/form.js';
import ValidityMixin from '../../globals/mixins/validity.js';
import { INPUT_SIZE, INPUT_TOOLTIP_ALIGNMENT, INPUT_TOOLTIP_DIRECTION, INPUT_TYPE } from './defs.js';
export { INPUT_SIZE, INPUT_TOOLTIP_ALIGNMENT, INPUT_TOOLTIP_DIRECTION, INPUT_TYPE } from './defs.js';
import styles from './text-input.scss.js';
export { FORM_ELEMENT_COLOR_SCHEME as INPUT_COLOR_SCHEME } from '../../globals/shared-enums.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Text Input element. Supports all the usual attributes for textual input types
 *
 * @element cds-text-input
 * @slot helper-text - The helper text.
 * @slot label-text - The label text.
 * @slot validity-message - The validity message. If present and non-empty, this input shows the UI of its invalid state.
 */
let CDSTextInput = class CDSTextInput extends ValidityMixin(FormMixin(LitElement)) {
    constructor() {
        super(...arguments);
        /**
         * `true` if there is an AI Label.
         */
        this._hasAILabel = false;
        /**
         * The internal value.
         */
        this._value = '';
        /**
         * May be any of the standard HTML autocomplete options
         */
        this.autocomplete = '';
        /**
         * Sets the input to be focussed automatically on page load. Defaults to false
         */
        this.autofocus = false;
        /**
         * Controls the disabled state of the input
         */
        this.disabled = false;
        /**
         * Specify whether to display the character counter
         */
        this.enableCounter = false;
        /**
         * The helper text.
         */
        this.helperText = '';
        /**
         * Specify if the currently value is invalid.
         */
        this.invalid = false;
        /**
         * Message which is displayed if the value is invalid.
         */
        this.invalidText = '';
        /**
         * Specify whether the control is currently in warning state
         */
        this.warn = false;
        /**
         * Provide the text that is displayed when the control is in warning state
         */
        this.warnText = '';
        /**
         * Specify whether you want the underlying label to be visually hidden
         */
        this.hideLabel = false;
        /**
         * Generic label that will be used as the textual representation of what this field is for
         */
        this.label = '';
        /**
         * Name for the input in the `FormData`
         */
        this.name = '';
        /**
         * Pattern to validate the input against for HTML validity checking
         */
        this.pattern = '';
        /**
         * Value to display when the input has an empty `value`
         */
        this.placeholder = '';
        /**
         * Specify if the component should be read-only
         */
        this.readonly = false;
        /**
         * Boolean property to set the required status
         */
        this.required = false;
        /**
         * The special validity message for `required`.
         */
        this.requiredValidityMessage = 'Please fill out this field.';
        /**
         * "Hide password" tooltip text on password visibility toggle
         */
        this.hidePasswordLabel = 'Hide password';
        /**
         * "Show password" tooltip text on password visibility toggle
         */
        this.showPasswordLabel = 'Show password';
        /**
         * Boolean property to render password visibility toggle
         */
        this.showPasswordVisibilityToggle = false;
        /**
         * The input box size.
         */
        this.size = INPUT_SIZE.MEDIUM;
        /**
         * true to use the inline version.
         */
        this.inline = false;
        /**
         * Specify the alignment of the tooltip to the icon-only button.
         * Can be one of: start, center, or end.
         */
        this.tooltipAlignment = INPUT_TOOLTIP_ALIGNMENT.CENTER;
        /**
         * Specify the direction of the tooltip for icon-only buttons.
         * Can be either top, right, bottom, or left.
         */
        this.tooltipDirection = INPUT_TOOLTIP_DIRECTION.BOTTOM;
        /**
         * The type of the input. Can be one of the types listed in the INPUT_TYPE enum
         */
        this.type = INPUT_TYPE.TEXT;
        /**
         * The validity message. If present and non-empty, this input shows the UI of its invalid state.
         */
        this.validityMessage = '';
    }
    /**
     * Handles `slotchange` event.
     */
    _handleSlotChange({ target }) {
        const hasContent = target
            .assignedNodes()
            .filter((elem) => elem.matches !== undefined
            ? elem.matches(this.constructor.aiLabelItem) ||
                // remove reference to slug in v12
                elem.matches(this.constructor.slugItem)
            : false);
        this._hasAILabel = Boolean(hasContent);
        hasContent[0].setAttribute('size', 'mini');
        this.requestUpdate();
    }
    /**
     * Handles `oninput` event on the `<input>`.
     *
     * @param event The event.
     * @param event.target The event target.
     */
    _handleInput({ target }) {
        this.value = target.value;
    }
    _handleFormdata(event) {
        const { formData } = event; // TODO: Wait for `FormDataEvent` being available in `lib.dom.d.ts`
        const { disabled, name, value } = this;
        if (!disabled) {
            formData.append(name, value);
        }
    }
    /**
     * The value of the input.
     */
    get value() {
        // FIXME: Figure out how to deal with TS2611
        // once we have the input we can directly query for the value
        if (this._input) {
            return this._input.value;
        }
        // but before then _value will work fine
        return this._value;
    }
    set value(value) {
        const oldValue = this._value;
        this._value = value;
        // make sure that lit-element updates the right properties
        this.requestUpdate('value', oldValue);
        // we set the value directly on the input (when available)
        // so that programatic manipulation updates the UI correctly
        if (this._input) {
            this._input.value = value;
        }
    }
    /**
     * Handles password visibility toggle button click
     */
    handleTogglePasswordVisibility() {
        this.type =
            this.type === INPUT_TYPE.PASSWORD ? INPUT_TYPE.TEXT : INPUT_TYPE.PASSWORD;
    }
    render() {
        const { disabled, enableCounter, helperText, hideLabel, inline, invalid, invalidText, label, maxCount, readonly, required, size, type, warn, warnText, value, _handleInput: handleInput, _hasAILabel: hasAILabel, _handleSlotChange: handleSlotChange, } = this;
        const invalidIcon = WarningFilled16({
            class: `${prefix}--text-input__invalid-icon`,
        });
        const warnIcon = WarningAltFilled16({
            class: `${prefix}--text-input__invalid-icon ${prefix}--text-input__invalid-icon--warning`,
        });
        const normalizedProps = {
            disabled: !readonly && disabled,
            invalid: !readonly && invalid,
            warn: !readonly && !invalid && warn,
            'slot-name': '',
            'slot-text': '',
            icon: null,
        };
        if (normalizedProps.invalid) {
            normalizedProps.icon = invalidIcon;
            normalizedProps['slot-name'] = 'invalid-text';
            normalizedProps['slot-text'] = invalidText;
        }
        else if (normalizedProps.warn) {
            normalizedProps.icon = warnIcon;
            normalizedProps['slot-name'] = 'warn-text';
            normalizedProps['slot-text'] = warnText;
        }
        const counterClasses = classMap({
            [`${prefix}--label`]: true,
            [`${prefix}--text-input__label-counter`]: true,
            [`${prefix}--label--disabled`]: disabled,
        });
        const inputWrapperClasses = classMap({
            [`${prefix}--form-item`]: true,
            [`${prefix}--text-input-wrapper`]: true,
            [`${prefix}--text-input-wrapper--inline`]: inline,
            [`${prefix}--text-input-wrapper--readonly`]: readonly,
            [`${prefix}--text-input-wrapper--inline--invalid`]: inline && normalizedProps.invalid,
        });
        const inputClasses = classMap({
            [`${prefix}--text-input`]: true,
            [`${prefix}--text-input--invalid`]: normalizedProps.invalid,
            [`${prefix}--text-input--warning`]: normalizedProps.warn,
            [`${prefix}--text-input--${size}`]: size,
            [`${prefix}--layout--size-${size}`]: size,
            [`${prefix}--password-input`]: type === INPUT_TYPE.PASSWORD,
        });
        const fieldOuterWrapperClasses = classMap({
            [`${prefix}--text-input__field-outer-wrapper`]: true,
            [`${prefix}--text-input__field-outer-wrapper--inline`]: inline,
        });
        const fieldWrapperClasses = classMap({
            [`${prefix}--text-input__field-wrapper`]: true,
            [`${prefix}--text-input__field-wrapper--warning`]: normalizedProps.warn,
            [`${prefix}--text-input__field-wrapper--slug`]: hasAILabel,
        });
        const labelClasses = classMap({
            [`${prefix}--label`]: true,
            [`${prefix}--visually-hidden`]: hideLabel,
            [`${prefix}--label--disabled`]: normalizedProps.disabled,
        });
        const helperTextClasses = classMap({
            [`${prefix}--form__helper-text`]: true,
            [`${prefix}--form__helper-text--disabled`]: normalizedProps.disabled,
        });
        const passwordIsVisible = type !== INPUT_TYPE.PASSWORD;
        const passwordVisibilityIcon = passwordIsVisible
            ? ViewOff16({ class: `${prefix}--icon-visibility-off` })
            : View16({ class: `${prefix}--icon-visibility-on` });
        const passwordVisibilityToggleClasses = classMap({
            [`${prefix}--text-input--password__visibility__toggle`]: true,
            [`${prefix}--btn`]: true,
            [`${prefix}--btn--icon-only`]: true,
            [`${prefix}--tooltip__trigger`]: true,
            [`${prefix}--tooltip--a11y`]: true,
            [`${prefix}--btn--disabled`]: normalizedProps.disabled,
            [`${prefix}--tooltip--${this.tooltipDirection}`]: this.tooltipDirection,
            [`${prefix}--tooltip--align-${this.tooltipAlignment}`]: this.tooltipAlignment,
        });
        const passwordButtonLabel = html `
      <span class="${prefix}--assistive-text">
        ${passwordIsVisible ? this.hidePasswordLabel : this.showPasswordLabel}
      </span>
    `;
        const passwordVisibilityButton = () => html `
      <button
        type="button"
        class="${passwordVisibilityToggleClasses}"
        ?disabled="${normalizedProps.disabled}"
        @click="${this.handleTogglePasswordVisibility}">
        ${!normalizedProps.disabled ? passwordButtonLabel : null}
        ${passwordVisibilityIcon}
      </button>
    `;
        const textCount = value === null || value === void 0 ? void 0 : value.length;
        const counter = enableCounter && maxCount
            ? html ` <label class="${counterClasses}">
            <slot name="label-text">${textCount}/${maxCount}</slot>
          </label>`
            : null;
        const labelText = label && !hideLabel
            ? html `<label class="${labelClasses}"> ${label} </label>`
            : null;
        const labelWrapper = html `<div class="${prefix}--text-input__label-wrapper">
      ${labelText} ${counter}
    </div>`;
        const helper = helperText
            ? html `<div
          class="${helperTextClasses}"
          id="helper-text"
          ?hidden="${normalizedProps.invalid || normalizedProps.warn}">
          <slot name="helper-text"> ${helperText} </slot>
        </div>`
            : null;
        return html `
      <div class="${inputWrapperClasses}">
        ${!inline
            ? labelWrapper
            : html `<div class="${prefix}--text-input__label-helper-wrapper">
              ${labelWrapper} ${helper}
            </div>`}
        <div class="${fieldOuterWrapperClasses}">
          <div class="${fieldWrapperClasses}" ?data-invalid="${invalid}">
            ${normalizedProps.icon}
            <input
              autocomplete="${this.autocomplete}"
              ?autofocus="${this.autofocus}"
              class="${inputClasses}"
              ?data-invalid="${invalid}"
              ?disabled="${disabled}"
              aria-describedby="helper-text"
              id="input"
              name="${ifNonEmpty(this.name)}"
              pattern="${ifNonEmpty(this.pattern)}"
              placeholder="${ifNonEmpty(this.placeholder)}"
              ?readonly="${readonly}"
              ?required="${required}"
              type="${ifNonEmpty(type)}"
              .value="${this._value}"
              maxlength="${ifNonEmpty(maxCount)}"
              @input="${handleInput}" />
            <slot name="ai-label" @slotchange="${handleSlotChange}"></slot>
            <slot name="slug" @slotchange="${handleSlotChange}"></slot>
            ${this.showPasswordVisibilityToggle &&
            (type === INPUT_TYPE.PASSWORD || type === INPUT_TYPE.TEXT)
            ? passwordVisibilityButton()
            : null}
          </div>
          ${!inline ? helper : null}
          <div
            class="${prefix}--form-requirement"
            ?hidden="${!normalizedProps.invalid && !normalizedProps.warn}">
            <slot name="${normalizedProps['slot-name']}">
              ${normalizedProps['slot-text']}
            </slot>
          </div>
        </div>
      </div>
    `;
    }
    updated() {
        var _a, _b, _c, _d, _e;
        const label = (_a = this.shadowRoot) === null || _a === void 0 ? void 0 : _a.querySelector("slot[name='ai-label']");
        if (label) {
            label === null || label === void 0 ? void 0 : label.classList.toggle(`${prefix}--slug--revert`, (_b = this.querySelector(`${prefix}-ai-label`)) === null || _b === void 0 ? void 0 : _b.hasAttribute('revert-active'));
        }
        else {
            (_d = (_c = this.shadowRoot) === null || _c === void 0 ? void 0 : _c.querySelector("slot[name='slug']")) === null || _d === void 0 ? void 0 : _d.classList.toggle(`${prefix}--slug--revert`, (_e = this.querySelector(`${prefix}-slug`)) === null || _e === void 0 ? void 0 : _e.hasAttribute('revert-active'));
        }
    }
    /**
     * A selector that will return the slug item.
     *
     * remove in v12
     */
    static get slugItem() {
        return `${prefix}-slug`;
    }
    /**
     * A selector that will return the AI Label item.
     */
    static get aiLabelItem() {
        return `${prefix}-ai-label`;
    }
};
CDSTextInput.shadowRootOptions = Object.assign(Object.assign({}, LitElement.shadowRootOptions), { delegatesFocus: true });
CDSTextInput.styles = styles; // `styles` here is a `CSSResult` generated by custom Vite loader
__decorate([
    query('input')
], CDSTextInput.prototype, "_input", void 0);
__decorate([
    property()
], CDSTextInput.prototype, "autocomplete", void 0);
__decorate([
    property({ type: Boolean })
], CDSTextInput.prototype, "autofocus", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSTextInput.prototype, "disabled", void 0);
__decorate([
    property({ type: Boolean, attribute: 'enable-counter', reflect: true })
], CDSTextInput.prototype, "enableCounter", void 0);
__decorate([
    property({ attribute: 'helper-text' })
], CDSTextInput.prototype, "helperText", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSTextInput.prototype, "invalid", void 0);
__decorate([
    property({ attribute: 'invalid-text' })
], CDSTextInput.prototype, "invalidText", void 0);
__decorate([
    property({ type: Number, attribute: 'max-count', reflect: true })
], CDSTextInput.prototype, "maxCount", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSTextInput.prototype, "warn", void 0);
__decorate([
    property({ attribute: 'warn-text' })
], CDSTextInput.prototype, "warnText", void 0);
__decorate([
    property({ attribute: 'hide-label', type: Boolean, reflect: true })
], CDSTextInput.prototype, "hideLabel", void 0);
__decorate([
    property({ attribute: 'label' })
], CDSTextInput.prototype, "label", void 0);
__decorate([
    property()
], CDSTextInput.prototype, "name", void 0);
__decorate([
    property()
], CDSTextInput.prototype, "pattern", void 0);
__decorate([
    property({ reflect: true })
], CDSTextInput.prototype, "placeholder", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSTextInput.prototype, "readonly", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSTextInput.prototype, "required", void 0);
__decorate([
    property({ attribute: 'required-validity-message' })
], CDSTextInput.prototype, "requiredValidityMessage", void 0);
__decorate([
    property()
], CDSTextInput.prototype, "hidePasswordLabel", void 0);
__decorate([
    property()
], CDSTextInput.prototype, "showPasswordLabel", void 0);
__decorate([
    property({
        type: Boolean,
        attribute: 'show-password-visibility-toggle',
        reflect: true,
    })
], CDSTextInput.prototype, "showPasswordVisibilityToggle", void 0);
__decorate([
    property({ reflect: true })
], CDSTextInput.prototype, "size", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSTextInput.prototype, "inline", void 0);
__decorate([
    property()
], CDSTextInput.prototype, "tooltipAlignment", void 0);
__decorate([
    property()
], CDSTextInput.prototype, "tooltipDirection", void 0);
__decorate([
    property({ reflect: true })
], CDSTextInput.prototype, "type", void 0);
__decorate([
    property({ attribute: 'validity-message' })
], CDSTextInput.prototype, "validityMessage", void 0);
__decorate([
    property({ reflect: true })
], CDSTextInput.prototype, "value", null);
CDSTextInput = __decorate([
    carbonElement(`${prefix}-text-input`)
], CDSTextInput);
var CDSTextInput$1 = CDSTextInput;

export { CDSTextInput$1 as default };
//# sourceMappingURL=text-input.js.map
