/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { LitElement, html } from 'lit';
import { property, query } from 'lit/decorators.js';
import { classMap } from 'lit/directives/class-map.js';
import WarningFilled16 from '../../icons/warning--filled/16.js';
import WarningAltFilled16 from '../../icons/warning--alt--filled/16.js';
import { prefix } from '../../globals/settings.js';
import ifNonEmpty from '../../globals/directives/if-non-empty.js';
import { ifDefined } from 'lit/directives/if-defined.js';
import CDSTextInput from '../text-input/text-input.js';
import styles from './textarea.scss.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Text area.
 *
 * @element cds-textarea
 * @slot helper-text - The helper text.
 * @slot label-text - The label text.
 * @slot validity-message - The validity message. If present and non-empty, this input shows the UI of its invalid state.
 */
let CDSTextarea = class CDSTextarea extends CDSTextInput {
    constructor() {
        super(...arguments);
        /**
         * ID to link the `label` and `textarea`
         */
        this.id = '';
        /**
         * Pattern to validate the textarea against for HTML validity checking
         */
        this.pattern = '';
        /**
         * Boolean property to set the required status
         */
        this.required = false;
        /**
         * The number of rows for the textarea to show by default
         */
        this.rows = 4;
    }
    /**
     * Handles `oninput` event on the `<input>`.
     *
     * @param event The event.
     * @param event.target The event target.
     */
    _handleInput({ target }) {
        this.value = target.value;
    }
    render() {
        var _a;
        const { enableCounter, maxCount } = this;
        const textCount = (_a = this.value) === null || _a === void 0 ? void 0 : _a.length;
        const invalidIcon = WarningFilled16({
            class: `${prefix}--text-area__invalid-icon`,
        });
        const warnIcon = WarningAltFilled16({
            class: `${prefix}--text-area__invalid-icon ${prefix}--text-area__invalid-icon--warning`,
        });
        const textareaClasses = classMap({
            [`${prefix}--text-area`]: true,
            [`${prefix}--text-area--warn`]: this.warn,
            [`${prefix}--text-area--invalid`]: this.invalid,
        });
        const textareaWrapperClasses = classMap({
            [`${prefix}--text-area__wrapper`]: true,
            [`${prefix}--text-area__wrapper--warn`]: this.warn,
            [`${prefix}--text-area__wrapper--readonly`]: this.readonly,
            [`${prefix}--text-area__wrapper--slug`]: this._hasAILabel,
        });
        const labelClasses = classMap({
            [`${prefix}--label`]: true,
            [`${prefix}--label--disabled`]: this.disabled,
        });
        const helperTextClasses = classMap({
            [`${prefix}--form__helper-text`]: true,
            [`${prefix}--form__helper-text--disabled`]: this.disabled,
        });
        const counter = enableCounter && maxCount
            ? html ` <label class="${labelClasses}">
            <slot name="label-text">${textCount}/${maxCount}</slot>
          </label>`
            : null;
        const icon = () => {
            if (this.invalid) {
                return invalidIcon;
            }
            else if (this.warn && !this.invalid) {
                return warnIcon;
            }
            return null;
        };
        return html `
      <div class="${prefix}--text-area__label-wrapper">
        ${this.hideLabel
            ? html ``
            : html `
              <label class="${labelClasses}" for="input">
                <slot name="label-text"> ${this.label} </slot>
              </label>
            `}
        ${counter}
      </div>
      <div class="${textareaWrapperClasses}" ?data-invalid="${this.invalid}">
        ${icon()}
        <textarea
          autocomplete="${this.autocomplete}"
          ?autofocus="${this.autofocus}"
          class="${textareaClasses}"
          cols="${ifDefined(this.cols)}"
          ?data-invalid="${this.invalid}"
          ?disabled="${this.disabled}"
          id="input"
          name="${ifNonEmpty(this.name)}"
          pattern="${ifNonEmpty(this.pattern)}"
          placeholder="${ifNonEmpty(this.placeholder)}"
          ?readonly="${this.readonly}"
          ?required="${this.required}"
          rows="${ifDefined(this.rows)}"
          .value="${this.value}"
          maxlength="${ifNonEmpty(this.maxCount)}"
          @input="${this._handleInput}"></textarea>
        <slot name="ai-label" @slotchange="${this._handleSlotChange}"></slot>
        <slot name="slug" @slotchange="${this._handleSlotChange}"></slot>
      </div>
      <div class="${helperTextClasses}" ?hidden="${this.invalid || this.warn}">
        <slot name="helper-text"> ${this.helperText} </slot>
      </div>
      <div
        class="${prefix}--form-requirement"
        ?hidden="${!this.invalid && !this.warn}">
        <slot name="${this.invalid ? 'invalid-text' : 'warn-text'}">
          ${this.invalid ? this.invalidText : this.warnText}
        </slot>
      </div>
    `;
    }
};
CDSTextarea.shadowRootOptions = Object.assign(Object.assign({}, LitElement.shadowRootOptions), { delegatesFocus: true });
CDSTextarea.styles = styles; // `styles` here is a `CSSResult` generated by custom Vite loader
__decorate([
    property({ type: Number })
], CDSTextarea.prototype, "cols", void 0);
__decorate([
    property()
], CDSTextarea.prototype, "id", void 0);
__decorate([
    property()
], CDSTextarea.prototype, "pattern", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSTextarea.prototype, "required", void 0);
__decorate([
    property()
], CDSTextarea.prototype, "rows", void 0);
__decorate([
    query('textarea')
], CDSTextarea.prototype, "_textarea", void 0);
CDSTextarea = __decorate([
    carbonElement(`${prefix}-textarea`)
], CDSTextarea);
var CDSTextarea$1 = CDSTextarea;

export { CDSTextarea$1 as default };
//# sourceMappingURL=textarea.js.map
