/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { html } from 'lit';
import { classMap } from 'lit/directives/class-map.js';
import { property } from 'lit/decorators.js';
import { prefix } from '../../globals/settings.js';
import CDSLink from '../link/link.js';
import { FORM_ELEMENT_COLOR_SCHEME } from '../../globals/shared-enums.js';
import styles from './tile.scss.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
// To Do: Replace with an an icon from `@carbon/icons`
// since the hollow AI Label in `ClickableTile` is not interactive
const aiLabelIcon = html ` <svg
  class="${prefix}--tile--slug-icon"
  width="24"
  height="24"
  viewBox="0 0 24 24"
  fill="none"
  xmlns="http://www.w3.org/2000/svg">
  <rect x="0.5" y="0.5" width="23" height="23" />
  <path
    d="M13.2436 16H11.5996L10.9276 13.864H7.95164L7.29164 16H5.68364L8.49164 7.624H10.4596L13.2436 16ZM10.5436 12.508L9.46364 9.064H9.40364L8.33564 12.508H10.5436ZM17.9341 16H14.1301V14.728H15.2341V8.896H14.1301V7.624H17.9341V8.896H16.8181V14.728H17.9341V16Z"
    fill="#161616" />
</svg>`;
/**
 * Clickable tile.
 *
 * @element cds-clickable-tile
 */
let CDSClickableTile = class CDSClickableTile extends CDSLink {
    constructor() {
        super(...arguments);
        /**
         * The color scheme.
         */
        this.colorScheme = FORM_ELEMENT_COLOR_SCHEME.REGULAR;
        /**
         * The a11y role for `<a>`.
         */
        this.linkRole = 'button';
        /**
         * Specify if the `ClickableTile` component should be rendered with rounded corners.
         * Only valid when `ai-label` prop is present
         */
        this.hasRoundedCorners = false;
        this.aiLabel = false;
        /**
         * deprecated - remove in v12
         */
        this.slug = false;
    }
    get _classes() {
        const { colorScheme, disabled, hasRoundedCorners, aiLabel, slug } = this;
        return classMap({
            [`${prefix}--link`]: true,
            [`${prefix}--link--disabled`]: disabled,
            [`${prefix}--tile`]: true,
            [`${prefix}--tile--clickable`]: true,
            [`${prefix}--tile--${colorScheme}`]: colorScheme,
            [`${prefix}--tile--slug-rounded`]: (aiLabel || slug) && hasRoundedCorners,
        });
    }
    /**
     * If using `slug`, set `ai-label` attribute to true so
     * the styles are applied for slug as well
     *
     * remove in v12
     */
    connectedCallback() {
        if (this.slug) {
            this.setAttribute('ai-Label', '');
            this.aiLabel = true;
        }
        super.connectedCallback();
    }
    /**
     * @returns The inner content.
     */
    _renderInner() {
        return html `
      ${super._renderInner()} ${this.aiLabel || this.slug ? aiLabelIcon : ''}
    `;
    }
    /**
     * A selector that will return the slug item.
     *
     * remove in v12
     */
    static get slugItem() {
        return `${prefix}-slug`;
    }
    /**
     * A selector that will return the AI Label item.
     */
    static get aiLabelItem() {
        return `${prefix}-ai-label`;
    }
};
CDSClickableTile.styles = styles;
__decorate([
    property({ attribute: 'color-scheme', reflect: true })
], CDSClickableTile.prototype, "colorScheme", void 0);
__decorate([
    property({ attribute: 'link-role' })
], CDSClickableTile.prototype, "linkRole", void 0);
__decorate([
    property({ type: Boolean, attribute: 'has-rounded-corners' })
], CDSClickableTile.prototype, "hasRoundedCorners", void 0);
__decorate([
    property({ type: Boolean })
], CDSClickableTile.prototype, "aiLabel", void 0);
__decorate([
    property({ type: Boolean })
], CDSClickableTile.prototype, "slug", void 0);
CDSClickableTile = __decorate([
    carbonElement(`${prefix}-clickable-tile`)
], CDSClickableTile);
var CDSClickableTile$1 = CDSClickableTile;

export { CDSClickableTile$1 as default };
//# sourceMappingURL=clickable-tile.js.map
