/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { html, LitElement } from 'lit';
import { property } from 'lit/decorators.js';
import { ifDefined } from 'lit/directives/if-defined.js';
import ChevronDown16 from '../../icons/chevron--down/16.js';
import { prefix } from '../../globals/settings.js';
import HostListener from '../../globals/decorators/host-listener.js';
import FocusMixin from '../../globals/mixins/focus.js';
import HostListenerMixin from '../../globals/mixins/host-listener.js';
import { FORM_ELEMENT_COLOR_SCHEME } from '../../globals/shared-enums.js';
import styles from './tile.scss.js';
import { classMap } from 'lit/directives/class-map.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2023
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Expandable tile.
 *
 * @element cds-expandable-tile
 * @fires cds-expandable-tile-beingtoggled
 *   The custom event fired before the expanded state is changed upon a user gesture.
 *   Cancellation of this event stops changing the user-initiated change in expanded state.
 * @fires cds-expandable-tile-toggled - The custom event fired after a the expanded state is changed upon a user gesture.
 */
let CDSExpandableTile = class CDSExpandableTile extends HostListenerMixin(FocusMixin(LitElement)) {
    constructor() {
        super(...arguments);
        /**
         * The computed height of the below-the-fold content.
         */
        this._belowTheContentHeight = 0;
        /**
         * `true` if there is an AI Label.
         */
        this._hasAILabel = false;
        this._handleClick = () => {
            if (!this.withInteractive) {
                this._handleExpand();
            }
        };
        /**
         * The color scheme.
         */
        this.colorScheme = FORM_ELEMENT_COLOR_SCHEME.REGULAR;
        /**
         * `true` to expand this expandable tile.
         */
        this.expanded = false;
        /**
         * Specify if the `ExpandableTile` component should be rendered with rounded corners.
         * Only valid when `ai-label` prop is present
         */
        this.hasRoundedCorners = false;
        /**
         * `true` to expand this expandable tile.
         */
        this.withInteractive = false;
    }
    /**
     * Handles `slotchange` event.
     */
    _handleSlotChange({ target }) {
        const hasContent = target
            .assignedNodes()
            .filter((elem) => {
            var _a, _b;
            return elem.matches !== undefined
                ? elem.matches((_a = this.constructor) === null || _a === void 0 ? void 0 : _a.aiLabelItem) ||
                    // remove reference of slug in v12
                    elem.matches((_b = this.constructor) === null || _b === void 0 ? void 0 : _b.slugItem)
                : false;
        });
        if (hasContent.length > 0) {
            this._hasAILabel = Boolean(hasContent);
            hasContent[0].setAttribute('size', 'xs');
        }
        this.requestUpdate();
    }
    /**
     * Handles `slotchange` event on the below-the-fold content.
     *
     * @param event The event.
     */
    _handleSlotChangeBelowTheFoldContent(event) {
        this._belowTheContentHeight = event.target
            .assignedNodes()
            .reduce((acc, item) => { var _a; return acc + ((_a = item.offsetHeight) !== null && _a !== void 0 ? _a : 0); }, 0);
        if (!this._belowTheContentHeight) {
            const element = getComputedStyle(this.querySelector('cds-tile-below-the-fold-content'));
            this._belowTheContentHeight = parseInt(element.height, 10);
        }
        this.requestUpdate();
    }
    _handleExpand() {
        const expanded = !this.expanded;
        this.focus();
        const init = {
            bubbles: true,
            composed: true,
            detail: {
                expanded,
            },
        };
        const constructor = this.constructor;
        const beforeChangeEvent = new CustomEvent(constructor.eventBeforeToggle, Object.assign(Object.assign({}, init), { cancelable: true }));
        if (this.dispatchEvent(beforeChangeEvent)) {
            this.expanded = expanded;
            const afterChangeEvent = new CustomEvent(constructor.eventToggle, init);
            this.dispatchEvent(afterChangeEvent);
        }
    }
    updated() {
        if (this._hasAILabel) {
            this.setAttribute('ai-label', '');
        }
        else {
            this.removeAttribute('ai-label');
        }
    }
    render() {
        const { expanded, withInteractive, _belowTheContentHeight: belowTheContentHeight, _handleSlotChangeBelowTheFoldContent: handleSlotChangeBelowTheFoldContent, } = this;
        const classes = classMap({
            [`${prefix}--tile__chevron`]: true,
            [`${prefix}--tile__chevron--interactive`]: withInteractive,
        });
        return html `
      <button
        class="${classes}"
        aria-labelledby="above-the-fold-content"
        aria-controls="below-the-fold-content"
        tabindex="0"
        @click="${withInteractive ? this._handleExpand : ''}"
        aria-expanded="${String(Boolean(expanded))}">
        ${ChevronDown16({
            id: 'icon',
        })}
      </button>
      <slot name="ai-label" @slotchange="${this._handleSlotChange}"></slot>
      <slot name="slug" @slotchange="${this._handleSlotChange}"></slot>
      <div id="content" class="${prefix}--tile-content">
        <div><slot name="above-the-fold-content"></slot></div>
        <div
          class="${prefix}-ce--expandable-tile--below-the-fold-content"
          style="${ifDefined(!expanded ? undefined : `max-height: ${belowTheContentHeight}px`)}">
          <slot @slotchange="${handleSlotChangeBelowTheFoldContent}"></slot>
        </div>
      </div>
    `;
    }
    /**
     * A selector that will return the slug item.
     *
     * remove in v12
     */
    static get slugItem() {
        return `${prefix}-slug`;
    }
    /**
     * A selector that will return the AI Label item.
     */
    static get aiLabelItem() {
        return `${prefix}-ai-label`;
    }
    /**
     * The name of the custom event fired before the expanded state is changed upon a user gesture.
     * Cancellation of this event stops changing the user-initiated change in expanded state.
     */
    static get eventBeforeToggle() {
        return `${prefix}-expandable-tile-beingtoggled`;
    }
    /**
     * The name of the custom event fired after a the expanded state is changed upon a user gesture.
     */
    static get eventToggle() {
        return `${prefix}-expandable-tile-toggled`;
    }
};
CDSExpandableTile.styles = styles; // `styles` here is a `CSSResult` generated by custom Vite loader
__decorate([
    HostListener('click')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSExpandableTile.prototype, "_handleClick", void 0);
__decorate([
    property({ attribute: 'color-scheme', reflect: true })
], CDSExpandableTile.prototype, "colorScheme", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSExpandableTile.prototype, "expanded", void 0);
__decorate([
    property({ type: Boolean, attribute: 'has-rounded-corners' })
], CDSExpandableTile.prototype, "hasRoundedCorners", void 0);
__decorate([
    property({ type: Boolean, reflect: true, attribute: 'with-interactive' })
], CDSExpandableTile.prototype, "withInteractive", void 0);
CDSExpandableTile = __decorate([
    carbonElement(`${prefix}-expandable-tile`)
], CDSExpandableTile);
var CDSExpandableTile$1 = CDSExpandableTile;

export { CDSExpandableTile$1 as default };
//# sourceMappingURL=expandable-tile.js.map
