/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { html, svg } from 'lit';
import { classMap } from 'lit/directives/class-map.js';
import { ifDefined } from 'lit/directives/if-defined.js';
import { prefix } from '../../globals/settings.js';
import SelectableTile from './selectable-tile.js';
import CheckmarkFilled16 from '../../icons/checkmark--filled/16.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Single-selectable tile.
 *
 * @element cds-radio-tile
 * @fires cds-radio-tile-selected
 *   The name of the custom event fired after this selectable tile changes its selected state.
 */
let CDSRadioTile = class CDSRadioTile extends SelectableTile {
    constructor() {
        super(...arguments);
        /**
         * The `type` attribute of the `<input>`.
         */
        this._inputType = 'radio';
    }
    /**
     * Handles `change` event on the `<input>` in the shadow DOM.
     */
    _handleChange() {
        this.selected = true;
        const { selected, name } = this;
        const { eventRadioChange } = this.constructor;
        this.dispatchEvent(new CustomEvent(eventRadioChange, {
            bubbles: true,
            composed: true,
            detail: {
                selected,
                name,
            },
        }));
    }
    render() {
        const { colorScheme, checkmarkLabel, name, selected, value, _inputType: inputType, _handleChange: handleChange, } = this;
        const classes = classMap({
            [`${prefix}--tile`]: true,
            [`${prefix}--tile--selectable`]: true,
            [`${prefix}--tile--is-selected`]: selected,
            [`${prefix}--tile--${colorScheme}`]: colorScheme,
        });
        return html `
      <input
        type="${inputType}"
        id="input"
        class="${prefix}--tile-input"
        tabindex="-1"
        name="${ifDefined(name)}"
        value="${ifDefined(value)}"
        .checked=${selected}
        @change=${handleChange} />
      <label for="input" class="${classes}" tabindex="0">
        <div class="${prefix}--tile__checkmark">
          ${CheckmarkFilled16({
            children: !checkmarkLabel
                ? undefined
                : svg `<title>${checkmarkLabel}</title>`,
        })}
        </div>
        <div class="${prefix}--tile-content"><slot></slot></div>
      </label>
    `;
    }
    /**
     * The name of the custom event fired after this selectable tile changes its selected state.
     */
    static get eventRadioChange() {
        return `${prefix}-radio-tile-selected`;
    }
};
CDSRadioTile = __decorate([
    carbonElement(`${prefix}-radio-tile`)
], CDSRadioTile);
var CDSRadioTile$1 = CDSRadioTile;

export { CDSRadioTile$1 as default };
//# sourceMappingURL=radio-tile.js.map
