/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { classMap } from 'lit/directives/class-map.js';
import { html, svg, LitElement } from 'lit';
import { query, property } from 'lit/decorators.js';
import { ifDefined } from 'lit/directives/if-defined.js';
import Checkbox16 from '../../icons/checkbox/16.js';
import CheckboxCheckedFilled16 from '../../icons/checkbox--checked--filled/16.js';
import { prefix } from '../../globals/settings.js';
import FocusMixin from '../../globals/mixins/focus.js';
import HostListenerMixin from '../../globals/mixins/host-listener.js';
import { FORM_ELEMENT_COLOR_SCHEME } from '../../globals/shared-enums.js';
import styles from './tile.scss.js';
import HostListener from '../../globals/decorators/host-listener.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2023
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Multi-selectable tile.
 *
 * @element cds-selectable-tile
 * @fires cds-selectable-tile-changed - The custom event fired after this selectable tile changes its selected state.
 */
let CDSSelectableTile = class CDSSelectableTile extends HostListenerMixin(FocusMixin(LitElement)) {
    constructor() {
        super(...arguments);
        /**
         * The `type` attribute of the `<input>`.
         */
        this._inputType = 'checkbox';
        /**
         * `true` if there is an AI Label.
         */
        this._hasAILabel = false;
        /**
         * Listener function for keyboard interaction.
         *
         * @param event to get the key pressed
         */
        this._handleKeydown = (event) => {
            var _a, _b;
            const { key } = event;
            if (key === ' ' ||
                (key === 'Enter' &&
                    !((_a = event.target) === null || _a === void 0 ? void 0 : _a.matches(this.constructor.aiLabelItem)) &&
                    !((_b = event.target) === null || _b === void 0 ? void 0 : _b.matches(this.constructor.slugItem)))) {
                this.selected = !this.selected;
            }
        };
        /**
         * The color scheme.
         */
        this.colorScheme = FORM_ELEMENT_COLOR_SCHEME.REGULAR;
        /**
         * Specify if the `SeletableTile` component should be rendered with rounded corners.
         * Only valid when `ai-label` prop is present
         */
        this.hasRoundedCorners = false;
        /**
         * `true` to show the selected state.
         */
        this.selected = false;
    }
    /**
     * Handles `slotchange` event.
     */
    _handleSlotChange({ target }) {
        const hasContent = target
            .assignedNodes()
            .filter((elem) => {
            var _a, _b;
            return elem.matches !== undefined
                ? elem.matches((_a = this.constructor) === null || _a === void 0 ? void 0 : _a.aiLabelItem) ||
                    // remove reference of slug in v12
                    elem.matches((_b = this.constructor) === null || _b === void 0 ? void 0 : _b.slugItem)
                : false;
        });
        if (hasContent.length > 0) {
            this._hasAILabel = Boolean(hasContent);
            hasContent[0].setAttribute('size', 'xs');
        }
        this.requestUpdate();
    }
    /**
     * Handles `change` event on the `<input>` in the shadow DOM.
     */
    _handleChange() {
        this.selected = this._inputNode.checked;
        const selected = this.selected;
        const { eventChange } = this.constructor;
        this.dispatchEvent(new CustomEvent(eventChange, {
            bubbles: true,
            composed: true,
            detail: {
                selected,
            },
        }));
    }
    /**
     * Handles the rendering of the icon.
     */
    _renderIcon() {
        const { selected, checkmarkLabel } = this;
        return html ` ${selected
            ? CheckboxCheckedFilled16({
                children: !checkmarkLabel
                    ? undefined
                    : svg `<title>${checkmarkLabel}</title>`,
            })
            : Checkbox16({
                children: !checkmarkLabel
                    ? undefined
                    : svg `<title>${checkmarkLabel}</title>`,
            })}`;
    }
    updated() {
        if (this._hasAILabel) {
            this.setAttribute('ai-label', '');
        }
        else {
            this.removeAttribute('ai-label');
        }
    }
    render() {
        const { colorScheme, hasRoundedCorners: hasRoundedCorners, name, selected, value, _inputType: inputType, _handleChange: handleChange, _hasAILabel: hasAILabel, } = this;
        const classes = classMap({
            [`${prefix}--tile`]: true,
            [`${prefix}--tile--selectable`]: true,
            [`${prefix}--tile--is-selected`]: selected,
            [`${prefix}--tile--${colorScheme}`]: colorScheme,
            [`${prefix}--tile--slug-rounded`]: hasAILabel && hasRoundedCorners,
        });
        return html `
      <input
        type="${inputType}"
        id="input"
        class="${prefix}--tile-input"
        tabindex="-1"
        name="${ifDefined(name)}"
        value="${ifDefined(value)}"
        .checked=${selected}
        @change=${handleChange} />
      <label for="input" class="${classes}" tabindex="0">
        <div
          class="${prefix}--tile__checkmark ${prefix}--tile__checkmark--persistent">
          ${this._renderIcon()}
        </div>
        <div class="${prefix}--tile-content"><slot></slot></div>
      </label>
      <slot name="ai-label" @slotchange="${this._handleSlotChange}"></slot>
      <slot name="slug" @slotchange="${this._handleSlotChange}"></slot>
    `;
    }
    /**
     * A selector that will return the slug item.
     *
     * remove in v12
     */
    static get slugItem() {
        return `${prefix}-slug`;
    }
    /**
     * A selector that will return the AI Label item.
     */
    static get aiLabelItem() {
        return `${prefix}-ai-label`;
    }
    /**
     * The name of the custom event fired after this selectable tile changes its selected state.
     */
    static get eventChange() {
        return `${prefix}-selectable-tile-changed`;
    }
};
CDSSelectableTile.shadowRootOptions = Object.assign(Object.assign({}, LitElement.shadowRootOptions), { delegatesFocus: true });
CDSSelectableTile.styles = styles;
__decorate([
    query('input')
], CDSSelectableTile.prototype, "_inputNode", void 0);
__decorate([
    HostListener('keydown')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSSelectableTile.prototype, "_handleKeydown", void 0);
__decorate([
    property({ attribute: 'checkmark-label' })
], CDSSelectableTile.prototype, "checkmarkLabel", void 0);
__decorate([
    property({ attribute: 'color-scheme', reflect: true })
], CDSSelectableTile.prototype, "colorScheme", void 0);
__decorate([
    property({ type: Boolean, attribute: 'has-rounded-corners' })
], CDSSelectableTile.prototype, "hasRoundedCorners", void 0);
__decorate([
    property()
], CDSSelectableTile.prototype, "name", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSSelectableTile.prototype, "selected", void 0);
__decorate([
    property()
], CDSSelectableTile.prototype, "value", void 0);
CDSSelectableTile = __decorate([
    carbonElement(`${prefix}-selectable-tile`)
], CDSSelectableTile);
var SelectableTile = CDSSelectableTile;

export { SelectableTile as default };
//# sourceMappingURL=selectable-tile.js.map
