/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { html, LitElement } from 'lit';
import { property } from 'lit/decorators.js';
import { prefix } from '../../globals/settings.js';
import { NAVIGATION_DIRECTION } from '../../globals/internal/radio-group-manager.js';
import HostListener from '../../globals/decorators/host-listener.js';
import HostListenerMixin from '../../globals/mixins/host-listener.js';
import styles from './tile.scss.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Map of navigation direction by key.
 */
const navigationDirectionForKey = {
    ArrowUp: NAVIGATION_DIRECTION.BACKWARD,
    Up: NAVIGATION_DIRECTION.BACKWARD,
    ArrowDown: NAVIGATION_DIRECTION.FORWARD,
    Down: NAVIGATION_DIRECTION.FORWARD, // IE
};
/**
 * Tile group.
 *
 * @element cds-tile-group
 * @fires cds-current-radio-tile-selection
 *   The name of the custom event fired after a radio tile changes its selected state.
 * @fires cds-current-selectable-tile-selections
 *   The name of the custom event fired after a selectable tile changes its selected state.
 */
let CDSTileGroup = class CDSTileGroup extends HostListenerMixin(LitElement) {
    constructor() {
        super(...arguments);
        /**
         * Keyboard listener to ensure keyboard navigation.
         *
         * @param event to get key pressed
         */
        this._handleKeydown = (event) => {
            const { target, key } = event;
            const { radioTiles, selectableTiles } = this;
            const navigationDirection = navigationDirectionForKey[key];
            const tiles = radioTiles.length ? radioTiles : selectableTiles;
            const currentIndex = [...tiles].findIndex((e) => e == target);
            const nextIndex = currentIndex + navigationDirection;
            const nextSibling = nextIndex !== -1
                ? tiles[nextIndex % tiles.length]
                : tiles[tiles.length - 1];
            if (navigationDirection) {
                event.preventDefault(); // Prevent default (scrolling) behavior
                if (this.radioTiles.length) {
                    this._handleKeydownRadio(nextSibling);
                }
                else {
                    this._handleKeydownSelectable(event, nextSibling);
                }
            }
            else if (key === ' ' || key === 'Enter') {
                this._handleKeydownSelectable(event);
            }
        };
        /**
         * Focus listener to focus on selected element upon focusing
         *
         * @param event to get focused
         */
        this._handleFocus = (event) => {
            const { relatedTarget, target } = event;
            if (this.radioTiles.length) {
                if (!this.currentRadioSelection) {
                    target.toggleAttribute('selected');
                    this.currentRadioSelection = target;
                }
                else if (!(relatedTarget === null || relatedTarget === void 0 ? void 0 : relatedTarget.matches(this.constructor.selectorRadioTile)) &&
                    target !== this.currentRadioSelection) {
                    this.currentRadioSelection.focus();
                }
            }
        };
        this.currentSelections = [];
    }
    _handleRadioClick(event) {
        const { target } = event;
        const { currentRadioSelection } = this;
        const { eventCurrentRadioTileSelection } = this
            .constructor;
        if (!currentRadioSelection) {
            this.currentRadioSelection = target;
        }
        else if (currentRadioSelection !== target) {
            currentRadioSelection.toggleAttribute('selected');
            this.currentRadioSelection = target;
        }
        this.dispatchEvent(new CustomEvent(eventCurrentRadioTileSelection, {
            bubbles: true,
            composed: true,
            detail: {
                target,
            },
        }));
    }
    _handleSelectableClick(event) {
        const { target } = event;
        const { currentSelections } = this;
        const { eventCurrentSelectableTilesSelection } = this
            .constructor;
        if (!currentSelections.includes(target)) {
            currentSelections.push(target);
        }
        else {
            currentSelections.splice(currentSelections.indexOf(target), 1);
        }
        target.toggleAttribute('selected');
        this.dispatchEvent(new CustomEvent(eventCurrentSelectableTilesSelection, {
            bubbles: true,
            composed: true,
            detail: {
                currentSelections,
            },
        }));
        event.stopPropagation();
        event.preventDefault();
    }
    /**
     * Click listener to ensure selectability.
     *
     * @param event click
     */
    _handleTileSelect(event) {
        if (this.radioTiles.length) {
            this._handleRadioClick(event);
        }
        else {
            this._handleSelectableClick(event);
        }
    }
    /**
     * Handle keyboard navigation for radio tiles
     *
     * @param nextSibling to focus on
     */
    _handleKeydownRadio(nextSibling) {
        const { currentRadioSelection } = this;
        const { eventCurrentRadioTileSelection } = this
            .constructor;
        if (currentRadioSelection) {
            currentRadioSelection.toggleAttribute('selected');
        }
        nextSibling.focus();
        nextSibling.toggleAttribute('selected');
        this.currentRadioSelection = nextSibling;
        this.dispatchEvent(new CustomEvent(eventCurrentRadioTileSelection, {
            bubbles: true,
            composed: true,
            detail: {
                nextSibling,
            },
        }));
    }
    /**
     * Handle keyboard navigation for selectable tiles
     *
     * @param event to get target
     * @param nextSibling to focus on
     */
    _handleKeydownSelectable(event, nextSibling) {
        const { target } = event;
        const { currentSelections } = this;
        const { eventCurrentSelectableTilesSelection } = this
            .constructor;
        if (nextSibling) {
            nextSibling.focus();
        }
        else {
            if (!currentSelections.includes(target)) {
                currentSelections.push(target);
            }
            else {
                currentSelections.splice(currentSelections.indexOf(target), 1);
            }
            this.dispatchEvent(new CustomEvent(eventCurrentSelectableTilesSelection, {
                bubbles: true,
                composed: true,
                detail: {
                    currentSelections,
                },
            }));
        }
    }
    firstUpdated() {
        if (!this.hasAttribute('role')) {
            this.setAttribute('role', 'group');
        }
        if (!this.radioTiles) {
            this.radioTiles = this.querySelectorAll(this.constructor.selectorRadioTile);
        }
        if (!this.selectableTiles) {
            this.selectableTiles = this.querySelectorAll(this.constructor.selectorSelectableTile);
        }
        if (this.disabled) {
            this.radioTiles.forEach((e) => e.toggleAttribute('disabled'));
            this.selectableTiles.forEach((e) => e.toggleAttribute('disabled'));
        }
    }
    render() {
        const { fieldsetClassName, disabled } = this;
        return html `
      <fieldset class="${fieldsetClassName}" ?disabled=${disabled}>
        <slot name="legend" class="${prefix}--label"></slot>
        <slot></slot>
      </fieldset>
    `;
    }
    /**
     * A selector that selects a radio tile component.
     */
    static get selectorRadioTile() {
        return `${prefix}-radio-tile`;
    }
    /**
     * A selector that selects a selectable tile component.
     */
    static get selectorSelectableTile() {
        return `${prefix}-selectable-tile`;
    }
    /**
     * The name of the custom event fired after a radio tile changes its selected state.
     */
    static get eventCurrentRadioTileSelection() {
        return `${prefix}-current-radio-tile-selection`;
    }
    /**
     * The name of the custom event fired after a radio tile changes its selected state.
     */
    static get eventCurrentSelectableTilesSelection() {
        return `${prefix}-current-selectable-tile-selections`;
    }
};
CDSTileGroup.styles = styles; // `styles` here is a `CSSResult` generated by custom Vite loader
__decorate([
    HostListener('click')
    // @ts-ignore
], CDSTileGroup.prototype, "_handleTileSelect", null);
__decorate([
    HostListener('keydown')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSTileGroup.prototype, "_handleKeydown", void 0);
__decorate([
    HostListener('focusin')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSTileGroup.prototype, "_handleFocus", void 0);
__decorate([
    property({ reflect: true, attribute: 'fieldset-class-name' })
], CDSTileGroup.prototype, "fieldsetClassName", void 0);
__decorate([
    property({ reflect: true, type: Boolean })
], CDSTileGroup.prototype, "disabled", void 0);
__decorate([
    property()
], CDSTileGroup.prototype, "currentRadioSelection", void 0);
__decorate([
    property()
], CDSTileGroup.prototype, "currentSelections", void 0);
__decorate([
    property()
], CDSTileGroup.prototype, "radioTiles", void 0);
__decorate([
    property()
], CDSTileGroup.prototype, "selectableTiles", void 0);
CDSTileGroup = __decorate([
    carbonElement(`${prefix}-tile-group`)
], CDSTileGroup);
var CDSTileGroup$1 = CDSTileGroup;

export { CDSTileGroup$1 as default };
//# sourceMappingURL=tile-group.js.map
