/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { LitElement, html } from 'lit';
import { property } from 'lit/decorators.js';
import { prefix } from '../../globals/settings.js';
import { FORM_ELEMENT_COLOR_SCHEME } from '../../globals/shared-enums.js';
export { FORM_ELEMENT_COLOR_SCHEME as TILE_COLOR_SCHEME } from '../../globals/shared-enums.js';
import styles from './tile.scss.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2023
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Basic tile.
 *
 * @element cds-tile
 */
let CDSTile = class CDSTile extends LitElement {
    constructor() {
        super(...arguments);
        /**
         * `true` if there is an AI Label.
         */
        this._hasAILabel = false;
        /**
         * The color scheme.
         */
        this.colorScheme = FORM_ELEMENT_COLOR_SCHEME.REGULAR;
        /**
         * Specify if the `Tile` component should be rendered with rounded corners.
         * Only valid when `ai-label` prop is present
         */
        this.hasRoundedCorners = false;
    }
    /**
     * Handles `slotchange` event.
     */
    _handleSlotChange({ target }) {
        const hasContent = target
            .assignedNodes()
            .filter((elem) => elem.matches !== undefined
            ? elem.matches(this.constructor.aiLabelItem) ||
                // remove reference of slug in v12
                elem.matches(this.constructor.slugItem)
            : false);
        if (hasContent.length > 0) {
            this._hasAILabel = Boolean(hasContent);
            hasContent[0].setAttribute('size', 'xs');
        }
        this.requestUpdate();
    }
    updated() {
        const anchorTag = this.querySelector('a');
        if (anchorTag) {
            anchorTag === null || anchorTag === void 0 ? void 0 : anchorTag.classList.add(`${prefix}--link`);
            anchorTag.before(document.createElement('br'));
            anchorTag.before(document.createElement('br'));
        }
        if (this._hasAILabel) {
            this.setAttribute('ai-label', '');
        }
        else {
            this.removeAttribute('ai-label');
        }
    }
    render() {
        return html ` <slot></slot
      ><slot name="ai-label" @slotchange="${this._handleSlotChange}"></slot
      ><slot name="slug" @slotchange="${this._handleSlotChange}"></slot>`;
    }
    /**
     * A selector that will return the slug item.
     *
     * remove in v12
     */
    static get slugItem() {
        return `${prefix}-slug`;
    }
    /**
     * A selector that will return the AI Label item.
     */
    static get aiLabelItem() {
        return `${prefix}-ai-label`;
    }
};
CDSTile.styles = styles;
__decorate([
    property({ attribute: 'color-scheme', reflect: true })
], CDSTile.prototype, "colorScheme", void 0);
__decorate([
    property({ type: Boolean, attribute: 'has-rounded-corners' })
], CDSTile.prototype, "hasRoundedCorners", void 0);
CDSTile = __decorate([
    carbonElement(`${prefix}-tile`)
], CDSTile);
var CDSTile$1 = CDSTile;

export { CDSTile$1 as default };
//# sourceMappingURL=tile.js.map
