/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { classMap } from 'lit/directives/class-map.js';
import { html, LitElement } from 'lit';
import { property } from 'lit/decorators.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';
import Information16 from '../../icons/information/16.js';
import { prefix } from '../../globals/settings.js';
import HostListener from '../../globals/decorators/host-listener.js';
import HostListenerMixin from '../../globals/mixins/host-listener.js';
import FocusMixin from '../../globals/mixins/focus.js';
import { POPOVER_ALIGNMENT } from '../popover/defs.js';
import PopoverController from '../../globals/controllers/popover-controller.js';
import styles from './toggletip.scss.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
var CDSToggletip_1;
/**
 * Definition tooltip.
 *
 * @element cds-toggletip
 */
let CDSToggletip = CDSToggletip_1 = class CDSToggletip extends HostListenerMixin(FocusMixin(LitElement)) {
    constructor() {
        super(...arguments);
        /**
         * Create popover controller instance
         */
        this.popoverController = new PopoverController(this);
        /**
         * How the tooltip is aligned to the trigger button.
         */
        this.alignment = POPOVER_ALIGNMENT.TOP;
        /**
         * Specify whether a auto align functionality should be applied
         */
        this.autoalign = false;
        /**
         * Set whether toggletip is open
         */
        this.open = false;
        this._handleClick = () => {
            this.open = !this.open;
        };
        /**
         * Handles `keydown` event on this element.
         */
        this._handleKeydown = async (event) => {
            if (event.key === 'Escape') {
                this.open = false;
            }
        };
        this._renderToggleTipLabel = () => {
            return html `
      <span class="${prefix}--toggletip-label">
        <slot></slot>
      </span>
    `;
        };
        this._renderTooltipButton = () => {
            return html `
      <button
        aria-controls="${this.id}"
        class="${prefix}--toggletip-button"
        @click=${this._handleClick}>
        ${Information16({ id: 'trigger' })}
      </button>
    `;
        };
        this._renderTooltipContent = () => {
            return this.autoalign
                ? html `
          <span class="${prefix}--popover-content">
            <div class="${prefix}--toggletip-content">
              <slot name="body-text"></slot>
              <div class="${prefix}--toggletip-actions">
                <slot
                  name="actions"
                  @slotchange="${this._handleActionsSlotChange}"></slot>
              </div>
            </div>
            <span class="${prefix}--popover-caret"></span>
          </span>
        `
                : html `
          <span class="${prefix}--popover">
            <span class="${prefix}--popover-content">
              <div class="${prefix}--toggletip-content">
                <slot name="body-text"></slot>
                <div class="${prefix}--toggletip-actions">
                  <slot
                    name="actions"
                    @slotchange="${this._handleActionsSlotChange}"></slot>
                </div>
              </div>
            </span>
            <span class="${prefix}--popover-caret"></span>
          </span>
        `;
        };
        this._renderInnerContent = () => {
            return html `
      ${this._renderTooltipButton()} ${this._renderTooltipContent()}
    `;
        };
    }
    /**
     * Handles `slotchange` event.
     */
    _handleActionsSlotChange({ target }) {
        const hasContent = target.assignedNodes();
        hasContent
            ? this.setAttribute('has-actions', '')
            : this.removeAttribute('has-actions');
    }
    /**
     * Handles `blur` event handler on the document this element is in.
     *
     * @param event The event.
     */
    _handleFocusOut(event) {
        if (!this.contains(event.relatedTarget)) {
            this.open = false;
        }
    }
    updated() {
        var _a, _b, _c, _d;
        if (this.autoalign && this.open) {
            // auto align functionality with @floating-ui/dom library
            const button = (_a = this.shadowRoot) === null || _a === void 0 ? void 0 : _a.querySelector(CDSToggletip_1.selectorToggletipButton);
            const tooltip = (_b = this.shadowRoot) === null || _b === void 0 ? void 0 : _b.querySelector(CDSToggletip_1.selectorToggletipContent);
            const arrowElement = (_c = this.shadowRoot) === null || _c === void 0 ? void 0 : _c.querySelector(CDSToggletip_1.selectorToggletipCaret);
            if (button && tooltip) {
                (_d = this.popoverController) === null || _d === void 0 ? void 0 : _d.setPlacement({
                    trigger: button,
                    target: tooltip,
                    arrowElement: arrowElement,
                    caret: true,
                    flip: true,
                    alignment: this.alignment,
                });
            }
        }
    }
    render() {
        const { alignment, open } = this;
        const classes = classMap({
            [`${prefix}--popover-container`]: true,
            [`${prefix}--popover--caret`]: true,
            [`${prefix}--popover--high-contrast`]: true,
            [`${prefix}--popover--open`]: open,
            [`${prefix}--popover--${alignment}`]: alignment,
            [`${prefix}--toggletip`]: true,
            [`${prefix}--toggletip--open`]: open,
        });
        return html `
      ${this._renderToggleTipLabel()}
      <span class="${classes}">
        ${this._renderInnerContent()}
      </span>
    </span>
    `;
    }
    /**
     * A selector that will return the toggletip content.
     */
    static get selectorToggletipContent() {
        return `.${prefix}--popover-content`;
    }
    /**
     * A selector that will return the toggletip caret.
     */
    static get selectorToggletipCaret() {
        return `.${prefix}--popover-caret`;
    }
    /**
     * A selector that will return the trigger element.
     */
    static get selectorToggletipButton() {
        return `.${prefix}--toggletip-button`;
    }
};
CDSToggletip.shadowRootOptions = Object.assign(Object.assign({}, LitElement.shadowRootOptions), { delegatesFocus: true });
CDSToggletip.styles = styles;
__decorate([
    property({ reflect: true })
], CDSToggletip.prototype, "alignment", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSToggletip.prototype, "autoalign", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSToggletip.prototype, "open", void 0);
__decorate([
    HostListener('keydown')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSToggletip.prototype, "_handleKeydown", void 0);
__decorate([
    HostListener('focusout')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSToggletip.prototype, "_handleFocusOut", null);
CDSToggletip = CDSToggletip_1 = __decorate([
    carbonElement(`${prefix}-toggletip`)
], CDSToggletip);
var CDSToggleTip = CDSToggletip;

export { CDSToggleTip as default };
//# sourceMappingURL=toggletip.js.map
