/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { classMap } from 'lit/directives/class-map.js';
import { html } from 'lit';
import { query, property } from 'lit/decorators.js';
import { ifDefined } from 'lit/directives/if-defined.js';
import { prefix } from '../../globals/settings.js';
import CDSCheckbox from '../checkbox/checkbox.js';
import { TOGGLE_SIZE } from './defs.js';
export { TOGGLE_SIZE } from './defs.js';
import styles from './toggle.scss.js';
import HostListener from '../../globals/decorators/host-listener.js';
import HostListenerMixin from '../../globals/mixins/host-listener.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Basic toggle.
 *
 * @element cds-toggle
 * @slot label-text - The label text.
 * @slot checked-text - The text for the checked state.
 * @slot unchecked-text - The text for the unchecked state.
 * @fires cds-toggle-changed - The custom event fired after this changebox changes its checked state.
 */
let CDSToggle = class CDSToggle extends HostListenerMixin(CDSCheckbox) {
    constructor() {
        super(...arguments);
        /**
         * Handles `keydown` event on the toggle button.
         */
        this._handleKeydown = async (event) => {
            const { key } = event;
            if (key === ' ') {
                this._handleChange();
            }
        };
        /**
         * The text for the checked state.
         */
        this.labelA = '';
        /**
         * Hide label text.
         */
        this.hideLabel = false;
        /**
         * Read only boolean.
         */
        this.readOnly = false;
        /**
         * Toggle size.
         */
        this.size = TOGGLE_SIZE.REGULAR;
        /**
         * The text for the unchecked state.
         */
        this.labelB = '';
    }
    /**
     * Handles `click` event on the `<button>` in the shadow DOM.
     */
    _handleChange() {
        const { checked, indeterminate } = this._checkboxNode;
        if (this.disabled || this.readOnly) {
            return;
        }
        this.checked = !checked;
        this.indeterminate = indeterminate;
        const { eventChange } = this.constructor;
        this.dispatchEvent(new CustomEvent(eventChange, {
            bubbles: true,
            composed: true,
            detail: {
                indeterminate,
            },
        }));
    }
    _renderCheckmark() {
        if (this.size !== TOGGLE_SIZE.SMALL) {
            return undefined;
        }
        return html `
      <svg
        class="${prefix}--toggle__check"
        width="6px"
        height="5px"
        viewBox="0 0 6 5">
        <path d="M2.2 2.7L5 0 6 1 2.2 5 0 2.7 1 1.5z" />
      </svg>
    `;
    }
    render() {
        const { checked, disabled, labelText, hideLabel, id, name, size, labelA, labelB, value, _handleChange: handleChange, } = this;
        const inputClasses = classMap({
            [`${prefix}--toggle__appearance`]: true,
            [`${prefix}--toggle__appearance--${size}`]: size,
        });
        const toggleClasses = classMap({
            [`${prefix}--toggle__switch`]: true,
            [`${prefix}--toggle__switch--checked`]: checked,
        });
        const stateText = checked ? labelA : labelB;
        return html `
      <button
        class="${prefix}--toggle__button"
        role="switch"
        type="button"
        aria-checked=${checked}
        aria-lable=${labelText}
        .checked="${checked}"
        name="${ifDefined(name)}"
        value="${ifDefined(value)}"
        ?disabled=${disabled}
        id="${id}"></button>
      <label for="${id}" class="${prefix}--toggle__label">
        <span class="${prefix}--toggle__label-text">${labelText}</span>
        <div class="${inputClasses}">
          <div class="${toggleClasses}" @click=${handleChange}>
            ${this._renderCheckmark()}
          </div>
          <span
            ?hidden="${hideLabel}"
            class="${prefix}--toggle__text"
            aria-hidden="true"
            >${stateText}</span
          >
        </div>
      </label>
    `;
    }
    /**
     * The name of the custom event fired after this changebox changes its checked state.
     */
    static get eventChange() {
        return `${prefix}-toggle-changed`;
    }
};
CDSToggle.styles = styles;
__decorate([
    query('button')
], CDSToggle.prototype, "_checkboxNode", void 0);
__decorate([
    HostListener('keydown')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSToggle.prototype, "_handleKeydown", void 0);
__decorate([
    property({ attribute: 'label-a' })
], CDSToggle.prototype, "labelA", void 0);
__decorate([
    property({ reflect: true, type: Boolean })
], CDSToggle.prototype, "hideLabel", void 0);
__decorate([
    property({ reflect: true, attribute: 'read-only', type: Boolean })
], CDSToggle.prototype, "readOnly", void 0);
__decorate([
    property({ reflect: true })
], CDSToggle.prototype, "size", void 0);
__decorate([
    property({ attribute: 'label-b' })
], CDSToggle.prototype, "labelB", void 0);
CDSToggle = __decorate([
    carbonElement(`${prefix}-toggle`)
], CDSToggle);
var CDSToggle$1 = CDSToggle;

export { CDSToggle$1 as default };
//# sourceMappingURL=toggle.js.map
