/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { classMap } from 'lit/directives/class-map.js';
import { LitElement, html } from 'lit';
import { property } from 'lit/decorators.js';
import Close20 from '../../icons/close/20.js';
import Menu20 from '../../icons/menu/20.js';
import { ifDefined } from 'lit/directives/if-defined.js';
import { prefix } from '../../globals/settings.js';
import FocusMixin from '../../globals/mixins/focus.js';
import './side-nav.js';
import styles from './header.scss.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';
import { SIDE_NAV_COLLAPSE_MODE } from './defs.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2023
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * The trigger button for side nav in header nav.
 *
 * @element cds-header-menu-button
 * @csspart button The button.
 * @csspart toggle-icon The toggle icon.
 * @fires cds-header-menu-button-toggled - The custom event fired after this header menu button is toggled upon a user gesture.
 */
let CDSHeaderMenuButton = class CDSHeaderMenuButton extends FocusMixin(LitElement) {
    constructor() {
        super(...arguments);
        /**
         * `true` if the button should represent its active state.
         */
        this.active = false;
        /**
         * The `aria-label` attribute for the button in its active state.
         */
        this.buttonLabelActive = 'Close navigation menu';
        /**
         * The `aria-label` attribute for the button in its inactive state.
         */
        this.buttonLabelInactive = 'Open navigation menu';
        /**
         * Collapse mode of the side nav.
         */
        this.collapseMode = SIDE_NAV_COLLAPSE_MODE.RESPONSIVE;
        /**
         * `true` if the button should be disabled.
         */
        this.disabled = false;
        /**
         * If `true` will style the side nav to sit below the header
         */
        this.isNotChildOfHeader = false;
    }
    _handleClick() {
        const active = !this.active;
        this.active = active;
        this.dispatchEvent(new CustomEvent(this.constructor.eventToggle, {
            bubbles: true,
            cancelable: true,
            composed: true,
            detail: {
                active,
            },
        }));
    }
    render() {
        const { active, buttonLabelActive, buttonLabelInactive, disabled, _handleClick: handleClick, } = this;
        const buttonLabel = active ? buttonLabelActive : buttonLabelInactive;
        const classes = classMap({
            [`${prefix}--header__action`]: true,
            [`${prefix}--header__menu-trigger`]: true,
            [`${prefix}--header__menu-toggle`]: true,
            [`${prefix}--header__action--active`]: active,
        });
        return html `
      <button
        part="button"
        class="${classes}"
        ?disabled=${disabled}
        aria-label="${ifDefined(buttonLabel)}"
        @click=${handleClick}>
        ${(active ? Close20 : Menu20)({ slot: 'toggle-icon' })}
      </button>
    `;
    }
    /**
     * The name of the custom event fired after this header menu button is toggled upon a user gesture.
     */
    static get eventToggle() {
        return `${prefix}-header-menu-button-toggled`;
    }
};
CDSHeaderMenuButton.shadowRootOptions = Object.assign(Object.assign({}, LitElement.shadowRootOptions), { delegatesFocus: true });
CDSHeaderMenuButton.styles = styles; // `styles` here is a `CSSResult` generated by custom Vite loader
__decorate([
    property({ type: Boolean, reflect: true })
], CDSHeaderMenuButton.prototype, "active", void 0);
__decorate([
    property({ attribute: 'button-label-active' })
], CDSHeaderMenuButton.prototype, "buttonLabelActive", void 0);
__decorate([
    property({ attribute: 'button-label-inactive' })
], CDSHeaderMenuButton.prototype, "buttonLabelInactive", void 0);
__decorate([
    property({ reflect: true, attribute: 'collapse-mode' })
], CDSHeaderMenuButton.prototype, "collapseMode", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSHeaderMenuButton.prototype, "disabled", void 0);
__decorate([
    property({
        type: Boolean,
        attribute: 'is-not-child-of-header',
    })
], CDSHeaderMenuButton.prototype, "isNotChildOfHeader", void 0);
CDSHeaderMenuButton = __decorate([
    carbonElement(`${prefix}-header-menu-button`)
], CDSHeaderMenuButton);
var CDSHeaderMenuButton$1 = CDSHeaderMenuButton;

export { CDSHeaderMenuButton$1 as default };
//# sourceMappingURL=header-menu-button.js.map
