/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { ifDefined } from 'lit/directives/if-defined.js';
import { html, LitElement } from 'lit';
import { query, property } from 'lit/decorators.js';
import { classMap } from 'lit/directives/class-map.js';
import ChevronDown16 from '../../icons/chevron--down/16.js';
import { prefix } from '../../globals/settings.js';
import FocusMixin from '../../globals/mixins/focus.js';
import HostListenerMixin from '../../globals/mixins/host-listener.js';
import HostListener from '../../globals/decorators/host-listener.js';
import { forEach } from '../../globals/internal/collection-helpers.js';
import styles from './header.scss.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Header menu.
 *
 * @element cds-header-menu
 * @csspart trigger The trigger button.
 * @csspart trigger-icon The trigger button icon.
 * @csspart menu-body The menu body.
 */
let CDSHeaderMenu = class CDSHeaderMenu extends HostListenerMixin(FocusMixin(LitElement)) {
    constructor() {
        super(...arguments);
        /**
         * keeps track if header menu has any active submenus
         */
        this._hasActiveChildren = false;
        /**
         * `true` if the menu should be expanded.
         */
        this.expanded = false;
        /**
         * Applies selected styles to the item if a user sets this to true and `aria-current !== 'page'`.
         */
        this.isActive = false;
        /**
         * The content of the trigger button.
         */
        this.triggerContent = '';
    }
    /**
     * Handles `click` event handler on this element.
     */
    _handleClick() {
        this._handleUserInitiatedToggle();
    }
    /**
     * Handler for the `keydown` event on the trigger button.
     */
    _handleKeydownTrigger({ key }) {
        if (key === 'Esc' || key === 'Escape') {
            this._handleUserInitiatedToggle(false);
        }
    }
    /**
     * Handles user-initiated toggling the open state.
     *
     * @param [force] If specified, forces the open state to the given one.
     */
    _handleUserInitiatedToggle(force = !this.expanded) {
        this.expanded = force;
        if (!force) {
            this._topMenuItem.focus();
        }
    }
    /**
     * Handles `blur` event handler on this element.
     */
    _handleBlur({ relatedTarget }) {
        if (!this.contains(relatedTarget)) {
            this.expanded = false;
        }
    }
    connectedCallback() {
        if (!this.hasAttribute('role')) {
            this.setAttribute('role', 'listitem');
        }
        const { selectorItem } = this.constructor;
        forEach(this.querySelectorAll(selectorItem), (elem) => {
            if (elem.isActive === true) {
                this._hasActiveChildren = true;
            }
        });
        super.connectedCallback();
    }
    updated(changedProperties) {
        if (changedProperties.has('expanded')) {
            const { selectorItem } = this.constructor;
            const { expanded } = this;
            forEach(this.querySelectorAll(selectorItem), (elem) => {
                elem.tabIndex = expanded ? 0 : -1;
            });
        }
    }
    render() {
        const { expanded, isActive, triggerContent, menuLabel, _hasActiveChildren, _handleClick: handleClick, } = this;
        const linkClasses = classMap({
            [`${prefix}--header__menu-item`]: true,
            [`${prefix}--header__menu-title`]: true,
            [`${prefix}--header__menu-item--current`]: isActive || (_hasActiveChildren && !expanded),
        });
        return html `
      <a
        part="trigger"
        role="button"
        tabindex="0"
        class="${linkClasses}"
        href="javascript:void 0"
        aria-haspopup="menu"
        aria-expanded="${String(Boolean(expanded))}"
        @click=${handleClick}>
        ${triggerContent}${ChevronDown16({
            part: 'trigger-icon',
            class: `${prefix}--header__menu-arrow`,
        })}
      </a>
      <ul
        part="menu-body"
        class="${prefix}--header__menu"
        aria-label="${ifDefined(menuLabel)}">
        <slot></slot>
      </ul>
    `;
    }
    /**
     * A selector that will return the menu items.
     */
    static get selectorItem() {
        return `${prefix}-header-menu-item`;
    }
};
CDSHeaderMenu.shadowRootOptions = Object.assign(Object.assign({}, LitElement.shadowRootOptions), { delegatesFocus: true });
CDSHeaderMenu.styles = styles; // `styles` here is a `CSSResult` generated by custom Vite loader
__decorate([
    query('[part="trigger"]')
], CDSHeaderMenu.prototype, "_topMenuItem", void 0);
__decorate([
    HostListener('keydown')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSHeaderMenu.prototype, "_handleKeydownTrigger", null);
__decorate([
    HostListener('focusout')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSHeaderMenu.prototype, "_handleBlur", null);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSHeaderMenu.prototype, "expanded", void 0);
__decorate([
    property({ type: Boolean, attribute: 'is-active', reflect: true })
], CDSHeaderMenu.prototype, "isActive", void 0);
__decorate([
    property({ attribute: 'trigger-content' })
], CDSHeaderMenu.prototype, "triggerContent", void 0);
__decorate([
    property({ attribute: 'menu-label' })
], CDSHeaderMenu.prototype, "menuLabel", void 0);
CDSHeaderMenu = __decorate([
    carbonElement(`${prefix}-header-menu`)
], CDSHeaderMenu);
var CDSHeaderMenu$1 = CDSHeaderMenu;

export { CDSHeaderMenu$1 as default };
//# sourceMappingURL=header-menu.js.map
