/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { LitElement, html } from 'lit';
import { query, property } from 'lit/decorators.js';
import ChevronDown20 from '../../icons/chevron--down/20.js';
import { prefix } from '../../globals/settings.js';
import { forEach } from '../../globals/internal/collection-helpers.js';
import FocusMixin from '../../globals/mixins/focus.js';
import styles from './side-nav.scss.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Side nav menu.
 *
 * @element cds-side-nav-menu
 * @fires cds-side-nav-menu-beingtoggled
 *   The name of the custom event fired before this side nav menu is being toggled upon a user gesture.
 *   Cancellation of this event stops the user-initiated action of toggling this side nav menu.
 * @fires cds-side-nav-menu-toggled
 *   The name of the custom event fired after this side nav menu is toggled upon a user gesture.
 * @slot title-icon - The icon.
 * @csspart expando The expando.
 * @csspart expando-icon-container The expando icon container.
 * @csspart expando-icon The expando icon.
 * @csspart title The title.
 * @csspart title-icon-container The title icon container.
 * @csspart menu-body The menu body.
 */
let CDSSideNavMenu = class CDSSideNavMenu extends FocusMixin(LitElement) {
    constructor() {
        super(...arguments);
        /**
         * `true` if this menu has an icon.
         */
        this._hasIcon = false;
        /**
         * `true` if the menu has active menu item.
         */
        this.active = false;
        /**
         * `true` if the menu should be open.
         */
        this.expanded = false;
        /**
         * Specify if this is a large variation of the side nav menu
         */
        this.large = false;
        /**
         * `true` if the menu should be forced collapsed upon side nav's expanded state.
         */
        this.forceCollapsed = false;
        /**
         * The title text.
         */
        this.title = '';
    }
    /**
     * Handles user-initiated toggle request of this side nav menu.
     *
     * @param expanded The new expanded state.
     */
    _handleUserInitiatedToggle(expanded = !this.expanded) {
        const { eventBeforeToggle, eventToggle } = this
            .constructor;
        const init = {
            bubbles: true,
            cancelable: true,
            composed: true,
            detail: {
                expanded,
            },
        };
        if (this.dispatchEvent(new CustomEvent(eventBeforeToggle, init))) {
            this.expanded = expanded;
            this.dispatchEvent(new CustomEvent(eventToggle, init));
        }
    }
    /**
     * Handler for the `click` event on the expando button.
     */
    _handleClickExpando() {
        this._handleUserInitiatedToggle();
    }
    /**
     * Handles `slotchange` event on the non-named `<slot>`.
     */
    _handleSlotChange({ target }) {
        const { _hasIcon: hasIcon } = this;
        forEach(target.assignedNodes(), (item) => {
            if (item.nodeType === Node.ELEMENT_NODE) {
                item.toggleAttribute(this.constructor.attribItemHasIcon, hasIcon);
            }
        });
    }
    /**
     * Handles `slotchange` event on the `<slot>` for the title icon.
     */
    _handleSlotChangeTitleIcon({ target }) {
        var _a;
        const constructor = this.constructor;
        const hasIcon = target.assignedNodes().length > 0;
        this._hasIcon = hasIcon;
        (_a = this._titleIconContainerNode) === null || _a === void 0 ? void 0 : _a.toggleAttribute('hidden', !hasIcon);
        forEach(this.querySelectorAll(constructor.selectorItem), (item) => {
            item.toggleAttribute(constructor.attribItemHasIcon, hasIcon);
        });
    }
    connectedCallback() {
        if (!this.hasAttribute('role')) {
            this.setAttribute('role', 'listitem');
        }
        super.connectedCallback();
    }
    updated(changedProperties) {
        if (changedProperties.has('expanded')) {
            const { selectorItem } = this.constructor;
            const { expanded } = this;
            forEach(this.querySelectorAll(selectorItem), (elem) => {
                elem.tabIndex = expanded ? 0 : -1;
            });
        }
    }
    render() {
        const { expanded, forceCollapsed, title, _handleClickExpando: handleClickExpando, _handleSlotChange: handleSlotChange, _handleSlotChangeTitleIcon: handleSlotChangeTitleIcon, } = this;
        return html `
      <button
        type="button"
        part="expando"
        aria-haspopup="true"
        aria-expanded="${String(Boolean(expanded && !forceCollapsed))}"
        class="${prefix}--side-nav__submenu"
        @click=${handleClickExpando}>
        <div
          id="title-icon-container"
          part="title-icon-container"
          hidden
          class="${prefix}--side-nav__icon">
          <slot
            name="title-icon"
            @slotchange=${handleSlotChangeTitleIcon}></slot>
        </div>
        <span part="title" class="${prefix}--side-nav__submenu-title"
          >${title}</span
        >
        <div
          part="expando-icon-container"
          class="${prefix}--side-nav__icon ${prefix}--side-nav__icon--small ${prefix}--side-nav__submenu-chevron">
          ${ChevronDown20({ part: 'expando-icon' })}
        </div>
      </button>
      <ul part="menu-body" class="${prefix}--side-nav__menu">
        <slot @slotchange=${handleSlotChange}></slot>
      </ul>
    `;
    }
    /**
     * A selector that will return the menu items.
     */
    static get selectorItem() {
        return `${prefix}-side-nav-menu-item`;
    }
    /**
     * The name of the custom event fired before this side nav menu is being toggled upon a user gesture.
     * Cancellation of this event stops the user-initiated action of toggling this side nav menu.
     */
    static get eventBeforeToggle() {
        return `${prefix}-side-nav-menu-beingtoggled`;
    }
    /**
     * The name of the custom event fired after this side nav menu is toggled upon a user gesture.
     */
    static get eventToggle() {
        return `${prefix}-side-nav-menu-toggled`;
    }
};
/**
 * The attribute name of the menu items, that is set if this menu has an icon.
 */
CDSSideNavMenu.attribItemHasIcon = 'parent-has-icon';
CDSSideNavMenu.shadowRootOptions = Object.assign(Object.assign({}, LitElement.shadowRootOptions), { delegatesFocus: true });
CDSSideNavMenu.styles = styles; // `styles` here is a `CSSResult` generated by custom Vite loader
__decorate([
    query('#title-icon-container')
], CDSSideNavMenu.prototype, "_titleIconContainerNode", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSSideNavMenu.prototype, "active", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSSideNavMenu.prototype, "expanded", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSSideNavMenu.prototype, "large", void 0);
__decorate([
    property({ type: Boolean, reflect: true, attribute: 'force-collapsed' })
], CDSSideNavMenu.prototype, "forceCollapsed", void 0);
__decorate([
    property()
], CDSSideNavMenu.prototype, "title", void 0);
CDSSideNavMenu = __decorate([
    carbonElement(`${prefix}-side-nav-menu`)
], CDSSideNavMenu);
var CDSSideNavMenu$1 = CDSSideNavMenu;

export { CDSSideNavMenu$1 as default };
//# sourceMappingURL=side-nav-menu.js.map
