/**
 * Copyright IBM Corp. 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { __decorate } from 'tslib';
import { html, LitElement } from 'lit';
import { property } from 'lit/decorators.js';
import { classMap } from 'lit/directives/class-map.js';
import { prefix } from '../../globals/settings.js';
import HostListenerMixin from '../../globals/mixins/host-listener.js';
import HostListener from '../../globals/decorators/host-listener.js';
import { forEach } from '../../globals/internal/collection-helpers.js';
import { SIDE_NAV_COLLAPSE_MODE } from './defs.js';
export { SIDE_NAV_COLLAPSE_MODE, SIDE_NAV_USAGE_MODE } from './defs.js';
import styles from './side-nav.scss.js';
import { carbonElement } from '../../globals/decorators/carbon-element.js';

/**
 * @license
 *
 * Copyright IBM Corp. 2019, 2024
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
/**
 * Side nav.
 *
 * @element cds-side-nav
 * @fires cds-header-menu-button-toggled
 *   The name of the custom event fired after the header menu button in the document is toggled upon a user gesture.
 */
let CDSSideNav = class CDSSideNav extends HostListenerMixin(LitElement) {
    constructor() {
        super(...arguments);
        /**
         * `true` if this side nav is hovered.
         */
        this._hovered = false;
        /**
         * The handle for `transitionend` event listener.
         */
        this._hTransition = null;
        /**
         * Handles `${prefix}-header-menu-button-toggle` event on the document.
         */
        this._handleButtonToggle = async (event) => {
            this.expanded = event.detail.active;
        };
        /**
         * Collapse mode of the side nav.
         */
        this.collapseMode = SIDE_NAV_COLLAPSE_MODE.RESPONSIVE;
        /**
         * `true` to expand the side nav.
         */
        this.expanded = false;
        /**
         * If `true` will style the side nav to sit below the header
         */
        this.isNotChildOfHeader = false;
        /**
         * Specify if the side-nav will be persistent above the lg breakpoint
         */
        this.isNotPersistent = false;
    }
    /**
     * Cleans the handle for `transitionend` event listener.
     */
    _cleanHTransition() {
        if (this._hTransition) {
            this._hTransition = this._hTransition.release();
        }
    }
    /**
     * Force child side nav menus collapsed upon the hover/expanded state of this side nav.
     */
    _updatedSideNavMenuForceCollapsedState() {
        const { expanded, _hovered: hovered } = this;
        forEach(this.querySelectorAll(this.constructor.selectorMenu), (item) => {
            item.forceCollapsed = !expanded && !hovered;
        });
    }
    connectedCallback() {
        if (!this.hasAttribute('role')) {
            this.setAttribute('role', 'navigation');
        }
        super.connectedCallback();
    }
    disconnectedCallback() {
        this._cleanHTransition();
        super.disconnectedCallback();
    }
    updated(changedProperties) {
        var _a;
        const doc = this.getRootNode();
        if (changedProperties.has('collapseMode')) {
            forEach(doc.querySelectorAll(this.constructor.selectorButtonToggle), (item) => {
                item.collapseMode = this.collapseMode;
            });
        }
        if (changedProperties.has('expanded')) {
            const headerItems = doc.querySelectorAll(this.constructor.selectorHeaderItems);
            forEach(doc.querySelectorAll(this.constructor.selectorButtonToggle), (item) => {
                item.active = this.expanded;
            });
            if (this.expanded) {
                forEach(headerItems, (item) => {
                    item.setAttribute('tabindex', '-1');
                });
                (_a = this.querySelector(this.constructor.selectorNavItems)) === null || _a === void 0 ? void 0 : _a.focus();
            }
            else {
                forEach(headerItems, (item) => {
                    item.removeAttribute('tabindex');
                });
            }
        }
        if (changedProperties.has('isNotChildOfHeader')) {
            forEach(doc.querySelectorAll(this.constructor.selectorButtonToggle), (item) => {
                item.isNotChildOfHeader =
                    this.isNotChildOfHeader;
            });
        }
    }
    /**
     * Handles `blur` event handler on this element.
     *
     * @param event The event.
     * @param event.relatedTarget The event relatedTarget.
     */
    _handleFocusOut({ relatedTarget }) {
        const { collapseMode } = this;
        if (collapseMode !== SIDE_NAV_COLLAPSE_MODE.FIXED) {
            if (!this.contains(relatedTarget)) {
                this.expanded = false;
            }
        }
    }
    /**
     * Handles `focus` event handler on this element.
     *
     */
    _handleFocusIn() {
        const { collapseMode } = this;
        if (collapseMode !== SIDE_NAV_COLLAPSE_MODE.FIXED) {
            this.expanded = true;
        }
    }
    /**
     * Handles the `mouseover` event for the side nav in rail mode.
     *
     */
    _handleNavMouseOver() {
        const { collapseMode } = this;
        if (collapseMode === SIDE_NAV_COLLAPSE_MODE.RAIL) {
            this.expanded = true;
            this._hovered = true;
            this._updatedSideNavMenuForceCollapsedState();
        }
    }
    /**
     * Handles the `mouseout` event for the side nav in rail mode.
     *
     */
    _handleNavMouseOut() {
        const { collapseMode } = this;
        if (collapseMode === SIDE_NAV_COLLAPSE_MODE.RAIL) {
            this.expanded = false;
            this._hovered = false;
            this._updatedSideNavMenuForceCollapsedState();
        }
    }
    /**
     * Handles the `click` event for the side nav overlay.
     *
     */
    _onOverlayClick() {
        this.expanded = false;
    }
    render() {
        const { collapseMode, expanded, isNotChildOfHeader, isNotPersistent } = this;
        const classes = classMap({
            [`${prefix}--side-nav__navigation`]: true,
            [`${prefix}--side-nav`]: true,
            [`${prefix}--side-nav--expanded`]: expanded,
            [`${prefix}--side-nav--collapsed`]: !expanded && collapseMode === SIDE_NAV_COLLAPSE_MODE.FIXED,
            [`${prefix}--side-nav--rail`]: collapseMode === SIDE_NAV_COLLAPSE_MODE.RAIL,
            [`${prefix}--side-nav--ux`]: !isNotChildOfHeader,
            [`${prefix}--side-nav--hidden`]: isNotPersistent,
        });
        const overlayClasses = classMap({
            [`${prefix}--side-nav__overlay`]: true,
            [`${prefix}--side-nav__overlay-active`]: expanded,
        });
        return html `${this.collapseMode === SIDE_NAV_COLLAPSE_MODE.FIXED
            ? null
            : html `<div
            class="${overlayClasses}"
            @click=${this._onOverlayClick}></div>`}
      <div
        class="${classes}"
        @mouseover="${this._handleNavMouseOver}"
        @mouseout="${this._handleNavMouseOut}">
        <slot></slot>
      </div>`;
    }
    /**
     * A selector that will return the toggle buttons.
     */
    static get selectorButtonToggle() {
        return `${prefix}-header-menu-button`;
    }
    /**
     * A selector that will return the header name + global action elements.
     */
    static get selectorHeaderItems() {
        return `${prefix}-header-name, ${prefix}-header-global-action`;
    }
    /**
     * A selector that will return side nav focusable items.
     */
    static get selectorNavItems() {
        return `${prefix}-side-nav-menu, ${prefix}-side-nav-menu-item, ${prefix}-side-nav-link`;
    }
    /**
     * A selector that will return side nav menus.
     */
    static get selectorMenu() {
        return `${prefix}-side-nav-menu`;
    }
    /**
     * The name of the custom event fired after the header menu button in the document is toggled upon a user gesture.
     */
    static get eventButtonToggle() {
        return `${prefix}-header-menu-button-toggled`;
    }
};
CDSSideNav.styles = styles; // `styles` here is a `CSSResult` generated by custom Vite loader
__decorate([
    HostListener('parentRoot:eventButtonToggle')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSSideNav.prototype, "_handleButtonToggle", void 0);
__decorate([
    property({ reflect: true, attribute: 'collapse-mode' })
], CDSSideNav.prototype, "collapseMode", void 0);
__decorate([
    property({ type: Boolean, reflect: true })
], CDSSideNav.prototype, "expanded", void 0);
__decorate([
    property({
        type: Boolean,
        attribute: 'is-not-child-of-header',
    })
], CDSSideNav.prototype, "isNotChildOfHeader", void 0);
__decorate([
    property({ type: Boolean, reflect: true, attribute: 'is-not-persistent' })
], CDSSideNav.prototype, "isNotPersistent", void 0);
__decorate([
    HostListener('focusout')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSSideNav.prototype, "_handleFocusOut", null);
__decorate([
    HostListener('focusin')
    // @ts-ignore: The decorator refers to this method but TS thinks this method is not referred to
], CDSSideNav.prototype, "_handleFocusIn", null);
CDSSideNav = __decorate([
    carbonElement(`${prefix}-side-nav`)
], CDSSideNav);
var CDSSideNav$1 = CDSSideNav;

export { CDSSideNav$1 as default };
//# sourceMappingURL=side-nav.js.map
