/**
 * chartjs-chart-wordcloud
 * https://github.com/sgratzl/chartjs-chart-wordcloud
 *
 * Copyright (c) 2021 Samuel Gratzl <sam@sgratzl.com>
 */

'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var chart_js = require('chart.js');
var helpers = require('chart.js/helpers');
var layout = require('d3-cloud');

function _interopDefaultLegacy (e) { return e && typeof e === 'object' && 'default' in e ? e : { 'default': e }; }

var layout__default = /*#__PURE__*/_interopDefaultLegacy(layout);

class WordElement extends chart_js.Element {
    static computeRotation(o, rnd) {
        if (o.rotationSteps <= 1) {
            return 0;
        }
        if (o.minRotation === o.maxRotation) {
            return o.minRotation;
        }
        const base = Math.min(o.rotationSteps, Math.floor(rnd() * o.rotationSteps)) / (o.rotationSteps - 1);
        const range = o.maxRotation - o.minRotation;
        return o.minRotation + base * range;
    }
    inRange(mouseX, mouseY) {
        const p = this.getProps(['x', 'y', 'width', 'height', 'scale']);
        if (p.scale <= 0) {
            return false;
        }
        const x = Number.isNaN(mouseX) ? p.x : mouseX;
        const y = Number.isNaN(mouseY) ? p.y : mouseY;
        return x >= p.x - p.width / 2 && x <= p.x + p.width / 2 && y >= p.y - p.height / 2 && y <= p.y + p.height / 2;
    }
    inXRange(mouseX) {
        return this.inRange(mouseX, Number.NaN);
    }
    inYRange(mouseY) {
        return this.inRange(Number.NaN, mouseY);
    }
    getCenterPoint() {
        return this.getProps(['x', 'y']);
    }
    tooltipPosition() {
        return this.getCenterPoint();
    }
    draw(ctx) {
        const { options } = this;
        const props = this.getProps(['x', 'y', 'width', 'height', 'text', 'scale']);
        if (props.scale <= 0) {
            return;
        }
        ctx.save();
        const f = helpers.toFont({ ...options, size: options.size * props.scale });
        ctx.font = f.string;
        ctx.fillStyle = options.color;
        ctx.textAlign = 'center';
        ctx.translate(props.x, props.y);
        ctx.rotate((options.rotate / 180) * Math.PI);
        if (options.strokeStyle) {
            ctx.strokeStyle = options.strokeStyle;
            ctx.strokeText(props.text, 0, 0);
        }
        ctx.fillText(props.text, 0, 0);
        ctx.restore();
    }
}
WordElement.id = 'word';
WordElement.defaults = {
    minRotation: -90,
    maxRotation: 0,
    rotationSteps: 2,
    padding: 1,
    strokeStyle: undefined,
    size: (ctx) => {
        const v = ctx.parsed.y;
        return v;
    },
    hoverColor: '#ababab',
};
WordElement.defaultRoutes = {
    color: 'color',
    family: 'font.family',
    style: 'font.style',
    weight: 'font.weight',
    lineHeight: 'font.lineHeight',
};

function patchController(type, config, controller, elements = [], scales = []) {
    chart_js.registry.addControllers(controller);
    if (Array.isArray(elements)) {
        chart_js.registry.addElements(...elements);
    }
    else {
        chart_js.registry.addElements(elements);
    }
    if (Array.isArray(scales)) {
        chart_js.registry.addScales(...scales);
    }
    else {
        chart_js.registry.addScales(scales);
    }
    const c = config;
    c.type = type;
    return c;
}

function rnd(seed = Date.now()) {
    let s = typeof seed === 'number' ? seed : Array.from(seed).reduce((acc, v) => acc + v.charCodeAt(0), 0);
    return () => {
        s = (s * 9301 + 49297) % 233280;
        return s / 233280;
    };
}
class WordCloudController extends chart_js.DatasetController {
    constructor() {
        super(...arguments);
        this.wordLayout = layout__default["default"]()
            .text((d) => d.text)
            .padding((d) => d.options.padding)
            .rotate((d) => d.options.rotate)
            .font((d) => d.options.family)
            .fontSize((d) => d.options.size)
            .fontStyle((d) => d.options.style)
            .fontWeight((d) => { var _a; return (_a = d.options.weight) !== null && _a !== void 0 ? _a : 1; });
        this.rand = Math.random;
    }
    update(mode) {
        super.update(mode);
        this.rand = rnd(this.chart.id);
        const meta = this._cachedMeta;
        const elems = (meta.data || []);
        this.updateElements(elems, 0, elems.length, mode);
    }
    updateElements(elems, start, count, mode) {
        var _a, _b, _c, _d;
        this.wordLayout.stop();
        const xScale = this._cachedMeta.xScale;
        const yScale = this._cachedMeta.yScale;
        const w = xScale.right - xScale.left;
        const h = yScale.bottom - yScale.top;
        const labels = this.chart.data.labels;
        const words = [];
        for (let i = start; i < start + count; i += 1) {
            const o = this.resolveDataElementOptions(i, mode);
            if (o.rotate == null) {
                o.rotate = WordElement.computeRotation(o, this.rand);
            }
            const properties = {
                options: { ...helpers.toFont(o), ...o },
                x: (_b = (_a = this._cachedMeta.xScale) === null || _a === void 0 ? void 0 : _a.getPixelForDecimal(0.5)) !== null && _b !== void 0 ? _b : 0,
                y: (_d = (_c = this._cachedMeta.yScale) === null || _c === void 0 ? void 0 : _c.getPixelForDecimal(0.5)) !== null && _d !== void 0 ? _d : 0,
                width: 10,
                height: 10,
                scale: 1,
                index: i,
                text: labels[i],
            };
            words.push(properties);
        }
        if (mode === 'reset') {
            words.forEach((tag) => {
                this.updateElement(elems[tag.index], tag.index, tag, mode);
            });
            return;
        }
        this.wordLayout.random(this.rand).words(words);
        const run = (factor = 1, tries = 3) => {
            this.wordLayout
                .size([w * factor, h * factor])
                .on('end', (tags, bounds) => {
                if (tags.length < labels.length) {
                    if (tries > 0) {
                        run(factor * 1.2, tries - 1);
                        return;
                    }
                    console.warn('cannot fit all text elements in three tries');
                }
                const wb = bounds[1].x - bounds[0].x;
                const hb = bounds[1].y - bounds[0].y;
                const dsOptions = this.options;
                const scale = dsOptions.fit ? Math.min(w / wb, h / hb) : 1;
                const indices = new Set(labels.map((_, i) => i));
                tags.forEach((tag) => {
                    indices.delete(tag.index);
                    this.updateElement(elems[tag.index], tag.index, {
                        options: tag.options,
                        scale,
                        x: xScale.left + scale * tag.x + w / 2,
                        y: yScale.top + scale * tag.y + h / 2,
                        width: scale * tag.width,
                        height: scale * tag.height,
                        text: tag.text,
                    }, mode);
                });
                indices.forEach((i) => this.updateElement(elems[i], i, { scale: 0 }, mode));
            })
                .start();
        };
        run();
    }
    draw() {
        const elements = this._cachedMeta.data;
        const { ctx } = this.chart;
        elements.forEach((elem) => elem.draw(ctx));
    }
    getLabelAndValue(index) {
        const r = super.getLabelAndValue(index);
        const labels = this.chart.data.labels;
        r.label = labels[index];
        return r;
    }
}
WordCloudController.id = 'wordCloud';
WordCloudController.defaults = {
    datasets: {
        fit: true,
        animation: {
            colors: {
                properties: ['color', 'strokeStyle'],
            },
            numbers: {
                properties: ['x', 'y', 'size', 'rotate'],
            },
        },
    },
    maintainAspectRatio: false,
    dataElementType: WordElement.id,
};
WordCloudController.overrides = {
    scales: {
        x: {
            type: 'linear',
            min: -1,
            max: 1,
            display: false,
        },
        y: {
            type: 'linear',
            min: -1,
            max: 1,
            display: false,
        },
    },
};
class WordCloudChart extends chart_js.Chart {
    constructor(item, config) {
        super(item, patchController('wordCloud', config, WordCloudController, WordElement));
    }
}
WordCloudChart.id = WordCloudController.id;

exports.WordCloudChart = WordCloudChart;
exports.WordCloudController = WordCloudController;
exports.WordElement = WordElement;
//# sourceMappingURL=index.cjs.js.map
