/**
 * SPDX-FileCopyrightText: © 2019 Liferay, Inc. <https://liferay.com>
 * SPDX-License-Identifier: BSD-3-Clause
 */

import { useCallback, useEffect, useRef, useState } from 'react';
import { ObserverType } from "./types.js";
const delay = fn => {
  return setTimeout(() => {
    fn();
  }, 100);
};
const modalOpenClassName = 'modal-open';
export const useModal = function () {
  let {
    defaultOpen = false,
    onClose
  } = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
  const [open, setOpen] = useState(defaultOpen);
  const [visible, setVisible] = useState([false, false]);
  const timerIdRef = useRef(null);
  const restoreTriggerRef = useRef(null);

  /**
   * Control the close of the modal to create the component's "unmount"
   * animation and call the onClose prop with delay.
   */
  const handleCloseModal = () => {
    document.body.classList.remove(modalOpenClassName);
    setVisible([false, true]);
    timerIdRef.current = delay(() => {
      if (onClose) {
        onClose();
      }
      if (restoreTriggerRef.current) {
        restoreTriggerRef.current.focus();
        restoreTriggerRef.current = null;
      }
      setOpen(false);
      setVisible([false, false]);
    });
  };
  const handleOpenModal = () => {
    document.body.classList.add(modalOpenClassName);
    setOpen(true);
    timerIdRef.current = delay(() => setVisible([true, true]));
  };
  const handleObserverDispatch = (type, payload) => {
    switch (type) {
      case ObserverType.Close:
        handleCloseModal();
        break;
      case ObserverType.Open:
        handleOpenModal();
        break;
      case ObserverType.RestoreFocus:
        restoreTriggerRef.current = payload;
        break;
      default:
        break;
    }
  };
  const onOpenChange = useCallback(value => {
    if (value) {
      handleOpenModal();
    } else {
      handleCloseModal();
    }
  }, []);
  useEffect(() => {
    return () => {
      document.body.classList.remove(modalOpenClassName);
      if (timerIdRef.current) {
        clearTimeout(timerIdRef.current);
      }
    };
  }, []);
  return {
    observer: {
      dispatch: handleObserverDispatch,
      mutation: visible
    },
    onClose: handleCloseModal,
    onOpenChange,
    open
  };
};