var _ = require('lodash');

function CldrRbnfRuleSet(config) {
  _.extend(this, config);
  this.ruleByValue = {};
}

CldrRbnfRuleSet.getSafeRendererName = function(rendererName) {
  return ('render-' + rendererName)
    .replace(/[^\w-]/g, '-')
    .replace(/[-_]+([0-9a-z])/gi, function($0, ch) {
      return ch.toUpperCase();
    })
    .replace('GREEKNUMERALMAJUSCULES', 'GreekNumeralMajuscules');
};

CldrRbnfRuleSet.prototype = {
  toFunctionAst: function() {
    var that = this;

    var isSeenByRuleSetType = {};

    function ruleToExpressionAst(rule) {
      var expressionAsts = [];

      var rbnf = rule.rbnf;

      // "If a rule body begins with an apostrophe, the apostrophe is ignored, but all text after it becomes
      // significant (this is how you can have a rule's rule text begin with whitespace)."
      // -- http://www.icu-project.org/apiref/icu4c/classRuleBasedNumberFormat.html
      rbnf = rbnf.replace(/^'/, '');

      rule.radix = rule.radix || 10;

      function getDivisor() {
        var divisor = 1;
        while (10 * divisor <= parseInt(rule.value, 10)) {
          // Inefficient, but won't suffer from Math.log rounding errors
          divisor *= 10;
        }
        return divisor;
      }

      // Replace is used for tokenization, the return value isn't used:
      rbnf.replace(
        /(?:([<>=])(?:(%%?[\w-]+)|([#,0.]+))?\1)|(?:\[([^\]]+)\])|([\x7f-\uffff:'.\s\w\d-]+|(?:\$\((cardinal|ordinal),([^)]+)\)))/gi,
        function(
          $0,
          specialChar,
          otherFormat,
          decimalFormat,
          optional,
          literal,
          cardinalOrOrdinal,
          dollarRule
        ) {
          // The meanings of the substitution token characters are as follows:
          if (dollarRule) {
            var callAst = {
              type: 'CallExpression',
              callee: {
                type: 'FunctionExpression',
                id: null,
                params: [
                  {
                    type: 'Identifier',
                    name: 'n'
                  }
                ],
                body: {
                  type: 'BlockStatement',
                  body:
                    cardinalOrOrdinal === 'cardinal'
                      ? that.cardinalPluralRuleAst
                      : that.ordinalPluralRuleAst
                }
              },
              arguments: [
                {
                  type: 'Identifier',
                  name: 'n'
                }
              ]
            };
            var objAst = {
              type: 'ObjectExpression',
              properties: []
            };
            dollarRule.split('}').forEach(function(fragment) {
              var pluralCaseAndValue = fragment.split('{');
              if (pluralCaseAndValue.length === 2) {
                objAst.properties.push({
                  type: 'Property',
                  key: {
                    type: 'Literal',
                    value: pluralCaseAndValue[0]
                  },
                  value: {
                    type: 'Literal',
                    value: pluralCaseAndValue[1]
                  }
                });
              }
            });
            expressionAsts.push({
              type: 'MemberExpression',
              object: objAst,
              property: callAst,
              computed: true
            });
          } else if (specialChar) {
            var expr;
            if (specialChar === '<') {
              // <<
              if (/^\d+$/.test(rule.value)) {
                // In normal rule: Divide the number by the rule's divisor and format the quotient
                expr = {
                  type: 'CallExpression',
                  callee: {
                    type: 'MemberExpression',
                    object: {
                      type: 'Identifier',
                      name: 'Math'
                    },
                    property: {
                      type: 'Identifier',
                      name: 'floor'
                    },
                    computed: false
                  },
                  arguments: [
                    {
                      type: 'BinaryExpression',
                      operator: '/',
                      left: {
                        type: 'Identifier',
                        name: 'n'
                      },
                      right: {
                        type: 'Literal',
                        value: getDivisor()
                      }
                    }
                  ]
                };
              } else if (rule.value === '-x') {
                throw new Error('<< not allowed in negative number rule');
              } else {
                // In fraction or master rule: Isolate the number's integral part and format it.
                expr = {
                  type: 'CallExpression',
                  callee: {
                    type: 'MemberExpression',
                    object: {
                      type: 'Identifier',
                      name: 'Math'
                    },
                    property: {
                      type: 'Identifier',
                      name: 'floor'
                    },
                    computed: false
                  },
                  arguments: [
                    {
                      type: 'Identifier',
                      name: 'n'
                    }
                  ]
                };
              }
            } else if (specialChar === '>') {
              // >>
              if (/\./.test(rule.value)) {
                // Fraction or master rule => parseInt(String(n).replace(/\d*\./, ''), 10)
                expr = {
                  type: 'CallExpression',
                  callee: {
                    type: 'Identifier',
                    name: 'parseInt'
                  },
                  arguments: [
                    {
                      type: 'CallExpression',
                      callee: {
                        type: 'MemberExpression',
                        computed: false,
                        object: {
                          type: 'CallExpression',
                          callee: {
                            type: 'Identifier',
                            name: 'String'
                          },
                          arguments: [
                            {
                              type: 'Identifier',
                              name: 'n'
                            }
                          ]
                        },
                        property: {
                          type: 'Identifier',
                          name: 'replace'
                        }
                      },
                      arguments: [
                        {
                          type: 'Literal',
                          value: /\d*\./
                        },
                        {
                          type: 'Literal',
                          value: ''
                        }
                      ]
                    },
                    {
                      type: 'Literal',
                      value: 10
                    }
                  ]
                };
              } else if (rule.value === '-x') {
                expr = {
                  type: 'UnaryExpression',
                  operator: '-',
                  prefix: true,
                  argument: {
                    type: 'Identifier',
                    name: 'n'
                  }
                };
              } else {
                expr = {
                  type: 'BinaryExpression',
                  operator: '%',
                  left: {
                    type: 'Identifier',
                    name: 'n'
                  },
                  right: {
                    type: 'Literal',
                    value: getDivisor()
                  }
                };
              }
            } else if (specialChar === '=') {
              // ==
              expr = {
                type: 'Identifier',
                name: 'n'
              };
            }
            // FIXME: >>> not supported

            // The substitution descriptor (i.e., the text between the token characters) may take one of three forms:
            if (otherFormat) {
              // A rule set name:
              // Perform the mathematical operation on the number, and format the result using the named rule set.
              var otherFormatName = CldrRbnfRuleSet.getSafeRendererName(
                otherFormat
              );
              isSeenByRuleSetType[otherFormatName] = true;
              // Turn into this.<otherFormatName>(<expr>)
              expressionAsts.push({
                type: 'CallExpression',
                callee: {
                  type: 'MemberExpression',
                  object: {
                    type: 'ThisExpression'
                  },
                  property: {
                    type: 'Identifier',
                    name: otherFormatName
                  },
                  computed: false
                },
                arguments: [expr]
              });
            } else if (decimalFormat) {
              // A DecimalFormat pattern:
              // Perform the mathematical operation on the number, and format the result using a DecimalFormat
              // with the specified pattern. The pattern must begin with 0 or #.
              expressionAsts.push({
                type: 'CallExpression',
                callee: {
                  type: 'MemberExpression',
                  object: {
                    type: 'ThisExpression'
                  },
                  property: {
                    type: 'Identifier',
                    name: 'renderNumber'
                  },
                  computed: false
                },
                arguments: [
                  expr,
                  {
                    type: 'Literal',
                    value: decimalFormat
                  }
                ]
              });
            } else {
              // Nothing:
              if (specialChar === '>') {
                // If you omit the substitution descriptor in a >> substitution in a fraction rule, format the result one digit at a time using the rule set containing the current rule.
                expressionAsts.push({
                  type: 'CallExpression',
                  callee: {
                    type: 'MemberExpression',
                    object: {
                      type: 'ThisExpression'
                    },
                    property: {
                      type: 'Identifier',
                      name: that.type
                    },
                    computed: false
                  },
                  arguments: [expr]
                });
              } else if (specialChar === '<') {
                // If you omit the substitution descriptor in a << substitution in a rule in a fraction rule set, format the result using the default rule set for this renderer.
                // FIXME: Should be the default rule set for this renderer!
                expressionAsts.push({
                  type: 'CallExpression',
                  callee: {
                    type: 'MemberExpression',
                    object: {
                      type: 'ThisExpression'
                    },
                    property: {
                      type: 'Identifier',
                      name: that.type
                    },
                    computed: false
                  },
                  arguments: [expr]
                });
              } else {
                throw new Error('== not supported!');
              }
            }
          } else if (optional) {
            // [ ... ]
            var optionalRuleExpressionAst = ruleToExpressionAst({
              radix: rule.radix,
              rbnf: optional,
              value: rule.value
            });
            expressionAsts.push({
              type: 'ConditionalExpression',
              test: {
                type: 'BinaryExpression',
                operator: '===',
                left: {
                  type: 'Identifier',
                  name: 'n'
                },
                right: {
                  type: 'Literal',
                  value: parseInt(rule.value, 10)
                }
              },
              consequent: {
                type: 'Literal',
                value: ''
              },
              alternate: optionalRuleExpressionAst
            });
          } else if (literal) {
            expressionAsts.push({
              type: 'Literal',
              value: literal
            });
          } else {
            throw new Error('Unknown token in ' + rule.rbnf);
          }
        }
      );
      if (expressionAsts.length === 0) {
        expressionAsts = [
          {
            type: 'Literal',
            value: ''
          }
        ];
      }
      var expressionAst = expressionAsts.shift();
      while (expressionAsts.length > 0) {
        expressionAst = {
          type: 'BinaryExpression',
          operator: '+',
          left: expressionAst,
          right: expressionAsts.shift()
        };
      }
      return expressionAst;
    }

    function conditionToStatementAst(conditionAst, rule) {
      return {
        type: 'IfStatement',
        test: conditionAst,
        consequent: {
          type: 'ReturnStatement',
          argument: ruleToExpressionAst(rule)
        }
      };
    }

    var statementAsts = [];
    if (this.ruleByValue['x.0'] || this.ruleByValue['x.x']) {
      // var isFractional = n !== Math.floor(n);
      statementAsts.push({
        type: 'VariableDeclaration',
        kind: 'var',
        declarations: [
          {
            type: 'VariableDeclarator',
            id: {
              type: 'Identifier',
              name: 'isFractional'
            },
            init: {
              type: 'BinaryExpression',
              operator: '!==',
              left: {
                type: 'Identifier',
                name: 'n'
              },
              right: {
                type: 'CallExpression',
                callee: {
                  type: 'MemberExpression',
                  computed: false,
                  object: {
                    type: 'Identifier',
                    name: 'Math'
                  },
                  property: {
                    type: 'Identifier',
                    name: 'floor'
                  }
                },
                arguments: [
                  {
                    type: 'Identifier',
                    name: 'n'
                  }
                ]
              }
            }
          }
        ]
      });
    }
    if (this.ruleByValue['x.0']) {
      statementAsts.push(
        conditionToStatementAst(
          {
            type: 'Identifier',
            name: 'isFractional'
          },
          this.ruleByValue['x.0']
        )
      );
    }
    if (this.ruleByValue['-x']) {
      statementAsts.push(
        conditionToStatementAst(
          {
            type: 'BinaryExpression',
            operator: '<',
            left: {
              type: 'Identifier',
              name: 'n'
            },
            right: {
              type: 'Literal',
              value: 0
            }
          },
          this.ruleByValue['-x']
        )
      );
    }
    if (this.ruleByValue['x.x']) {
      statementAsts.push(
        conditionToStatementAst(
          {
            type: 'LogicalExpression',
            operator: '&&',
            left: {
              type: 'Identifier',
              name: 'isFractional'
            },
            right: {
              type: 'BinaryExpression',
              operator: '>',
              left: {
                type: 'Identifier',
                name: 'n'
              },
              right: {
                type: 'Literal',
                value: 1
              }
            }
          },
          this.ruleByValue['x.x']
        )
      );
    }
    if (this.ruleByValue['0.x']) {
      statementAsts.push(
        conditionToStatementAst(
          {
            type: 'LogicalExpression',
            operator: '&&',
            left: {
              type: 'BinaryExpression',
              operator: '>',
              left: {
                type: 'Identifier',
                name: 'n'
              },
              right: {
                type: 'Literal',
                value: 0
              }
            },
            right: {
              type: 'BinaryExpression',
              operator: '<',
              left: {
                type: 'Identifier',
                name: 'n'
              },
              right: {
                type: 'Literal',
                value: 1
              }
            }
          },
          this.ruleByValue['0.x']
        )
      );
    }

    Object.keys(this.ruleByValue)
      .filter(function(value) {
        return /^\d+$/.test(value);
      })
      .map(function(value) {
        return parseInt(value, 10);
      })
      .sort(function(a, b) {
        return b - a;
      })
      .forEach(function(numericalValue) {
        statementAsts.push(
          conditionToStatementAst(
            {
              type: 'BinaryExpression',
              operator: '>=',
              left: {
                type: 'Identifier',
                name: 'n'
              },
              right: {
                type: 'Literal',
                value: numericalValue
              }
            },
            this.ruleByValue[numericalValue]
          )
        );
      }, this);

    return {
      functionAst: {
        type: 'FunctionExpression',
        params: [
          {
            type: 'Identifier',
            name: 'n'
          }
        ],
        body: {
          type: 'BlockStatement',
          body: statementAsts
        }
      },
      dependencies: Object.keys(isSeenByRuleSetType)
    };
  }
};

module.exports = CldrRbnfRuleSet;
