"use strict";

var _codemirror = _interopRequireDefault(require("codemirror"));

var _graphql = require("graphql");

var _jsonParse = _interopRequireDefault(require("../utils/jsonParse"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _slicedToArray(arr, i) { return _arrayWithHoles(arr) || _iterableToArrayLimit(arr, i) || _nonIterableRest(); }

function _nonIterableRest() { throw new TypeError("Invalid attempt to destructure non-iterable instance"); }

function _iterableToArrayLimit(arr, i) { if (!(Symbol.iterator in Object(arr) || Object.prototype.toString.call(arr) === "[object Arguments]")) { return; } var _arr = []; var _n = true; var _d = false; var _e = undefined; try { for (var _i = arr[Symbol.iterator](), _s; !(_n = (_s = _i.next()).done); _n = true) { _arr.push(_s.value); if (i && _arr.length === i) break; } } catch (err) { _d = true; _e = err; } finally { try { if (!_n && _i["return"] != null) _i["return"](); } finally { if (_d) throw _e; } } return _arr; }

function _arrayWithHoles(arr) { if (Array.isArray(arr)) return arr; }

/**
 * Registers a "lint" helper for CodeMirror.
 *
 * Using CodeMirror's "lint" addon: https://codemirror.net/demo/lint.html
 * Given the text within an editor, this helper will take that text and return
 * a list of linter issues ensuring that correct variables were provided.
 *
 * Options:
 *
 *   - variableToType: { [variable: string]: GraphQLInputType }
 *
 */
_codemirror["default"].registerHelper('lint', 'graphql-variables', function (text, options, editor) {
  // If there's no text, do nothing.
  if (!text) {
    return [];
  } // First, linter needs to determine if there are any parsing errors.


  var ast;

  try {
    ast = (0, _jsonParse["default"])(text);
  } catch (syntaxError) {
    if (syntaxError.stack) {
      throw syntaxError;
    }

    return [lintError(editor, syntaxError, syntaxError.message)];
  } // If there are not yet known variables, do nothing.


  var variableToType = options.variableToType;

  if (!variableToType) {
    return [];
  } // Then highlight any issues with the provided variables.


  return validateVariables(editor, variableToType, ast);
}); // Given a variableToType object, a source text, and a JSON AST, produces a
// list of CodeMirror annotations for any variable validation errors.


function validateVariables(editor, variableToType, variablesAST) {
  var errors = [];
  variablesAST.members.forEach(function (member) {
    var variableName = member.key.value;
    var type = variableToType[variableName];

    if (!type) {
      errors.push(lintError(editor, member.key, "Variable \"$".concat(variableName, "\" does not appear in any GraphQL query.")));
    } else {
      validateValue(type, member.value).forEach(function (_ref) {
        var _ref2 = _slicedToArray(_ref, 2),
            node = _ref2[0],
            message = _ref2[1];

        errors.push(lintError(editor, node, message));
      });
    }
  });
  return errors;
} // Returns a list of validation errors in the form Array<[Node, String]>.


function validateValue(type, valueAST) {
  // Validate non-nullable values.
  if (type instanceof _graphql.GraphQLNonNull) {
    if (valueAST.kind === 'Null') {
      return [[valueAST, "Type \"".concat(type, "\" is non-nullable and cannot be null.")]];
    }

    return validateValue(type.ofType, valueAST);
  }

  if (valueAST.kind === 'Null') {
    return [];
  } // Validate lists of values, accepting a non-list as a list of one.


  if (type instanceof _graphql.GraphQLList) {
    var itemType = type.ofType;

    if (valueAST.kind === 'Array') {
      return mapCat(valueAST.values, function (item) {
        return validateValue(itemType, item);
      });
    }

    return validateValue(itemType, valueAST);
  } // Validate input objects.


  if (type instanceof _graphql.GraphQLInputObjectType) {
    if (valueAST.kind !== 'Object') {
      return [[valueAST, "Type \"".concat(type, "\" must be an Object.")]];
    } // Validate each field in the input object.


    var providedFields = Object.create(null);
    var fieldErrors = mapCat(valueAST.members, function (member) {
      var fieldName = member.key.value;
      providedFields[fieldName] = true;
      var inputField = type.getFields()[fieldName];

      if (!inputField) {
        return [[member.key, "Type \"".concat(type, "\" does not have a field \"").concat(fieldName, "\".")]];
      }

      var fieldType = inputField ? inputField.type : undefined;
      return validateValue(fieldType, member.value);
    }); // Look for missing non-nullable fields.

    Object.keys(type.getFields()).forEach(function (fieldName) {
      if (!providedFields[fieldName]) {
        var fieldType = type.getFields()[fieldName].type;

        if (fieldType instanceof _graphql.GraphQLNonNull) {
          fieldErrors.push([valueAST, "Object of type \"".concat(type, "\" is missing required field \"").concat(fieldName, "\".")]);
        }
      }
    });
    return fieldErrors;
  } // Validate common scalars.


  if (type.name === 'Boolean' && valueAST.kind !== 'Boolean' || type.name === 'String' && valueAST.kind !== 'String' || type.name === 'ID' && valueAST.kind !== 'Number' && valueAST.kind !== 'String' || type.name === 'Float' && valueAST.kind !== 'Number' || type.name === 'Int' && ( // eslint-disable-next-line no-bitwise
  valueAST.kind !== 'Number' || (valueAST.value | 0) !== valueAST.value)) {
    return [[valueAST, "Expected value of type \"".concat(type, "\".")]];
  } // Validate enums and custom scalars.


  if (type instanceof _graphql.GraphQLEnumType || type instanceof _graphql.GraphQLScalarType) {
    if (valueAST.kind !== 'String' && valueAST.kind !== 'Number' && valueAST.kind !== 'Boolean' && valueAST.kind !== 'Null' || isNullish(type.parseValue(valueAST.value))) {
      return [[valueAST, "Expected value of type \"".concat(type, "\".")]];
    }
  }

  return [];
} // Give a parent text, an AST node with location, and a message, produces a
// CodeMirror annotation object.


function lintError(editor, node, message) {
  return {
    message: message,
    severity: 'error',
    type: 'validation',
    from: editor.posFromIndex(node.start),
    to: editor.posFromIndex(node.end)
  };
}

function isNullish(value) {
  // eslint-disable-next-line no-self-compare
  return value === null || value === undefined || value !== value;
}

function mapCat(array, mapper) {
  return Array.prototype.concat.apply([], array.map(mapper));
}