"use strict";

var _codemirror = _interopRequireDefault(require("codemirror"));

var _graphqlLanguageServiceInterface = require("graphql-language-service-interface");

var _graphql = require("graphql");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 *  Copyright (c) 2020 GraphQL Contributors
 *  All rights reserved.
 *
 *  This source code is licensed under the BSD-style license found in the
 *  LICENSE file in the root directory of this source tree. An additional grant
 *  of patent rights can be found in the PATENTS file in the same directory.
 *
 * 
 *
 */

/**
 * Registers a "hint" helper for CodeMirror.
 *
 * Using CodeMirror's "hint" addon: https://codemirror.net/demo/complete.html
 * Given an editor, this helper will take the token at the cursor and return a
 * list of suggested tokens.
 *
 * Options:
 *
 *   - schema: GraphQLSchema provides the hinter with positionally relevant info
 *
 * Additional Events:
 *
 *   - hasCompletion (codemirror, data, token) - signaled when the hinter has a
 *     new list of completion suggestions.
 *
 */
_codemirror.default.registerHelper('hint', 'graphql', (editor, options) => {
  const schema = options.schema;

  if (!schema) {
    return;
  }

  const cur = editor.getCursor();
  const token = editor.getTokenAt(cur);
  const rawResults = getAutocompleteSuggestions(schema, editor.getValue(), token);
  const tokenStart = token.type !== null && /"|\w/.test(token.string[0]) ? token.start : token.end;
  const results = {
    list: rawResults.map(item => ({
      text: item.label,
      type: item.type,
      description: item.documentation,
      isDeprecated: item.isDeprecated,
      deprecationReason: item.deprecationReason
    })),
    from: {
      line: cur.line,
      column: tokenStart
    },
    to: {
      line: cur.line,
      column: token.end
    }
  };

  if (results && results.list && results.list.length > 0) {
    results.from = _codemirror.default.Pos(results.from.line, results.from.column);
    results.to = _codemirror.default.Pos(results.to.line, results.to.column);

    _codemirror.default.signal(editor, 'hasCompletion', editor, results, token);
  }

  return results;
});
/**
 * Given GraphQLSchema, queryText, and context of the current position within
 * the source text, provide a list of typeahead entries.
 */


function getAutocompleteSuggestions(schema, queryText, token) {
  const state = token.state.kind === 'Invalid' ? token.state.prevState : token.state; // relieve flow errors by checking if `state` exists

  if (!state) {
    return [];
  }

  const kind = state.kind;
  const step = state.step;
  const typeInfo = (0, _graphqlLanguageServiceInterface.getTypeInfo)(schema, token.state); // Definition kinds

  if (kind === 'Document') {
    return (0, _graphqlLanguageServiceInterface.hintList)(token, [{
      label: 'query'
    }, {
      label: 'mutation'
    }, {
      label: 'subscription'
    }, {
      label: 'fragment'
    }, {
      label: '{'
    }]);
  } // Field names


  if (kind === 'SelectionSet' || kind === 'Field' || kind === 'AliasedField') {
    return getSuggestionsForFieldNames(token, typeInfo, schema);
  } // Argument names


  if (kind === 'Arguments' || kind === 'Argument' && step === 0) {
    const argDefs = typeInfo.argDefs;

    if (argDefs) {
      return (0, _graphqlLanguageServiceInterface.hintList)(token, argDefs.map(argDef => {
        var _argDef$description;

        return {
          label: argDef.name,
          type: argDef.type,
          documentation: (_argDef$description = argDef.description) !== null && _argDef$description !== void 0 ? _argDef$description : undefined
        };
      }));
    }
  } // Input Object fields


  if (kind === 'ObjectValue' || kind === 'ObjectField' && step === 0) {
    if (typeInfo.objectFieldDefs) {
      const objectFields = (0, _graphqlLanguageServiceInterface.objectValues)(typeInfo.objectFieldDefs);
      return (0, _graphqlLanguageServiceInterface.hintList)(token, objectFields.map(field => {
        var _field$description;

        return {
          label: field.name,
          type: field.type,
          documentation: (_field$description = field.description) !== null && _field$description !== void 0 ? _field$description : undefined
        };
      }));
    }
  } // Input values: Enum and Boolean


  if (kind === 'EnumValue' || kind === 'ListValue' && step === 1 || kind === 'ObjectField' && step === 2 || kind === 'Argument' && step === 2) {
    return getSuggestionsForInputValues(token, typeInfo);
  } // Fragment type conditions


  if (kind === 'TypeCondition' && step === 1 || kind === 'NamedType' && state.prevState != null && state.prevState.kind === 'TypeCondition') {
    return getSuggestionsForFragmentTypeConditions(token, typeInfo, schema);
  } // Fragment spread names


  if (kind === 'FragmentSpread' && step === 1) {
    return getSuggestionsForFragmentSpread(token, typeInfo, schema, queryText);
  } // Variable definition types


  if (kind === 'VariableDefinition' && step === 2 || kind === 'ListType' && step === 1 || kind === 'NamedType' && state.prevState && (state.prevState.kind === 'VariableDefinition' || state.prevState.kind === 'ListType')) {
    return getSuggestionsForVariableDefinition(token, schema);
  } // Directive names


  if (kind === 'Directive') {
    return getSuggestionsForDirective(token, state, schema);
  }

  return [];
} // Helper functions to get suggestions for each kinds


function getSuggestionsForFieldNames(token, typeInfo, schema) {
  if (typeInfo.parentType) {
    const parentType = typeInfo.parentType;
    const fields = (0, _graphql.isObjectType)(parentType) || (0, _graphql.isInterfaceType)(parentType) ? (0, _graphqlLanguageServiceInterface.objectValues)(parentType.getFields()) : [];

    if ((0, _graphql.isCompositeType)(parentType)) {
      fields.push(_graphql.TypeNameMetaFieldDef);
    }

    if (parentType === schema.getQueryType()) {
      fields.push(_graphql.SchemaMetaFieldDef, _graphql.TypeMetaFieldDef);
    }

    return (0, _graphqlLanguageServiceInterface.hintList)(token, fields.map(field => {
      var _field$description2;

      return {
        label: field.name,
        type: field.type,
        documentation: (_field$description2 = field.description) !== null && _field$description2 !== void 0 ? _field$description2 : undefined,
        isDeprecated: field.isDeprecated,
        deprecationReason: field.deprecationReason
      };
    }));
  }

  return [];
}

function getSuggestionsForInputValues(token, typeInfo) {
  const namedInputType = (0, _graphql.getNamedType)(typeInfo.inputType);

  if (namedInputType instanceof _graphql.GraphQLEnumType) {
    const values = namedInputType.getValues();
    return (0, _graphqlLanguageServiceInterface.hintList)(token, values.map(value => {
      var _value$description;

      return {
        label: value.name,
        type: namedInputType,
        documentation: (_value$description = value.description) !== null && _value$description !== void 0 ? _value$description : undefined,
        isDeprecated: value.isDeprecated,
        deprecationReason: value.deprecationReason
      };
    }));
  } else if (namedInputType === _graphql.GraphQLBoolean) {
    return (0, _graphqlLanguageServiceInterface.hintList)(token, [{
      label: 'true',
      type: _graphql.GraphQLBoolean,
      documentation: 'Not false.'
    }, {
      label: 'false',
      type: _graphql.GraphQLBoolean,
      documentation: 'Not true.'
    }]);
  }

  return [];
}

function getSuggestionsForFragmentTypeConditions(token, typeInfo, schema) {
  let possibleTypes;

  if (typeInfo.parentType) {
    if ((0, _graphql.isAbstractType)(typeInfo.parentType)) {
      const abstractType = (0, _graphql.assertAbstractType)(typeInfo.parentType); // Collect both the possible Object types as well as the interfaces
      // they implement.

      const possibleObjTypes = schema.getPossibleTypes(abstractType);
      const possibleIfaceMap = Object.create(null);
      possibleObjTypes.forEach(type => {
        type.getInterfaces().forEach(iface => {
          possibleIfaceMap[iface.name] = iface;
        });
      });
      possibleTypes = possibleObjTypes.concat((0, _graphqlLanguageServiceInterface.objectValues)(possibleIfaceMap));
    } else {
      // The parent type is a non-abstract Object type, so the only possible
      // type that can be used is that same type.
      possibleTypes = [typeInfo.parentType];
    }
  } else {
    const typeMap = schema.getTypeMap();
    possibleTypes = (0, _graphqlLanguageServiceInterface.objectValues)(typeMap).filter(_graphql.isCompositeType);
  }

  return (0, _graphqlLanguageServiceInterface.hintList)(token, possibleTypes.map(type => {
    const namedType = (0, _graphql.getNamedType)(type);
    return {
      label: String(type),
      documentation: namedType && namedType.description || ''
    };
  }));
}

function getSuggestionsForFragmentSpread(token, typeInfo, schema, queryText) {
  const typeMap = schema.getTypeMap();
  const defState = (0, _graphqlLanguageServiceInterface.getDefinitionState)(token.state);
  const fragments = (0, _graphqlLanguageServiceInterface.getFragmentDefinitions)(queryText); // Filter down to only the fragments which may exist here.

  const relevantFrags = fragments.filter(frag => // Only include fragments with known types.
  typeMap[frag.typeCondition.name.value] && // Only include fragments which are not cyclic.
  !(defState && defState.kind === 'FragmentDefinition' && defState.name === frag.name.value) && // Only include fragments which could possibly be spread here.
  (0, _graphql.isCompositeType)(typeInfo.parentType) && (0, _graphql.isCompositeType)(typeMap[frag.typeCondition.name.value]) && (0, _graphql.doTypesOverlap)(schema, typeInfo.parentType, typeMap[frag.typeCondition.name.value]));
  return (0, _graphqlLanguageServiceInterface.hintList)(token, relevantFrags.map(frag => ({
    label: frag.name.value,
    type: typeMap[frag.typeCondition.name.value],
    documentation: `fragment ${frag.name.value} on ${frag.typeCondition.name.value}`
  })));
}

function getSuggestionsForVariableDefinition(token, schema) {
  const inputTypeMap = schema.getTypeMap();
  const inputTypes = (0, _graphqlLanguageServiceInterface.objectValues)(inputTypeMap).filter(_graphql.isInputType);
  return (0, _graphqlLanguageServiceInterface.hintList)(token, inputTypes.map(type => ({
    label: type.name,
    documentation: type.description
  })));
}

function getSuggestionsForDirective(token, state, schema) {
  if (state.prevState && state.prevState.kind) {
    const directives = schema.getDirectives().filter(directive => (0, _graphqlLanguageServiceInterface.canUseDirective)(state.prevState, directive));
    return (0, _graphqlLanguageServiceInterface.hintList)(token, directives.map(directive => ({
      label: directive.name,
      documentation: directive.description || ''
    })));
  }

  return [];
}