"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
var codemirror_1 = __importDefault(require("codemirror"));
var graphql_1 = require("graphql");
var jsonParse_1 = __importDefault(require("../utils/jsonParse"));
codemirror_1.default.registerHelper('lint', 'graphql-variables', function (text, options, editor) {
    if (!text) {
        return [];
    }
    var ast;
    try {
        ast = jsonParse_1.default(text);
    }
    catch (syntaxError) {
        if (syntaxError.stack) {
            throw syntaxError;
        }
        return [lintError(editor, syntaxError, syntaxError.message)];
    }
    var variableToType = options.variableToType;
    if (!variableToType) {
        return [];
    }
    return validateVariables(editor, variableToType, ast);
});
function validateVariables(editor, variableToType, variablesAST) {
    var errors = [];
    variablesAST.members.forEach(function (member) {
        var _a;
        if (member) {
            var variableName = (_a = member.key) === null || _a === void 0 ? void 0 : _a.value;
            var type = variableToType[variableName];
            if (!type) {
                errors.push(lintError(editor, member.key, "Variable \"$" + variableName + "\" does not appear in any GraphQL query."));
            }
            else {
                validateValue(type, member.value).forEach(function (_a) {
                    var node = _a[0], message = _a[1];
                    errors.push(lintError(editor, node, message));
                });
            }
        }
    });
    return errors;
}
function validateValue(type, valueAST) {
    if (!type || !valueAST) {
        return [];
    }
    if (type instanceof graphql_1.GraphQLNonNull) {
        if (valueAST.kind === 'Null') {
            return [[valueAST, "Type \"" + type + "\" is non-nullable and cannot be null."]];
        }
        return validateValue(type.ofType, valueAST);
    }
    if (valueAST.kind === 'Null') {
        return [];
    }
    if (type instanceof graphql_1.GraphQLList) {
        var itemType_1 = type.ofType;
        if (valueAST.kind === 'Array') {
            var values = valueAST.values || [];
            return mapCat(values, function (item) { return validateValue(itemType_1, item); });
        }
        return validateValue(itemType_1, valueAST);
    }
    if (type instanceof graphql_1.GraphQLInputObjectType) {
        if (valueAST.kind !== 'Object') {
            return [[valueAST, "Type \"" + type + "\" must be an Object."]];
        }
        var providedFields_1 = Object.create(null);
        var fieldErrors_1 = mapCat(valueAST.members, function (member) {
            var _a;
            var fieldName = (_a = member === null || member === void 0 ? void 0 : member.key) === null || _a === void 0 ? void 0 : _a.value;
            providedFields_1[fieldName] = true;
            var inputField = type.getFields()[fieldName];
            if (!inputField) {
                return [
                    [
                        member.key,
                        "Type \"" + type + "\" does not have a field \"" + fieldName + "\".",
                    ],
                ];
            }
            var fieldType = inputField ? inputField.type : undefined;
            return validateValue(fieldType, member.value);
        });
        Object.keys(type.getFields()).forEach(function (fieldName) {
            if (!providedFields_1[fieldName]) {
                var fieldType = type.getFields()[fieldName].type;
                if (fieldType instanceof graphql_1.GraphQLNonNull) {
                    fieldErrors_1.push([
                        valueAST,
                        "Object of type \"" + type + "\" is missing required field \"" + fieldName + "\".",
                    ]);
                }
            }
        });
        return fieldErrors_1;
    }
    if ((type.name === 'Boolean' && valueAST.kind !== 'Boolean') ||
        (type.name === 'String' && valueAST.kind !== 'String') ||
        (type.name === 'ID' &&
            valueAST.kind !== 'Number' &&
            valueAST.kind !== 'String') ||
        (type.name === 'Float' && valueAST.kind !== 'Number') ||
        (type.name === 'Int' &&
            (valueAST.kind !== 'Number' || (valueAST.value | 0) !== valueAST.value))) {
        return [[valueAST, "Expected value of type \"" + type + "\"."]];
    }
    if (type instanceof graphql_1.GraphQLEnumType || type instanceof graphql_1.GraphQLScalarType) {
        if ((valueAST.kind !== 'String' &&
            valueAST.kind !== 'Number' &&
            valueAST.kind !== 'Boolean' &&
            valueAST.kind !== 'Null') ||
            isNullish(type.parseValue(valueAST.value))) {
            return [[valueAST, "Expected value of type \"" + type + "\"."]];
        }
    }
    return [];
}
function lintError(editor, node, message) {
    return {
        message: message,
        severity: 'error',
        type: 'validation',
        from: editor.posFromIndex(node.start),
        to: editor.posFromIndex(node.end),
    };
}
function isNullish(value) {
    return value === null || value === undefined || value !== value;
}
function mapCat(array, mapper) {
    return Array.prototype.concat.apply([], array.map(mapper));
}
//# sourceMappingURL=lint.js.map