"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const Loader_1 = require("./Loader");
const Util = require("./Util");
/**
 * Compile a configuration stream to a JavaScript source file.
 * @param {LoaderProperties} properties Properties for the loader.
 * @param {string} configPath Path of the config file.
 * @param {"stream".internal.Stream} configStreamRaw Stream of the config file contents.
 * @param {string} configResourceUri URI of the config element to compile.
 * @param {string} exportVariableName An optional variable name to export instead of the default runner.
 * @return {Promise<string>} A string resolving to the JavaScript contents.
 */
function compileConfig(properties, configPath, configStreamRaw, configResourceUri, exportVariableName) {
    exportVariableName = exportVariableName ? Util.uriToVariableName(exportVariableName) : exportVariableName;
    const loader = new Loader_1.Loader(properties);
    return loader.registerAvailableModuleResources()
        .then(() => {
        return Promise.all([loader._getContexts(), loader._getImportPaths()]).then(([contexts, importPaths]) => {
            const configStream = Util.parseRdf(configStreamRaw, configPath, properties.mainModulePath, false, true, contexts, importPaths);
            const moduleLines = [];
            return loader.instantiateFromStream(configResourceUri, configStream, { serializations: moduleLines })
                .then((serializationVariableName) => {
                let document = moduleLines.join('\n');
                if (exportVariableName !== serializationVariableName) {
                    // Remove the instantiation of the runner component, as it will not be needed anymore.
                    document = document.replace('new (require(\'@comunica/runner\').Runner)', '');
                }
                return document + '\n' + 'module.exports = ' + (exportVariableName || serializationVariableName) + ';';
            });
        });
    });
}
exports.compileConfig = compileConfig;
//# sourceMappingURL=CompileUtil.js.map