/// <reference types="node" />
import { Stream } from "stream";
import { RdfClassLoader } from "./rdf/RdfClassLoader";
import { Resource } from "./rdf/Resource";
import { IComponentFactory, ICreationSettings } from "./factory/IComponentFactory";
/**
 * A loader class for component configs.
 * Modules must first be registered to this loader.
 * After that, components can be instantiated.
 * Components with the same URI will only be instantiated once.
 */
export declare class Loader {
    _properties: LoaderProperties;
    _componentResources: {
        [id: string]: Resource;
    };
    /**
     * Require overrides.
     * Require name as path, require override as value.
     */
    overrideRequireNames: {
        [id: string]: string;
    };
    _runTypeConfigs: {
        [id: string]: Resource[];
    };
    _instances: {
        [id: string]: any;
    };
    _registrationFinalized: boolean;
    /**
     * Shared mapping from resource URI to resource instance for al RDF class loaders.
     */
    resources: {
        [id: string]: Resource;
    };
    constructor(properties?: LoaderProperties);
    _getContexts(): Promise<{
        [id: string]: string;
    }>;
    _getImportPaths(): Promise<{
        [id: string]: string;
    }>;
    /**
     * @returns {RdfClassLoader} A new RDF class loader for loading modules and components
     * @private
     */
    _newModuleLoader(): RdfClassLoader;
    /**
     * @returns {RdfClassLoader} A new RDF class loader for loading component configs
     * @private
     */
    _newConfigLoader(): RdfClassLoader;
    /**
     * Register a new component.
     * This will ensure that component configs referring to this component as type are recognized.
     * @param componentResource A component resource.
     * @private
     */
    _registerComponentResource(componentResource: Resource): void;
    /**
     * Check if the given resource is a valid component.
     * @param componentResource A resource.
     * @returns {boolean} If the resource is a valid component.
     * @private
     */
    _isValidComponent(componentResource: Resource): boolean;
    /**
     * Require that the given resource is a valid component,
     * otherwise and error is thrown.
     * @param componentResource A resource.
     * @param referencingComponent The optional component referencing the given component.
     * @private
     */
    _requireValidComponent(componentResource: Resource, referencingComponent?: Resource): void;
    /**
     * Let the given component inherit parameters from the given component(s) if applicable.
     * @param componentResource The component resource
     * @param classes The component classes to inherit from.
     * @private
     */
    _inheritParameters(componentResource: any, classes?: Resource[]): void;
    /**
     * Let the given component inherit constructor mappings from the given component(s) if applicable.
     * @param componentResource The component resource
     * @private
     */
    _inheritConstructorParameters(componentResource: any): void;
    /**
     * Let the given object inherit the given fields from the given component(s) if applicable.
     * @param object The object resource
     * @param classes The objects to inherit from.
     * @private
     */
    _inheritObjectFields(object: any, classes?: Resource[]): void;
    /**
     * Register a new module and its components.
     * This will ensure that component configs referring to components as types of this module are recognized.
     * @param moduleResource A module resource.
     */
    registerModuleResource(moduleResource: Resource): void;
    /**
     * Register new modules and their components.
     * This will ensure that component configs referring to components as types of these modules are recognized.
     * @param moduleResourceStream A triple stream containing modules.
     * @returns {Promise<T>} A promise that resolves once loading has finished.
     */
    registerModuleResourcesStream(moduleResourceStream: Stream): Promise<void>;
    /**
     * Register new modules and their components.
     * This will ensure that component configs referring to components as types of these modules are recognized.
     * @param moduleResourceUrl An RDF document URL
     * @param fromPath The path to base relative paths on. This will typically be __dirname.
     * @returns {Promise<T>} A promise that resolves once loading has finished.
     */
    registerModuleResourcesUrl(moduleResourceUrl: string, fromPath?: string): Promise<void>;
    /**
     * Register all reachable modules and their components.
     * This will interpret the package.json from the main module and all its dependencies for discovering modules.
     * This will ensure that component configs referring to components as types of these modules are recognized.
     * @returns {Promise<T>} A promise that resolves once loading has finished.
     */
    registerAvailableModuleResources(): Promise<void>;
    /**
     * Get a component config constructor based on a Resource.
     * @param configResource A config resource.
     * @returns The component factory.
     */
    getConfigConstructor(configResource: Resource): IComponentFactory;
    /**
     * Instantiate a component based on a Resource.
     * @param configResource A config resource.
     * @param settings The settings for creating the instance.
     * @returns {any} The run instance.
     */
    instantiate(configResource: Resource, settings?: ICreationSettings): Promise<any>;
    /**
     * Let then given config inherit parameter values from referenced passed configs.
     * @param configResource The config
     * @param componentResource The component
     * @private
     */
    _inheritParameterValues(configResource: Resource, componentResource: any): void;
    /**
     * Set the loader to a state where it doesn't accept anymore module and component registrations.
     * This is required for post-processing the components, for actions such as parameter inheritance,
     * index creation and cleanup.
     */
    finalizeRegistration(): void;
    _checkFinalizeRegistration(): void;
    /**
     * Get a component config constructor based on a config URI.
     * @param configResourceUri The config resource URI.
     * @param configResourceStream A triple stream containing at least the given config.
     * @returns {Promise<T>} A promise resolving to the component constructor.
     */
    getConfigConstructorFromStream(configResourceUri: string, configResourceStream: Stream): Promise<IComponentFactory>;
    /**
     * Instantiate a component based on a config URI and a stream.
     * @param configResourceUri The config resource URI.
     * @param configResourceStream A triple stream containing at least the given config.
     * @param settings The settings for creating the instance.
     * @returns {Promise<T>} A promise resolving to the run instance.
     */
    instantiateFromStream(configResourceUri: string, configResourceStream: Stream, settings?: ICreationSettings): Promise<any>;
    /**
     * Run a component config based on a config URI.
     * @param configResourceUri The config resource URI.
     * @param configResourceUrl An RDF document URL
     * @param fromPath The path to base relative paths on. This will typically be __dirname.
     *                 Default is the current running directory.
     * @returns {Promise<T>} A promise resolving to the run instance.
     */
    getConfigConstructorFromUrl(configResourceUri: string, configResourceUrl: string, fromPath?: string): Promise<IComponentFactory>;
    /**
     * Instantiate a component based on a config URI.
     * @param configResourceUri The config resource URI.
     * @param configResourceUrl An RDF document URL
     * @param fromPath The path to base relative paths on. This will typically be __dirname.
     *                 Default is the current running directory.
     * @param settings The settings for creating the instance.
     * @returns {Promise<T>} A promise resolving to the run instance.
     */
    instantiateFromUrl(configResourceUri: string, configResourceUrl: string, fromPath?: string, settings?: ICreationSettings): Promise<any>;
    /**
     * Instantiate a component based on component URI and a set of parameters.
     * @param componentUri The URI of a component.
     * @param params A dictionary with named parameters.
     * @param settings The settings for creating the instance.
     * @returns {any} The run instance.
     */
    instantiateManually(componentUri: string, params: {
        [id: string]: string;
    }, settings?: ICreationSettings): any;
}
export interface LoaderProperties {
    scanGlobal?: boolean;
    absolutizeRelativePaths?: boolean;
    contexts?: {
        [id: string]: string;
    };
    importPaths?: {
        [id: string]: string;
    };
    mainModulePath?: string;
}
