/// <reference types="node" />
import { Stream } from "stream";
import { Resource } from "./rdf/Resource";
declare class Util {
    static readonly PREFIXES: {
        [id: string]: string;
    };
    static NODE_MODULES_PACKAGE_CONTENTS: {
        [id: string]: string;
    };
    private static MAIN_MODULE_PATH;
    private static MAIN_MODULE_PATHS;
    private static cachedAvailableNodeModules;
    /**
     * Get the file contents from a file path or URL
     * @param path The file path or url.
     * @param fromPath The path to base relative paths on.
     *                 Default is the current running directory.
     * @returns {Promise<T>} A promise resolving to the data stream.
     * @private
     */
    static getContentsFromUrlOrPath(path: string, fromPath?: string): Promise<Stream>;
    /**
     * Parse the given data stream to a triple stream.
     * @param rdfDataStream The data stream.
     * @param path The file path or url.
     * @param fromPath The path to base relative paths on.
     *                 Default is the current running directory.
     * @param absolutizeRelativePaths If relative paths 'file://' should be made absolute 'file:///'.
     * @param ignoreImports If imports should be ignored. Default: false
     * @param contexts The cached JSON-LD contexts
     * @param importPaths The cached import paths.
     * @returns A triple stream.
     * @private
     */
    static parseRdf(rdfDataStream: Stream, path: string, fromPath?: string, ignoreImports?: boolean, absolutizeRelativePaths?: boolean, contexts?: {
        [id: string]: string;
    }, importPaths?: {
        [id: string]: string;
    }): Stream;
    /**
     * Apply parameter values for the given parameter.
     * @param resourceScope The resource scope to map in.
     * @param param A parameter type.
     * @param paramValueMapping A mapping from parameter to value.
     * @return The parameter value(s) or undefined
     */
    static applyParameterValues(resourceScope: Resource, param: any, paramValueMapping: any): any;
    /**
     * Apply the given datatype to the given literal.
     * Checks if the datatype is correct and casts to the correct js type.
     * Will throw an error if the type has an invalid value.
     * Will be ignored if the value is not a literal or the type is not recognized.
     * @param value The value.
     * @param param The parameter.
     * @returns {any} The tranformed value.
     */
    static captureType(value: any, param: any): any;
    /**
     * Set the main module path.
     * This will also update the main module paths.
     * @param {string} path A path.
     */
    static setMainModulePath(path: string): void;
    static initDefaultMainModulePath(): string;
    /**
     * @returns {any} The path to the current main Node module.
     */
    static getMainModulePath(): string;
    /**
     * Set the main module paths.
     * @param {string[]} paths A list paths. Like require.main.paths, but starting from the main module path.
     */
    static setMainModulePaths(paths: string[]): void;
    /**
     * Set the main module paths.
     * @return {string[]} A list of paths. Like require.main.paths, but starting from the main module path.
     */
    static getMainModulePaths(): string[];
    /**
     * Get all currently available node module paths.
     * @param path The path to start from.
     * @param cb A callback for each absolute path.
     * @param ignorePaths The paths that should be ignored.
     */
    static getAvailableNodeModules(path: string, cb: (path: string) => any, ignorePaths?: {
        [key: string]: boolean;
    }): void;
    /**
     * Get the package.json file from the given path.
     * Require's will be cached.
     * @param path The path.
     * @returns {any} The package.json or null.
     */
    static getPackageJson(path: string): any;
    /**
     * Get all component files paths reachable from the given path.
     * This checks all available node modules and checks their package.json
     * for `lsd:module` and `lsd:components`.
     * @param path The path to search from.
     * @return A promise resolving to a mapping of module URI to component file name
     */
    static getModuleComponentPaths(path: string): Promise<{
        [id: string]: string;
    }>;
    /**
     * Get all currently available component files paths.
     * This checks all available node modules and checks their package.json
     * for `lsd:module` and `lsd:components`.
     * @param scanGlobal If global modules should also be scanned next to local modules.
     * @return A promise resolving to a mapping of module URI to component file name
     */
    static getAvailableModuleComponentPaths(scanGlobal: boolean): Promise<{
        [id: string]: string;
    }>;
    /**
     * Get all JSON-LD contexts reachable from the given path.
     * This checks all available node modules and checks their package.json
     * for `lsd:contexts`.
     * @param path The path to search from.
     * @return A promise resolving to a mapping of context URL to context contents
     */
    static getContextPaths(path: string): Promise<{
        [id: string]: string;
    }>;
    /**
     * Get all currently available JSON-LD contexts.
     * This checks all available node modules and checks their package.json
     * for `lsd:contexts`.
     * @param scanGlobal If global modules should also be scanned next to local modules.
     * @return A promise resolving to a mapping of context URL to context contents
     */
    static getAvailableContexts(scanGlobal: boolean): Promise<{
        [id: string]: string;
    }>;
    /**
     * Get all import paths reachable from the given path.
     * This checks all available node modules and checks their package.json
     * for `lsd:importPaths`.
     * @param path The path to search from.
     * @return A promise resolving to a mapping of an import prefix URL to an import prefix path
     */
    static getImportPaths(path: string): Promise<{
        [id: string]: string;
    }>;
    /**
     * Get all currently import prefix paths.
     * This checks all available node modules and checks their package.json
     * for `lsd:importPaths`.
     * @param scanGlobal If global modules should also be scanned next to local modules.
     * @return A promise resolving to a mapping of an import prefix URL to an import prefix path
     */
    static getAvailableImportPaths(scanGlobal: boolean): Promise<{
        [id: string]: string;
    }>;
    /**
     * Add a file path to an error message.
     * @param e The original error message.
     * @param filePath The file path.
     * @param fromPath The optional base path.
     * @returns {Error} The new error with file path context.
     */
    static addFilePathToError(e: Error, filePath: string, fromPath?: string): Error;
    /**
     * Deterministically converts a URI to a variable name that is safe for usage within JavaScript.
     * @param {string} uri A URI.
     * @return {string} A variable name.
     */
    static uriToVariableName(uri: string): string;
}
export = Util;
