"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const stream_1 = require("stream");
const Resource_1 = require("./Resource");
const Util = require("../Util");
/**
 * An RdfClassLoader is a writable stream that accepts triple streams and loads resources in-memory.
 * Class and property URI's can be bound to names.
 */
class RdfClassLoader extends stream_1.Writable {
    constructor(options) {
        super({ objectMode: true });
        this._options = {};
        this._classes = {};
        this._properties = {};
        this._uniqueProperties = {};
        /**
         * Mapping from resource URI to resource instance.
         */
        this.resources = {};
        /**
         * Mapping from resource type name to an array of resource instances.
         */
        this.typedResources = {};
        this._options = options || { normalizeLists: true };
        this._captureAllProperties = this._options.captureAllProperties || false;
        this._captureAllClasses = this._options.captureAllClasses || false;
        if (this._options['normalizeLists']) {
            this.bindProperty('__listFirst', Util.PREFIXES['rdf'] + 'first');
            this.bindProperty('__listRest', Util.PREFIXES['rdf'] + 'rest');
            this.on('finish', () => {
                // Normalize lists
                let keys = Object.keys(this.resources);
                let listNodes = [];
                for (let i = keys.length - 1; i >= 0; i--) {
                    let element = this.resources[keys[i]];
                    if (element.__listFirst && element.__listRest) {
                        listNodes.push(keys[i]);
                        element.list = element.__listRest[0]['value'] === Util.PREFIXES['rdf'] + 'nil' ? [element.__listFirst[0]] : [element.__listFirst[0]].concat(element.__listRest[0]['list']);
                        delete element.__listFirst;
                        delete element.__listRest;
                        delete this.resources[keys[i]];
                    }
                }
                delete this.resources[Util.PREFIXES['rdf'] + 'nil'];
            });
        }
    }
    /**
     * Bind the given class name to the given URI.
     * @param className The class name.
     * @param uri The class URI.
     */
    bindClass(className, uri) {
        this._classes[uri] = className;
    }
    /**
     * Bind the given property field name to the given URI.
     * @param fieldName The field name.
     * @param uri The predicate URI.
     * @param unique If the property should be unique.
     */
    bindProperty(fieldName, uri, unique) {
        this._properties[uri] = fieldName;
        if (unique) {
            this.setUniqueProperty(fieldName);
        }
    }
    /**
     * Indicate that the given property is unique for a resource.
     * @param fieldName The field name of the property.
     */
    setUniqueProperty(fieldName) {
        this._uniqueProperties[fieldName] = true;
    }
    _getOrMakeResource(value) {
        let instance = this.resources[value];
        if (!instance) {
            instance = this.resources[value] = new Resource_1.Resource(value);
        }
        return instance;
    }
    _pushPredicate(fieldName, triple) {
        let subjectInstance = this._getOrMakeResource(triple.subject);
        let objectInstance = this._getOrMakeResource(triple.object);
        if (!this._uniqueProperties[fieldName]) {
            if (!subjectInstance[fieldName]) {
                subjectInstance[fieldName] = [];
            }
            if (subjectInstance[fieldName].indexOf(objectInstance) < 0) {
                subjectInstance[fieldName].push(objectInstance);
            }
        }
        else {
            if (subjectInstance[fieldName]) {
                this.emit('error', new Error('Predicate ' + triple.predicate + ' with field ' + fieldName
                    + ' was indicated as unique, while the objects ' + subjectInstance[fieldName].value
                    + ' and ' + objectInstance.value + ' were found for subject ' + subjectInstance.value + '.'));
            }
            else {
                subjectInstance[fieldName] = objectInstance;
            }
        }
    }
    _write(triple, encoding, done) {
        // Store fields for the configured predicates
        let fieldName = this._properties[triple.predicate];
        if (this._captureAllProperties) {
            this._pushPredicate(triple.predicate, triple);
        }
        if (fieldName) {
            this._pushPredicate(fieldName, triple);
        }
        // Store types for the configured classes
        if (triple.predicate === Util.PREFIXES['rdf'] + 'type') {
            let subjectInstance = this._getOrMakeResource(triple.subject);
            let typeName = this._classes[triple.object];
            if (!typeName && this._captureAllClasses) {
                typeName = triple.object;
            }
            if (typeName) {
                if (!this.typedResources[typeName]) {
                    this.typedResources[typeName] = [];
                }
                this.typedResources[typeName].push(subjectInstance);
            }
        }
        done();
    }
}
exports.RdfClassLoader = RdfClassLoader;
//# sourceMappingURL=RdfClassLoader.js.map