"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const stream_1 = require("stream");
const Path = require("path");
/**
 * A RdfStreamIncluder takes a triple stream and detects owl:includes to automatically include other files.
 */
class RdfStreamIncluder extends stream_1.PassThrough {
    constructor(constants, fromPath, followImports, absolutizeRelativePaths, contexts, importPaths) {
        super({ objectMode: true });
        this._runningImporters = 1;
        this._readableState.objectMode = true;
        this._constants = constants;
        this._fromPath = fromPath;
        this._followImports = followImports;
        this._absolutizeRelativePaths = absolutizeRelativePaths;
        this._contexts = contexts;
        this._importPaths = importPaths;
    }
    push(data, encoding) {
        if (data) {
            if (this._followImports && data.predicate === this._constants.PREFIXES['owl'] + 'imports') {
                this._runningImporters++;
                var relativeFilePath = data.object;
                // Try overriding path using defined import paths
                if (this._importPaths) {
                    for (const prefix of Object.keys(this._importPaths)) {
                        if (relativeFilePath.startsWith(prefix)) {
                            relativeFilePath = this._importPaths[prefix] + relativeFilePath.substr(prefix.length);
                            break;
                        }
                    }
                }
                this._constants.getContentsFromUrlOrPath(relativeFilePath, this._fromPath)
                    .then((rawStream) => {
                    let data = this._constants.parseRdf(rawStream, null, this._fromPath, true, this._absolutizeRelativePaths, this._contexts, this._importPaths);
                    data.on('data', (subData) => this.push(subData))
                        .on('error', (e) => this.emit('error', require("../Util").addFilePathToError(e, relativeFilePath, this._fromPath)))
                        .on('end', () => this.push(null));
                }).catch((e) => this.emit('error', require("../Util").addFilePathToError(e, relativeFilePath, this._fromPath)));
            }
            if (this._absolutizeRelativePaths) {
                data.subject = this._absolutize(data.subject);
                data.predicate = this._absolutize(data.predicate);
                data.object = this._absolutize(data.object);
            }
            return super.push(data);
        }
        else if (!--this._runningImporters) {
            super.push(null);
        }
    }
    _absolutize(uri) {
        // Make relative paths absolute
        var match = RdfStreamIncluder.RELATIVE_PATH_MATCHER.exec(uri);
        if (match) {
            return '"file:///' + Path.join(this._fromPath, match[1]) + '"' + this._constants.PREFIXES['xsd'] + 'string';
        }
        return uri;
    }
}
RdfStreamIncluder.RELATIVE_PATH_MATCHER = /^"file:\/\/([^\/].*)".*$/;
exports.RdfStreamIncluder = RdfStreamIncluder;
//# sourceMappingURL=RdfStreamIncluder.js.map