/// <reference types="node" />
import * as http from "http";
import { Writable } from "stream";
import { ActorInitSparql } from "./ActorInitSparql";
import { IQueryOptions } from "./QueryDynamic";
/**
 * An HTTP service that exposes a Comunica engine as a SPARQL endpoint.
 */
export declare class HttpServiceSparqlEndpoint {
    private static readonly MIME_PLAIN;
    private static readonly MIME_JSON;
    private readonly context;
    private readonly timeout;
    private readonly port;
    private readonly invalidateCacheBeforeQuery;
    private readonly engine;
    constructor(args?: IHttpServiceSparqlEndpointArgs);
    static runArgsInProcess(moduleRootPath: string, defaultConfigPath: string): void;
    /**
     * Start the HTTP service.
     * @param {module:stream.internal.Writable} stdout The output stream to log to.
     * @param {module:stream.internal.Writable} stderr The error stream to log errors to.
     */
    run(stdout: Writable, stderr: Writable): Promise<void>;
    /**
     * Handles an HTTP request.
     * @param {ActorInitSparql} engine A SPARQL engine.
     * @param {{type: string; quality: number}[]} variants Allowed variants.
     * @param {module:stream.internal.Writable} stdout Output stream.
     * @param {module:stream.internal.Writable} stderr Error output stream.
     * @param {module:http.IncomingMessage} request Request object.
     * @param {module:http.ServerResponse} response Response object.
     */
    protected handleRequest(engine: ActorInitSparql, variants: {
        type: string;
        quality: number;
    }[], stdout: Writable, stderr: Writable, request: http.IncomingMessage, response: http.ServerResponse): Promise<void>;
    /**
     * Writes the result of the given SPARQL query.
     * @param {ActorInitSparql} engine A SPARQL engine.
     * @param {module:stream.internal.Writable} stdout Output stream.
     * @param {module:stream.internal.Writable} stderr Error output stream.
     * @param {module:http.IncomingMessage} request Request object.
     * @param {module:http.ServerResponse} response Response object.
     * @param {string} sparql The SPARQL query string.
     * @param {string} mediaType The requested response media type.
     * @param {boolean} headOnly If only the header should be written.
     */
    protected writeQueryResult(engine: ActorInitSparql, stdout: Writable, stderr: Writable, request: http.IncomingMessage, response: http.ServerResponse, sparql: string, mediaType: string, headOnly: boolean): void;
    /**
     * Parses the body of a SPARQL POST request
     * @param {module:http.IncomingMessage} request Request object.
     * @return {Promise<string>} A promise resolving to a query string.
     */
    protected parseBody(request: http.IncomingMessage): Promise<string>;
}
export interface IHttpServiceSparqlEndpointArgs extends IQueryOptions {
    context?: any;
    timeout?: number;
    port?: number;
    invalidateCacheBeforeQuery?: boolean;
}
