import { ActionContext, Actor, IAction, IActorArgs, IActorTest } from "@comunica/core";
import { AsyncIterator } from "asynciterator";
import * as RDF from "rdf-js";
import { Algebra } from "sparqlalgebrajs";
import { BindingsStream } from "./Bindings";
/**
 * A comunica actor for query-operation events.
 *
 * Actor types:
 * * Input:  IActionQueryOperation:      A SPARQL Algebra operation.
 * * Test:   <none>
 * * Output: IActorQueryOperationOutput: A bindings stream.
 *
 * @see IActionQueryOperation
 * @see IActorQueryOperationOutput
 */
export declare abstract class ActorQueryOperation extends Actor<IActionQueryOperation, IActorTest, IActorQueryOperationOutput> {
    constructor(args: IActorArgs<IActionQueryOperation, IActorTest, IActorQueryOperationOutput>);
    /**
     * Safely cast a query operation output to a bindings output.
     * This will throw a runtime error if the output is of the incorrect type.
     * @param {IActorQueryOperationOutput} output A query operation output.
     * @return {IActorQueryOperationOutputBindings} A bindings query operation output.
     */
    static getSafeBindings(output: IActorQueryOperationOutput): IActorQueryOperationOutputBindings;
    /**
     * Safely cast a query operation output to a quads output.
     * This will throw a runtime error if the output is of the incorrect type.
     * @param {IActorQueryOperationOutput} output A query operation output.
     * @return {IActorQueryOperationOutputQuads} A quads query operation output.
     */
    static getSafeQuads(output: IActorQueryOperationOutput): IActorQueryOperationOutputQuads;
    /**
     * Safely cast a query operation output to a boolean output.
     * This will throw a runtime error if the output is of the incorrect type.
     * @param {IActorQueryOperationOutput} output A query operation output.
     * @return {IActorQueryOperationOutputBoolean} A boolean query operation output.
     */
    static getSafeBoolean(output: IActorQueryOperationOutput): IActorQueryOperationOutputBoolean;
    /**
     * Convert a metadata callback to a lazy callback where the response value is cached.
     * @param {() => Promise<{[p: string]: any}>} metadata A metadata callback
     * @return {() => Promise<{[p: string]: any}>} The callback where the response will be cached.
     */
    static cachifyMetadata(metadata: () => Promise<{
        [id: string]: any;
    }>): () => Promise<{
        [id: string]: any;
    }>;
    /**
     * Throw an error if the output type does not match the expected type.
     * @param {IActorQueryOperationOutput} output A query operation output.
     * @param {string} expectedType The expected output type.
     */
    static validateQueryOutput(output: IActorQueryOperationOutput, expectedType: string): void;
    static getExpressionContext(context: ActionContext): IExpressionContext;
}
export interface IExpressionContext {
    now?: Date;
    baseIRI?: string;
}
export interface IActionQueryOperation extends IAction {
    /**
     * The query operation to handle.
     */
    operation: Algebra.Operation;
}
/**
 * Query operation output.
 * @see IActorQueryOperationOutputBindings, IActorQueryOperationOutputQuads, IActorQueryOperationOutputBoolean
 */
export interface IActorQueryOperationOutput {
    /**
     * The type of output.
     */
    type: string;
    /**
     * The resulting action context.
     */
    context?: ActionContext;
}
/**
 * Super interface for query operation outputs that represent some for of stream.
 * @see IActorQueryOperationOutputBindings, IActorQueryOperationOutputQuads
 */
export interface IActorQueryOperationOutputStream extends IActorQueryOperationOutput {
    /**
     * Callback that returns a promise that resolves to the metadata about the stream.
     * This can contain things like the estimated number of total stream elements,
     * or the order in which the bindings appear.
     * This callback can be invoked multiple times.
     * The actors that return this metadata will make sure that multiple calls properly cache this promise.
     * Metadata will not be collected until this callback is invoked.
     */
    metadata?: () => Promise<{
        [id: string]: any;
    }>;
}
/**
 * Query operation output for a bindings stream.
 * For example: SPARQL SELECT results
 */
export interface IActorQueryOperationOutputBindings extends IActorQueryOperationOutputStream {
    /**
     * The type of output.
     */
    type: 'bindings';
    /**
     * The stream of bindings resulting from the given operation.
     */
    bindingsStream: BindingsStream;
    /**
     * The list of variable names (without '?') for which bindings are provided in the stream.
     */
    variables: string[];
}
/**
 * Query operation output for quads.
 * For example: SPARQL CONSTRUCT results
 */
export interface IActorQueryOperationOutputQuads extends IActorQueryOperationOutputStream {
    /**
     * The type of output.
     */
    type: 'quads';
    /**
     * The stream of quads.
     */
    quadStream: RDF.Stream & AsyncIterator<RDF.Quad>;
}
/**
 * Query operation output for quads.
 * For example: SPARQL ASK results
 */
export interface IActorQueryOperationOutputBoolean extends IActorQueryOperationOutput {
    /**
     * The type of output.
     */
    type: 'boolean';
    /**
     * A promise resolving to the boolean output of the operation.
     */
    booleanResult: Promise<boolean>;
}
/**
 * @type {string} Context entry for current metadata.
 *                I.e., the metadata that was used to determine the next BGP operation.
 * @value {any} A metadata hash.
 */
export declare const KEY_CONTEXT_BGP_CURRENTMETADATA: string;
/**
 * @type {string} Context entry for an array of parent metadata.
 *                I.e., an array of the metadata that was present before materializing the current BGP operations.
 *                This can be passed in 'bgp' actions.
 *                The array entries should correspond to the pattern entries in the BGP.
 * @value {any} An array of metadata hashes.
 */
export declare const KEY_CONTEXT_BGP_PARENTMETADATA: string;
/**
 * @type {string} Context entry for indicating which patterns were bound from variables.
 *                I.e., an array of the same length as the value of KEY_CONTEXT_BGP_PARENTMETADATA,
 *                where each array value corresponds to the pattern bindings for the corresponding pattern.
 * @value {any} An array of {@link IPatternBindings}.
 */
export declare const KEY_CONTEXT_BGP_PATTERNBINDINGS: string;
/**
 * @type {string} Context entry for parent metadata.
 *                I.e., the metadata that was present before materializing the current operation.
 *                This can be passed in 'pattern' actions.
 * @value {any} A metadata hash.
 */
export declare const KEY_CONTEXT_PATTERN_PARENTMETADATA: string;
/**
 * @type {string} Context entry for query's base IRI.
 * @value {any} A string.
 */
export declare const KEY_CONTEXT_BASEIRI: string;
/**
 * @type {string} A timestamp representing the current time.
 *                This is required for certain SPARQL operations such as NOW().
 * @value {any} a date.
 */
export declare const KEY_CONTEXT_QUERY_TIMESTAMP: string;
/**
 * Binds a quad pattern term's position to a variable.
 */
export interface IPatternBindings {
    [termPosition: string]: RDF.Variable;
}
