"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@comunica/core");
const asynciterator_1 = require("asynciterator");
/**
 * A comunica actor for joining 2 binding streams.
 *
 * Actor types:
 * * Input:  IActionRdfJoin:      The streams that need to be joined.
 * * Test:   <none>
 * * Output: IActorRdfJoinOutput: The resulting joined stream.
 *
 * @see IActionRdfJoin
 * @see IActorQueryOperationOutput
 */
class ActorRdfJoin extends core_1.Actor {
    constructor(args, maxEntries) {
        super(args);
        this.maxEntries = maxEntries;
    }
    /**
     * Returns an array containing all the variable names that occur in all bindings streams.
     * @param {IActionRdfJoin} action
     * @returns {string[]}
     */
    static overlappingVariables(action) {
        return require('lodash.intersection').apply(this, action.entries.map((entry) => entry.variables));
    }
    /**
     * Returns the variables that will occur in the joined bindings.
     * @param {IActionRdfJoin} action
     * @returns {string[]}
     */
    static joinVariables(action) {
        return require('lodash.union').apply(this, action.entries.map((entry) => entry.variables));
    }
    /**
     * Returns the result of joining bindings, or `null` if no join is possible.
     * @param {Bindings[]} bindings
     * @returns {Bindings}
     */
    static join(...bindings) {
        try {
            return bindings.reduce((acc, val) => acc.mergeWith((l, r) => {
                if (!l.equals(r)) {
                    throw new Error();
                }
                return l;
            }, val));
        }
        catch (e) {
            return null;
        }
    }
    /**
     * Checks if all metadata objects are present in the action, and if they have the specified key.
     * @param {IActionRdfJoin} action
     * @param {string} key
     * @returns {boolean}
     */
    static async iteratorsHaveMetadata(action, key) {
        return Promise.all(action.entries.map(async (entry) => {
            const metadata = await entry.metadata();
            if (!metadata.hasOwnProperty(key)) {
                throw new Error();
            }
        })).then(() => true).catch(() => false);
    }
    /**
     * Default test function for join actors.
     * Checks whether all iterators have metadata.
     * If yes: call the abstract getIterations method, if not: return Infinity.
     * @param {IActionRdfJoin} action The input action containing the relevant iterators
     * @returns {Promise<IMediatorTypeIterations>} The calculated estime.
     */
    async test(action) {
        if (action.entries.length <= 1) {
            return { iterations: 0 };
        }
        if (this.maxEntries && action.entries.length > this.maxEntries) {
            throw new Error(this.name + ' supports ' + this.maxEntries
                + ' sources at most. The input contained ' + action.entries.length + '.');
        }
        for (const entry of action.entries) {
            if (entry.type !== 'bindings') {
                throw new Error('Invalid type of a join entry: Expected \'bindings\' but got \'' + entry.type + '\'');
            }
        }
        if (!(await ActorRdfJoin.iteratorsHaveMetadata(action, 'totalItems'))) {
            return { iterations: Infinity };
        }
        return { iterations: await this.getIterations(action) };
    }
    /**
     * Returns default input for 0 or 1 entries. Calls the getOutput function otherwise
     * @param {IActionRdfJoin} action
     * @returns {Promise<IActorQueryOperationOutput>}
     */
    async run(action) {
        if (action.entries.length === 0) {
            return {
                bindingsStream: new asynciterator_1.EmptyIterator(),
                metadata: () => Promise.resolve({ totalItems: 0 }),
                type: 'bindings',
                variables: [],
            };
        }
        if (action.entries.length === 1) {
            return action.entries[0];
        }
        const result = this.getOutput(action);
        if (await ActorRdfJoin.iteratorsHaveMetadata(action, 'totalItems')) {
            const totalItems = () => Promise.all(action.entries
                .map((entry) => entry.metadata()))
                .then((metadatas) => metadatas.reduce((acc, val) => acc * val.totalItems, 1));
            // update the result promise to also add the estimated total items
            const unwrapped = await result;
            if (unwrapped.metadata) {
                const oldMetadata = unwrapped.metadata;
                unwrapped.metadata = () => oldMetadata().then(async (metadata) => {
                    // don't overwrite metadata if it was generated by implementation
                    if (!metadata.hasOwnProperty('totalItems')) {
                        metadata.totalItems = await totalItems();
                    }
                    return metadata;
                });
            }
            else {
                unwrapped.metadata = () => totalItems().then((t) => ({ totalItems: t }));
            }
            return unwrapped;
        }
        return result;
    }
}
exports.ActorRdfJoin = ActorRdfJoin;
//# sourceMappingURL=ActorRdfJoin.js.map