"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * A Runner is used to instantiate a comunica workflow.
 *
 * It is instantiated dynamically using a Components.js config file.
 * A bus and a list of actors are injected into this runner dynamically.
 *
 * The {@link Runner#run} function must be called to instantiate the workflow.
 */
class Runner {
    constructor(args) {
        require('lodash.assign')(this, args);
        if (!this.busInit) {
            throw new Error('A valid "busInit" argument must be provided.');
        }
        if (!this.actors) {
            throw new Error('A valid "actors" argument must be provided.');
        }
    }
    /**
     * Run a comunica workflow.
     *
     * @param {IActionInit} action An 'init' action.
     * @return {Promise<void>}     A promise that resolves when the init actors are triggered.
     */
    async run(action) {
        const replies = await Promise.all(this.busInit.publish(action));
        return Promise.all(replies.map((reply) => reply.actor.runObservable(action)));
    }
    /**
     * Initialize the actors.
     * This should be used for doing things that take a while,
     * such as opening files.
     *
     * @return {Promise<void>} A promise that resolves when the actors have been initialized.
     */
    initialize() {
        return Promise.all(this.actors.map((actor) => actor.initialize())).then(() => true);
    }
    /**
     * Deinitialize the actors.
     * This should be used for cleaning up things when the application is shut down,
     * such as closing files and removing temporary files.
     *
     * @return {Promise<void>} A promise that resolves when the actors have been deinitialized.
     */
    async deinitialize() {
        return Promise.all(this.actors.map((actor) => actor.deinitialize())).then(() => true);
    }
}
exports.Runner = Runner;
//# sourceMappingURL=Runner.js.map