"use strict";
var __values = (this && this.__values) || function(o) {
    var s = typeof Symbol === "function" && Symbol.iterator, m = s && o[s], i = 0;
    if (m) return m.call(o);
    if (o && typeof o.length === "number") return {
        next: function () {
            if (o && i >= o.length) o = void 0;
            return { value: o && o[i++], done: !o };
        }
    };
    throw new TypeError(s ? "Object is not iterable." : "Symbol.iterator is not defined.");
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
var AstNode_1 = __importDefault(require("./AstNode"));
var messages_1 = require("@cucumber/messages");
var Parser_1 = require("./Parser");
var Errors_1 = require("./Errors");
var createLocation_1 = __importDefault(require("./cli/createLocation"));
var AstBuilder = /** @class */ (function () {
    function AstBuilder(newId) {
        this.newId = newId;
        if (!newId) {
            throw new Error('No newId');
        }
        this.reset();
    }
    AstBuilder.prototype.reset = function () {
        this.stack = [new AstNode_1.default(Parser_1.RuleType.None)];
        this.comments = [];
    };
    AstBuilder.prototype.startRule = function (ruleType) {
        this.stack.push(new AstNode_1.default(ruleType));
    };
    AstBuilder.prototype.endRule = function () {
        var node = this.stack.pop();
        var transformedNode = this.transformNode(node);
        this.currentNode().add(node.ruleType, transformedNode);
    };
    AstBuilder.prototype.build = function (token) {
        if (token.matchedType === Parser_1.TokenType.Comment) {
            this.comments.push(messages_1.messages.GherkinDocument.Comment.create({
                location: this.getLocation(token),
                text: token.matchedText,
            }));
        }
        else {
            this.currentNode().add(token.matchedType, token);
        }
    };
    AstBuilder.prototype.getResult = function () {
        return this.currentNode().getSingle(Parser_1.RuleType.GherkinDocument);
    };
    AstBuilder.prototype.currentNode = function () {
        return this.stack[this.stack.length - 1];
    };
    AstBuilder.prototype.getLocation = function (token, column) {
        return !column
            ? token.location
            : createLocation_1.default({ line: token.location.line, column: column });
    };
    AstBuilder.prototype.getTags = function (node) {
        var e_1, _a, e_2, _b;
        var tags = [];
        var tagsNode = node.getSingle(Parser_1.RuleType.Tags);
        if (!tagsNode) {
            return tags;
        }
        var tokens = tagsNode.getTokens(Parser_1.TokenType.TagLine);
        try {
            for (var tokens_1 = __values(tokens), tokens_1_1 = tokens_1.next(); !tokens_1_1.done; tokens_1_1 = tokens_1.next()) {
                var token = tokens_1_1.value;
                try {
                    for (var _c = (e_2 = void 0, __values(token.matchedItems)), _d = _c.next(); !_d.done; _d = _c.next()) {
                        var tagItem = _d.value;
                        tags.push(messages_1.messages.GherkinDocument.Feature.Tag.create({
                            location: this.getLocation(token, tagItem.column),
                            name: tagItem.text,
                            id: this.newId(),
                        }));
                    }
                }
                catch (e_2_1) { e_2 = { error: e_2_1 }; }
                finally {
                    try {
                        if (_d && !_d.done && (_b = _c.return)) _b.call(_c);
                    }
                    finally { if (e_2) throw e_2.error; }
                }
            }
        }
        catch (e_1_1) { e_1 = { error: e_1_1 }; }
        finally {
            try {
                if (tokens_1_1 && !tokens_1_1.done && (_a = tokens_1.return)) _a.call(tokens_1);
            }
            finally { if (e_1) throw e_1.error; }
        }
        return tags;
    };
    AstBuilder.prototype.getCells = function (tableRowToken) {
        var _this = this;
        return tableRowToken.matchedItems.map(function (cellItem) {
            return messages_1.messages.GherkinDocument.Feature.TableRow.TableCell.create({
                location: _this.getLocation(tableRowToken, cellItem.column),
                value: cellItem.text,
            });
        });
    };
    AstBuilder.prototype.getDescription = function (node) {
        return node.getSingle(Parser_1.RuleType.Description);
    };
    AstBuilder.prototype.getSteps = function (node) {
        return node.getItems(Parser_1.RuleType.Step);
    };
    AstBuilder.prototype.getTableRows = function (node) {
        var _this = this;
        var rows = node.getTokens(Parser_1.TokenType.TableRow).map(function (token) {
            return messages_1.messages.GherkinDocument.Feature.TableRow.create({
                id: _this.newId(),
                location: _this.getLocation(token),
                cells: _this.getCells(token),
            });
        });
        this.ensureCellCount(rows);
        return rows;
    };
    AstBuilder.prototype.ensureCellCount = function (rows) {
        if (rows.length === 0) {
            return;
        }
        var cellCount = rows[0].cells.length;
        rows.forEach(function (row) {
            if (row.cells.length !== cellCount) {
                throw Errors_1.AstBuilderException.create('inconsistent cell count within the table', row.location);
            }
        });
    };
    AstBuilder.prototype.transformNode = function (node) {
        var e_3, _a, e_4, _b, e_5, _c;
        switch (node.ruleType) {
            case Parser_1.RuleType.Step: {
                var stepLine = node.getToken(Parser_1.TokenType.StepLine);
                var dataTable = node.getSingle(Parser_1.RuleType.DataTable);
                var docString = node.getSingle(Parser_1.RuleType.DocString);
                return messages_1.messages.GherkinDocument.Feature.Step.create({
                    id: this.newId(),
                    location: this.getLocation(stepLine),
                    keyword: stepLine.matchedKeyword,
                    text: stepLine.matchedText,
                    dataTable: dataTable,
                    docString: docString,
                });
            }
            case Parser_1.RuleType.DocString: {
                var separatorToken = node.getTokens(Parser_1.TokenType.DocStringSeparator)[0];
                var mediaType = separatorToken.matchedText.length > 0
                    ? separatorToken.matchedText
                    : undefined;
                var lineTokens = node.getTokens(Parser_1.TokenType.Other);
                var content = lineTokens.map(function (t) { return t.matchedText; }).join('\n');
                var result = messages_1.messages.GherkinDocument.Feature.Step.DocString.create({
                    location: this.getLocation(separatorToken),
                    content: content,
                    delimiter: separatorToken.line.trimmedLineText.substring(0, 3),
                });
                // conditionally add this like this (needed to make tests pass on node 0.10 as well as 4.0)
                if (mediaType) {
                    result.mediaType = mediaType;
                }
                return result;
            }
            case Parser_1.RuleType.DataTable: {
                var rows = this.getTableRows(node);
                return messages_1.messages.GherkinDocument.Feature.Step.DataTable.create({
                    location: rows[0].location,
                    rows: rows,
                });
            }
            case Parser_1.RuleType.Background: {
                var backgroundLine = node.getToken(Parser_1.TokenType.BackgroundLine);
                var description = this.getDescription(node);
                var steps = this.getSteps(node);
                return messages_1.messages.GherkinDocument.Feature.Background.create({
                    location: this.getLocation(backgroundLine),
                    keyword: backgroundLine.matchedKeyword,
                    name: backgroundLine.matchedText,
                    description: description,
                    steps: steps,
                });
            }
            case Parser_1.RuleType.ScenarioDefinition: {
                var tags = this.getTags(node);
                var scenarioNode = node.getSingle(Parser_1.RuleType.Scenario);
                var scenarioLine = scenarioNode.getToken(Parser_1.TokenType.ScenarioLine);
                var description = this.getDescription(scenarioNode);
                var steps = this.getSteps(scenarioNode);
                var examples = scenarioNode.getItems(Parser_1.RuleType.ExamplesDefinition);
                return messages_1.messages.GherkinDocument.Feature.Scenario.create({
                    id: this.newId(),
                    tags: tags,
                    location: this.getLocation(scenarioLine),
                    keyword: scenarioLine.matchedKeyword,
                    name: scenarioLine.matchedText,
                    description: description,
                    steps: steps,
                    examples: examples,
                });
            }
            case Parser_1.RuleType.ExamplesDefinition: {
                var tags = this.getTags(node);
                var examplesNode = node.getSingle(Parser_1.RuleType.Examples);
                var examplesLine = examplesNode.getToken(Parser_1.TokenType.ExamplesLine);
                var description = this.getDescription(examplesNode);
                var exampleTable = examplesNode.getSingle(Parser_1.RuleType.ExamplesTable);
                return messages_1.messages.GherkinDocument.Feature.Scenario.Examples.create({
                    tags: tags,
                    location: this.getLocation(examplesLine),
                    keyword: examplesLine.matchedKeyword,
                    name: examplesLine.matchedText,
                    description: description,
                    tableHeader: exampleTable !== undefined ? exampleTable[0] : undefined,
                    tableBody: exampleTable !== undefined ? exampleTable.slice(1) : undefined,
                });
            }
            case Parser_1.RuleType.ExamplesTable: {
                return this.getTableRows(node);
            }
            case Parser_1.RuleType.Description: {
                var lineTokens = node.getTokens(Parser_1.TokenType.Other);
                // Trim trailing empty lines
                var end = lineTokens.length;
                while (end > 0 && lineTokens[end - 1].line.trimmedLineText === '') {
                    end--;
                }
                lineTokens = lineTokens.slice(0, end);
                return lineTokens.map(function (token) { return token.matchedText; }).join('\n');
            }
            case Parser_1.RuleType.Feature: {
                var header = node.getSingle(Parser_1.RuleType.FeatureHeader);
                if (!header) {
                    return null;
                }
                var tags = this.getTags(header);
                var featureLine = header.getToken(Parser_1.TokenType.FeatureLine);
                if (!featureLine) {
                    return null;
                }
                var children = [];
                var background = node.getSingle(Parser_1.RuleType.Background);
                if (background) {
                    children.push(messages_1.messages.GherkinDocument.Feature.FeatureChild.create({
                        background: background,
                    }));
                }
                try {
                    for (var _d = __values(node.getItems(Parser_1.RuleType.ScenarioDefinition)), _e = _d.next(); !_e.done; _e = _d.next()) {
                        var scenario = _e.value;
                        children.push(messages_1.messages.GherkinDocument.Feature.FeatureChild.create({
                            scenario: scenario,
                        }));
                    }
                }
                catch (e_3_1) { e_3 = { error: e_3_1 }; }
                finally {
                    try {
                        if (_e && !_e.done && (_a = _d.return)) _a.call(_d);
                    }
                    finally { if (e_3) throw e_3.error; }
                }
                try {
                    for (var _f = __values(node.getItems(Parser_1.RuleType.Rule)), _g = _f.next(); !_g.done; _g = _f.next()) {
                        var rule = _g.value;
                        children.push(messages_1.messages.GherkinDocument.Feature.FeatureChild.create({
                            rule: rule,
                        }));
                    }
                }
                catch (e_4_1) { e_4 = { error: e_4_1 }; }
                finally {
                    try {
                        if (_g && !_g.done && (_b = _f.return)) _b.call(_f);
                    }
                    finally { if (e_4) throw e_4.error; }
                }
                var description = this.getDescription(header);
                var language = featureLine.matchedGherkinDialect;
                return messages_1.messages.GherkinDocument.Feature.create({
                    tags: tags,
                    location: this.getLocation(featureLine),
                    language: language,
                    keyword: featureLine.matchedKeyword,
                    name: featureLine.matchedText,
                    description: description,
                    children: children,
                });
            }
            case Parser_1.RuleType.Rule: {
                var header = node.getSingle(Parser_1.RuleType.RuleHeader);
                if (!header) {
                    return null;
                }
                var ruleLine = header.getToken(Parser_1.TokenType.RuleLine);
                if (!ruleLine) {
                    return null;
                }
                var children = [];
                var background = node.getSingle(Parser_1.RuleType.Background);
                if (background) {
                    children.push(messages_1.messages.GherkinDocument.Feature.FeatureChild.create({
                        background: background,
                    }));
                }
                try {
                    for (var _h = __values(node.getItems(Parser_1.RuleType.ScenarioDefinition)), _j = _h.next(); !_j.done; _j = _h.next()) {
                        var scenario = _j.value;
                        children.push(messages_1.messages.GherkinDocument.Feature.FeatureChild.create({
                            scenario: scenario,
                        }));
                    }
                }
                catch (e_5_1) { e_5 = { error: e_5_1 }; }
                finally {
                    try {
                        if (_j && !_j.done && (_c = _h.return)) _c.call(_h);
                    }
                    finally { if (e_5) throw e_5.error; }
                }
                var description = this.getDescription(header);
                return messages_1.messages.GherkinDocument.Feature.FeatureChild.Rule.create({
                    location: this.getLocation(ruleLine),
                    keyword: ruleLine.matchedKeyword,
                    name: ruleLine.matchedText,
                    description: description,
                    children: children,
                });
            }
            case Parser_1.RuleType.GherkinDocument: {
                var feature = node.getSingle(Parser_1.RuleType.Feature);
                return messages_1.messages.GherkinDocument.create({
                    feature: feature,
                    comments: this.comments,
                });
            }
            default:
                return node;
        }
    };
    return AstBuilder;
}());
exports.default = AstBuilder;
//# sourceMappingURL=AstBuilder.js.map