"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var messages_1 = require("@cucumber/messages");
function compile(gherkinDocument, uri, newId) {
    var pickles = [];
    if (gherkinDocument.feature == null) {
        return pickles;
    }
    var feature = gherkinDocument.feature;
    var language = feature.language;
    var featureTags = feature.tags;
    var featureBackgroundSteps = [];
    feature.children.forEach(function (stepsContainer) {
        if (stepsContainer.background) {
            featureBackgroundSteps = [].concat(stepsContainer.background.steps);
        }
        else if (stepsContainer.rule) {
            compileRule(featureTags, featureBackgroundSteps, stepsContainer.rule, language, pickles, uri, newId);
        }
        else if (stepsContainer.scenario.examples.length === 0) {
            compileScenario(featureTags, featureBackgroundSteps, stepsContainer.scenario, language, pickles, uri, newId);
        }
        else {
            compileScenarioOutline(featureTags, featureBackgroundSteps, stepsContainer.scenario, language, pickles, uri, newId);
        }
    });
    return pickles;
}
exports.default = compile;
function compileRule(featureTags, featureBackgroundSteps, rule, language, pickles, uri, newId) {
    var ruleBackgroundSteps = [].concat(featureBackgroundSteps);
    rule.children.forEach(function (stepsContainer) {
        if (stepsContainer.background) {
            ruleBackgroundSteps = ruleBackgroundSteps.concat(stepsContainer.background.steps);
        }
        else if (stepsContainer.scenario.examples.length === 0) {
            compileScenario(featureTags, ruleBackgroundSteps, stepsContainer.scenario, language, pickles, uri, newId);
        }
        else {
            compileScenarioOutline(featureTags, ruleBackgroundSteps, stepsContainer.scenario, language, pickles, uri, newId);
        }
    });
}
function compileScenario(featureTags, backgroundSteps, scenario, language, pickles, uri, newId) {
    var steps = scenario.steps.length === 0
        ? []
        : backgroundSteps.map(function (step) { return pickleStep(step, [], null, newId); });
    var tags = [].concat(featureTags).concat(scenario.tags);
    scenario.steps.forEach(function (step) { return steps.push(pickleStep(step, [], null, newId)); });
    var pickle = messages_1.messages.Pickle.create({
        id: newId(),
        uri: uri,
        astNodeIds: [scenario.id],
        tags: pickleTags(tags),
        name: scenario.name,
        language: language,
        steps: steps,
    });
    pickles.push(pickle);
}
function compileScenarioOutline(featureTags, backgroundSteps, scenario, language, pickles, uri, newId) {
    scenario.examples
        .filter(function (e) { return e.tableHeader !== null; })
        .forEach(function (examples) {
        var variableCells = examples.tableHeader.cells;
        examples.tableBody.forEach(function (valuesRow) {
            var steps = scenario.steps.length === 0
                ? []
                : backgroundSteps.map(function (step) { return pickleStep(step, [], null, newId); });
            var tags = []
                .concat(featureTags)
                .concat(scenario.tags)
                .concat(examples.tags);
            scenario.steps.forEach(function (scenarioOutlineStep) {
                var step = pickleStep(scenarioOutlineStep, variableCells, valuesRow, newId);
                steps.push(step);
            });
            pickles.push(messages_1.messages.Pickle.create({
                id: newId(),
                uri: uri,
                astNodeIds: [scenario.id, valuesRow.id],
                name: interpolate(scenario.name, variableCells, valuesRow.cells),
                language: language,
                steps: steps,
                tags: pickleTags(tags),
            }));
        });
    });
}
function createPickleArguments(step, variableCells, valueCells) {
    if (step.dataTable) {
        var argument = step.dataTable;
        var table = {
            rows: argument.rows.map(function (row) {
                return {
                    cells: row.cells.map(function (cell) {
                        return {
                            location: cell.location,
                            value: interpolate(cell.value, variableCells, valueCells),
                        };
                    }),
                };
            }),
        };
        return {
            dataTable: table,
        };
    }
    else if (step.docString) {
        var argument = step.docString;
        var docString = messages_1.messages.PickleStepArgument.PickleDocString.create({
            content: interpolate(argument.content, variableCells, valueCells),
        });
        if (argument.mediaType) {
            docString.mediaType = interpolate(argument.mediaType, variableCells, valueCells);
        }
        return {
            docString: docString,
        };
    }
}
function interpolate(name, variableCells, valueCells) {
    variableCells.forEach(function (variableCell, n) {
        var valueCell = valueCells[n];
        var search = new RegExp('<' + variableCell.value + '>', 'g');
        // JS Specific - dollar sign needs to be escaped with another dollar sign
        // https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/String/replace#Specifying_a_string_as_a_parameter
        var replacement = valueCell.value.replace(new RegExp('\\$', 'g'), '$$$$');
        name = name.replace(search, replacement);
    });
    return name;
}
function pickleStep(step, variableCells, valuesRow, newId) {
    var astNodeIds = [step.id];
    if (valuesRow) {
        astNodeIds.push(valuesRow.id);
    }
    var valueCells = valuesRow ? valuesRow.cells : [];
    return messages_1.messages.Pickle.PickleStep.create({
        id: newId(),
        text: interpolate(step.text, variableCells, valueCells),
        argument: createPickleArguments(step, variableCells, valueCells),
        astNodeIds: astNodeIds,
    });
}
function pickleTags(tags) {
    return tags.map(pickleTag);
}
function pickleTag(tag) {
    return messages_1.messages.Pickle.PickleTag.create({
        name: tag.name,
        astNodeId: tag.id,
    });
}
//# sourceMappingURL=compile.js.map