// Copyright (c) 2022 Roberto Raggi <roberto.raggi@gmail.com>
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
// SOFTWARE.
import { cxx } from "./cxx.js";
import { ASTCursor } from "./ASTCursor.js";
import { Token } from "./Token.js";
export class AST {
    handle;
    kind;
    parser;
    constructor(handle, kind, parser) {
        this.handle = handle;
        this.kind = kind;
        this.parser = parser;
    }
    walk() {
        return new ASTCursor(this, this.parser);
    }
    getKind() {
        return this.kind;
    }
    is(kind) {
        return this.kind === kind;
    }
    isNot(kind) {
        return this.kind !== kind;
    }
    getHandle() {
        return this.handle;
    }
    getStartLocation() {
        return cxx.getStartLocation(this.handle, this.parser.getUnitHandle());
    }
    getEndLocation() {
        return cxx.getEndLocation(this.handle, this.parser.getUnitHandle());
    }
    static from(handle, parser) {
        if (handle) {
            const kind = cxx.getASTKind(handle);
            const ast = new AST_CONSTRUCTORS[kind](handle, kind, parser);
            return ast;
        }
        return;
    }
}
export class AttributeAST extends AST {
}
export class CoreDeclaratorAST extends AST {
}
export class DeclarationAST extends AST {
}
export class DeclaratorModifierAST extends AST {
}
export class ExceptionDeclarationAST extends AST {
}
export class ExpressionAST extends AST {
}
export class FunctionBodyAST extends AST {
}
export class InitializerAST extends AST {
}
export class LambdaCaptureAST extends AST {
}
export class MemInitializerAST extends AST {
}
export class NameAST extends AST {
}
export class NewInitializerAST extends AST {
}
export class PtrOperatorAST extends AST {
}
export class RequirementAST extends AST {
}
export class SpecifierAST extends AST {
}
export class StatementAST extends AST {
}
export class TemplateArgumentAST extends AST {
}
export class UnitAST extends AST {
}
export class TypeIdAST extends AST {
    accept(visitor, context) {
        return visitor.visitTypeId(this, context);
    }
    *getTypeSpecifierList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 0); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getDeclarator() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
}
export class NestedNameSpecifierAST extends AST {
    accept(visitor, context) {
        return visitor.visitNestedNameSpecifier(this, context);
    }
    getScopeToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    *getNameList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 1); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
}
export class UsingDeclaratorAST extends AST {
    accept(visitor, context) {
        return visitor.visitUsingDeclarator(this, context);
    }
    getTypenameToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getNestedNameSpecifier() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getName() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
}
export class HandlerAST extends AST {
    accept(visitor, context) {
        return visitor.visitHandler(this, context);
    }
    getCatchToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getLparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getExceptionDeclaration() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getRparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
    getStatement() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 4), this.parser);
    }
}
export class EnumBaseAST extends AST {
    accept(visitor, context) {
        return visitor.visitEnumBase(this, context);
    }
    getColonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    *getTypeSpecifierList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 1); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
}
export class EnumeratorAST extends AST {
    accept(visitor, context) {
        return visitor.visitEnumerator(this, context);
    }
    getName() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    *getAttributeList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 1); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getEqualToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
}
export class DeclaratorAST extends AST {
    accept(visitor, context) {
        return visitor.visitDeclarator(this, context);
    }
    *getPtrOpList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 0); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getCoreDeclarator() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    *getModifiers() {
        for (let it = cxx.getASTSlot(this.getHandle(), 2); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
}
export class InitDeclaratorAST extends AST {
    accept(visitor, context) {
        return visitor.visitInitDeclarator(this, context);
    }
    getDeclarator() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getRequiresClause() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getInitializer() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
}
export class BaseSpecifierAST extends AST {
    accept(visitor, context) {
        return visitor.visitBaseSpecifier(this, context);
    }
    *getAttributeList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 0); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getName() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
}
export class BaseClauseAST extends AST {
    accept(visitor, context) {
        return visitor.visitBaseClause(this, context);
    }
    getColonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    *getBaseSpecifierList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 1); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
}
export class NewTypeIdAST extends AST {
    accept(visitor, context) {
        return visitor.visitNewTypeId(this, context);
    }
    *getTypeSpecifierList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 0); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
}
export class RequiresClauseAST extends AST {
    accept(visitor, context) {
        return visitor.visitRequiresClause(this, context);
    }
    getRequiresToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
}
export class ParameterDeclarationClauseAST extends AST {
    accept(visitor, context) {
        return visitor.visitParameterDeclarationClause(this, context);
    }
    *getParameterDeclarationList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 0); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getCommaToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getEllipsisToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
}
export class ParametersAndQualifiersAST extends AST {
    accept(visitor, context) {
        return visitor.visitParametersAndQualifiers(this, context);
    }
    getLparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getParameterDeclarationClause() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getRparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    *getCvQualifierList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 3); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getRefToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 4), this.parser);
    }
    *getAttributeList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 5); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
}
export class LambdaIntroducerAST extends AST {
    accept(visitor, context) {
        return visitor.visitLambdaIntroducer(this, context);
    }
    getLbracketToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getCaptureDefaultToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    *getCaptureList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 2); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getRbracketToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
}
export class LambdaDeclaratorAST extends AST {
    accept(visitor, context) {
        return visitor.visitLambdaDeclarator(this, context);
    }
    getLparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getParameterDeclarationClause() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getRparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    *getDeclSpecifierList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 3); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    *getAttributeList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 4); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getTrailingReturnType() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 5), this.parser);
    }
    getRequiresClause() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 6), this.parser);
    }
}
export class TrailingReturnTypeAST extends AST {
    accept(visitor, context) {
        return visitor.visitTrailingReturnType(this, context);
    }
    getMinusGreaterToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getTypeId() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
}
export class CtorInitializerAST extends AST {
    accept(visitor, context) {
        return visitor.visitCtorInitializer(this, context);
    }
    getColonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    *getMemInitializerList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 1); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
}
export class RequirementBodyAST extends AST {
    accept(visitor, context) {
        return visitor.visitRequirementBody(this, context);
    }
    getLbraceToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    *getRequirementList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 1); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getRbraceToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
}
export class TypeConstraintAST extends AST {
    accept(visitor, context) {
        return visitor.visitTypeConstraint(this, context);
    }
    getNestedNameSpecifier() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getName() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
}
export class GlobalModuleFragmentAST extends AST {
    accept(visitor, context) {
        return visitor.visitGlobalModuleFragment(this, context);
    }
    getModuleToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getSemicolonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    *getDeclarationList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 2); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
}
export class PrivateModuleFragmentAST extends AST {
    accept(visitor, context) {
        return visitor.visitPrivateModuleFragment(this, context);
    }
    getModuleToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getColonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getPrivateToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getSemicolonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
    *getDeclarationList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 4); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
}
export class ModuleDeclarationAST extends AST {
    accept(visitor, context) {
        return visitor.visitModuleDeclaration(this, context);
    }
    getExportToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getModuleToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getModuleName() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getModulePartition() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
    *getAttributeList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 4); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getSemicolonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 5), this.parser);
    }
}
export class ModuleNameAST extends AST {
    accept(visitor, context) {
        return visitor.visitModuleName(this, context);
    }
}
export class ImportNameAST extends AST {
    accept(visitor, context) {
        return visitor.visitImportName(this, context);
    }
    getHeaderToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getModulePartition() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getModuleName() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
}
export class ModulePartitionAST extends AST {
    accept(visitor, context) {
        return visitor.visitModulePartition(this, context);
    }
    getColonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getModuleName() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
}
export class SimpleRequirementAST extends RequirementAST {
    accept(visitor, context) {
        return visitor.visitSimpleRequirement(this, context);
    }
    getExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getSemicolonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
}
export class CompoundRequirementAST extends RequirementAST {
    accept(visitor, context) {
        return visitor.visitCompoundRequirement(this, context);
    }
    getLbraceToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getRbraceToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getNoexceptToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
    getMinusGreaterToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 4), this.parser);
    }
    getTypeConstraint() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 5), this.parser);
    }
    getSemicolonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 6), this.parser);
    }
}
export class TypeRequirementAST extends RequirementAST {
    accept(visitor, context) {
        return visitor.visitTypeRequirement(this, context);
    }
    getTypenameToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getNestedNameSpecifier() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getName() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getSemicolonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
}
export class NestedRequirementAST extends RequirementAST {
    accept(visitor, context) {
        return visitor.visitNestedRequirement(this, context);
    }
    getRequiresToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getSemicolonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
}
export class TypeTemplateArgumentAST extends TemplateArgumentAST {
    accept(visitor, context) {
        return visitor.visitTypeTemplateArgument(this, context);
    }
    getTypeId() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class ExpressionTemplateArgumentAST extends TemplateArgumentAST {
    accept(visitor, context) {
        return visitor.visitExpressionTemplateArgument(this, context);
    }
    getExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class ParenMemInitializerAST extends MemInitializerAST {
    accept(visitor, context) {
        return visitor.visitParenMemInitializer(this, context);
    }
    getName() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getLparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    *getExpressionList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 2); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getRparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
    getEllipsisToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 4), this.parser);
    }
}
export class BracedMemInitializerAST extends MemInitializerAST {
    accept(visitor, context) {
        return visitor.visitBracedMemInitializer(this, context);
    }
    getName() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getBracedInitList() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getEllipsisToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
}
export class ThisLambdaCaptureAST extends LambdaCaptureAST {
    accept(visitor, context) {
        return visitor.visitThisLambdaCapture(this, context);
    }
    getThisToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class DerefThisLambdaCaptureAST extends LambdaCaptureAST {
    accept(visitor, context) {
        return visitor.visitDerefThisLambdaCapture(this, context);
    }
    getStarToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getThisToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
}
export class SimpleLambdaCaptureAST extends LambdaCaptureAST {
    accept(visitor, context) {
        return visitor.visitSimpleLambdaCapture(this, context);
    }
    getIdentifierToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getEllipsisToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
}
export class RefLambdaCaptureAST extends LambdaCaptureAST {
    accept(visitor, context) {
        return visitor.visitRefLambdaCapture(this, context);
    }
    getAmpToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getIdentifierToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getEllipsisToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
}
export class RefInitLambdaCaptureAST extends LambdaCaptureAST {
    accept(visitor, context) {
        return visitor.visitRefInitLambdaCapture(this, context);
    }
    getAmpToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getEllipsisToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getIdentifierToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getInitializer() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
}
export class InitLambdaCaptureAST extends LambdaCaptureAST {
    accept(visitor, context) {
        return visitor.visitInitLambdaCapture(this, context);
    }
    getEllipsisToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getIdentifierToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getInitializer() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
}
export class EqualInitializerAST extends InitializerAST {
    accept(visitor, context) {
        return visitor.visitEqualInitializer(this, context);
    }
    getEqualToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
}
export class BracedInitListAST extends InitializerAST {
    accept(visitor, context) {
        return visitor.visitBracedInitList(this, context);
    }
    getLbraceToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    *getExpressionList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 1); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getCommaToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getRbraceToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
}
export class ParenInitializerAST extends InitializerAST {
    accept(visitor, context) {
        return visitor.visitParenInitializer(this, context);
    }
    getLparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    *getExpressionList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 1); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getRparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
}
export class NewParenInitializerAST extends NewInitializerAST {
    accept(visitor, context) {
        return visitor.visitNewParenInitializer(this, context);
    }
    getLparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    *getExpressionList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 1); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getRparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
}
export class NewBracedInitializerAST extends NewInitializerAST {
    accept(visitor, context) {
        return visitor.visitNewBracedInitializer(this, context);
    }
    getBracedInit() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class EllipsisExceptionDeclarationAST extends ExceptionDeclarationAST {
    accept(visitor, context) {
        return visitor.visitEllipsisExceptionDeclaration(this, context);
    }
    getEllipsisToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class TypeExceptionDeclarationAST extends ExceptionDeclarationAST {
    accept(visitor, context) {
        return visitor.visitTypeExceptionDeclaration(this, context);
    }
    *getAttributeList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 0); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    *getTypeSpecifierList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 1); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getDeclarator() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
}
export class DefaultFunctionBodyAST extends FunctionBodyAST {
    accept(visitor, context) {
        return visitor.visitDefaultFunctionBody(this, context);
    }
    getEqualToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getDefaultToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getSemicolonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
}
export class CompoundStatementFunctionBodyAST extends FunctionBodyAST {
    accept(visitor, context) {
        return visitor.visitCompoundStatementFunctionBody(this, context);
    }
    getCtorInitializer() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getStatement() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
}
export class TryStatementFunctionBodyAST extends FunctionBodyAST {
    accept(visitor, context) {
        return visitor.visitTryStatementFunctionBody(this, context);
    }
    getTryToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getCtorInitializer() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getStatement() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    *getHandlerList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 3); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
}
export class DeleteFunctionBodyAST extends FunctionBodyAST {
    accept(visitor, context) {
        return visitor.visitDeleteFunctionBody(this, context);
    }
    getEqualToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getDeleteToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getSemicolonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
}
export class TranslationUnitAST extends UnitAST {
    accept(visitor, context) {
        return visitor.visitTranslationUnit(this, context);
    }
    *getDeclarationList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 0); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
}
export class ModuleUnitAST extends UnitAST {
    accept(visitor, context) {
        return visitor.visitModuleUnit(this, context);
    }
    getGlobalModuleFragment() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getModuleDeclaration() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    *getDeclarationList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 2); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getPrivateModuleFragmentAST() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
}
export class ThisExpressionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitThisExpression(this, context);
    }
    getThisToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class CharLiteralExpressionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitCharLiteralExpression(this, context);
    }
    getLiteralToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class BoolLiteralExpressionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitBoolLiteralExpression(this, context);
    }
    getLiteralToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class IntLiteralExpressionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitIntLiteralExpression(this, context);
    }
    getLiteralToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class FloatLiteralExpressionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitFloatLiteralExpression(this, context);
    }
    getLiteralToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class NullptrLiteralExpressionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitNullptrLiteralExpression(this, context);
    }
    getLiteralToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class StringLiteralExpressionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitStringLiteralExpression(this, context);
    }
}
export class UserDefinedStringLiteralExpressionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitUserDefinedStringLiteralExpression(this, context);
    }
    getLiteralToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class IdExpressionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitIdExpression(this, context);
    }
    getName() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class RequiresExpressionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitRequiresExpression(this, context);
    }
    getRequiresToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getLparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getParameterDeclarationClause() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getRparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
    getRequirementBody() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 4), this.parser);
    }
}
export class NestedExpressionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitNestedExpression(this, context);
    }
    getLparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getRparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
}
export class RightFoldExpressionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitRightFoldExpression(this, context);
    }
    getLparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getOpToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getEllipsisToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
    getRparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 4), this.parser);
    }
}
export class LeftFoldExpressionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitLeftFoldExpression(this, context);
    }
    getLparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getEllipsisToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getOpToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
    getRparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 4), this.parser);
    }
}
export class FoldExpressionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitFoldExpression(this, context);
    }
    getLparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getLeftExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getOpToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getEllipsisToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
    getFoldOpToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 4), this.parser);
    }
    getRightExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 5), this.parser);
    }
    getRparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 6), this.parser);
    }
}
export class LambdaExpressionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitLambdaExpression(this, context);
    }
    getLambdaIntroducer() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getLessToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    *getTemplateParameterList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 2); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getGreaterToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
    getRequiresClause() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 4), this.parser);
    }
    getLambdaDeclarator() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 5), this.parser);
    }
    getStatement() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 6), this.parser);
    }
}
export class SizeofExpressionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitSizeofExpression(this, context);
    }
    getSizeofToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
}
export class SizeofTypeExpressionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitSizeofTypeExpression(this, context);
    }
    getSizeofToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getLparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getTypeId() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getRparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
}
export class SizeofPackExpressionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitSizeofPackExpression(this, context);
    }
    getSizeofToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getEllipsisToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getLparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getIdentifierToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
    getRparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 4), this.parser);
    }
}
export class TypeidExpressionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitTypeidExpression(this, context);
    }
    getTypeidToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getLparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getRparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
}
export class TypeidOfTypeExpressionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitTypeidOfTypeExpression(this, context);
    }
    getTypeidToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getLparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getTypeId() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getRparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
}
export class AlignofExpressionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitAlignofExpression(this, context);
    }
    getAlignofToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getLparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getTypeId() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getRparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
}
export class TypeTraitsExpressionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitTypeTraitsExpression(this, context);
    }
    getTypeTraitsToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getLparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    *getTypeIdList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 2); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getRparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
}
export class UnaryExpressionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitUnaryExpression(this, context);
    }
    getOpToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
}
export class BinaryExpressionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitBinaryExpression(this, context);
    }
    getLeftExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getOpToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getRightExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
}
export class AssignmentExpressionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitAssignmentExpression(this, context);
    }
    getLeftExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getOpToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getRightExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
}
export class BracedTypeConstructionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitBracedTypeConstruction(this, context);
    }
    getTypeSpecifier() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getBracedInitList() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
}
export class TypeConstructionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitTypeConstruction(this, context);
    }
    getTypeSpecifier() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getLparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    *getExpressionList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 2); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getRparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
}
export class CallExpressionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitCallExpression(this, context);
    }
    getBaseExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getLparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    *getExpressionList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 2); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getRparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
}
export class SubscriptExpressionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitSubscriptExpression(this, context);
    }
    getBaseExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getLbracketToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getIndexExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getRbracketToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
}
export class MemberExpressionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitMemberExpression(this, context);
    }
    getBaseExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getAccessToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getTemplateToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getName() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
}
export class PostIncrExpressionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitPostIncrExpression(this, context);
    }
    getBaseExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getOpToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
}
export class ConditionalExpressionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitConditionalExpression(this, context);
    }
    getCondition() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getQuestionToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getIftrueExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getColonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
    getIffalseExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 4), this.parser);
    }
}
export class ImplicitCastExpressionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitImplicitCastExpression(this, context);
    }
    getExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class CastExpressionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitCastExpression(this, context);
    }
    getLparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getTypeId() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getRparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
}
export class CppCastExpressionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitCppCastExpression(this, context);
    }
    getCastToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getLessToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getTypeId() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getGreaterToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
    getLparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 4), this.parser);
    }
    getExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 5), this.parser);
    }
    getRparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 6), this.parser);
    }
}
export class NewExpressionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitNewExpression(this, context);
    }
    getScopeToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getNewToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getTypeId() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getNewInitalizer() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
}
export class DeleteExpressionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitDeleteExpression(this, context);
    }
    getScopeToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getDeleteToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getLbracketToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getRbracketToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
    getExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 4), this.parser);
    }
}
export class ThrowExpressionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitThrowExpression(this, context);
    }
    getThrowToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
}
export class NoexceptExpressionAST extends ExpressionAST {
    accept(visitor, context) {
        return visitor.visitNoexceptExpression(this, context);
    }
    getNoexceptToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getLparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getRparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
}
export class LabeledStatementAST extends StatementAST {
    accept(visitor, context) {
        return visitor.visitLabeledStatement(this, context);
    }
    getIdentifierToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getColonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getStatement() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
}
export class CaseStatementAST extends StatementAST {
    accept(visitor, context) {
        return visitor.visitCaseStatement(this, context);
    }
    getCaseToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getColonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getStatement() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
}
export class DefaultStatementAST extends StatementAST {
    accept(visitor, context) {
        return visitor.visitDefaultStatement(this, context);
    }
    getDefaultToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getColonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getStatement() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
}
export class ExpressionStatementAST extends StatementAST {
    accept(visitor, context) {
        return visitor.visitExpressionStatement(this, context);
    }
    getExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getSemicolonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
}
export class CompoundStatementAST extends StatementAST {
    accept(visitor, context) {
        return visitor.visitCompoundStatement(this, context);
    }
    getLbraceToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    *getStatementList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 1); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getRbraceToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
}
export class IfStatementAST extends StatementAST {
    accept(visitor, context) {
        return visitor.visitIfStatement(this, context);
    }
    getIfToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getConstexprToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getLparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getInitializer() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
    getCondition() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 4), this.parser);
    }
    getRparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 5), this.parser);
    }
    getStatement() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 6), this.parser);
    }
    getElseStatement() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 7), this.parser);
    }
}
export class SwitchStatementAST extends StatementAST {
    accept(visitor, context) {
        return visitor.visitSwitchStatement(this, context);
    }
    getSwitchToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getLparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getInitializer() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getCondition() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
    getRparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 4), this.parser);
    }
    getStatement() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 5), this.parser);
    }
}
export class WhileStatementAST extends StatementAST {
    accept(visitor, context) {
        return visitor.visitWhileStatement(this, context);
    }
    getWhileToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getLparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getCondition() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getRparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
    getStatement() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 4), this.parser);
    }
}
export class DoStatementAST extends StatementAST {
    accept(visitor, context) {
        return visitor.visitDoStatement(this, context);
    }
    getDoToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getStatement() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getWhileToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getLparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
    getExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 4), this.parser);
    }
    getRparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 5), this.parser);
    }
    getSemicolonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 6), this.parser);
    }
}
export class ForRangeStatementAST extends StatementAST {
    accept(visitor, context) {
        return visitor.visitForRangeStatement(this, context);
    }
    getForToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getLparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getInitializer() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getRangeDeclaration() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
    getColonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 4), this.parser);
    }
    getRangeInitializer() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 5), this.parser);
    }
    getRparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 6), this.parser);
    }
    getStatement() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 7), this.parser);
    }
}
export class ForStatementAST extends StatementAST {
    accept(visitor, context) {
        return visitor.visitForStatement(this, context);
    }
    getForToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getLparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getInitializer() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getCondition() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
    getSemicolonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 4), this.parser);
    }
    getExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 5), this.parser);
    }
    getRparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 6), this.parser);
    }
    getStatement() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 7), this.parser);
    }
}
export class BreakStatementAST extends StatementAST {
    accept(visitor, context) {
        return visitor.visitBreakStatement(this, context);
    }
    getBreakToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getSemicolonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
}
export class ContinueStatementAST extends StatementAST {
    accept(visitor, context) {
        return visitor.visitContinueStatement(this, context);
    }
    getContinueToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getSemicolonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
}
export class ReturnStatementAST extends StatementAST {
    accept(visitor, context) {
        return visitor.visitReturnStatement(this, context);
    }
    getReturnToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getSemicolonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
}
export class GotoStatementAST extends StatementAST {
    accept(visitor, context) {
        return visitor.visitGotoStatement(this, context);
    }
    getGotoToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getIdentifierToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getSemicolonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
}
export class CoroutineReturnStatementAST extends StatementAST {
    accept(visitor, context) {
        return visitor.visitCoroutineReturnStatement(this, context);
    }
    getCoreturnToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getSemicolonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
}
export class DeclarationStatementAST extends StatementAST {
    accept(visitor, context) {
        return visitor.visitDeclarationStatement(this, context);
    }
    getDeclaration() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class TryBlockStatementAST extends StatementAST {
    accept(visitor, context) {
        return visitor.visitTryBlockStatement(this, context);
    }
    getTryToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getStatement() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    *getHandlerList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 2); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
}
export class AccessDeclarationAST extends DeclarationAST {
    accept(visitor, context) {
        return visitor.visitAccessDeclaration(this, context);
    }
    getAccessToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getColonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
}
export class FunctionDefinitionAST extends DeclarationAST {
    accept(visitor, context) {
        return visitor.visitFunctionDefinition(this, context);
    }
    *getAttributeList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 0); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    *getDeclSpecifierList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 1); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getDeclarator() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getRequiresClause() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
    getFunctionBody() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 4), this.parser);
    }
}
export class ConceptDefinitionAST extends DeclarationAST {
    accept(visitor, context) {
        return visitor.visitConceptDefinition(this, context);
    }
    getConceptToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getName() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getEqualToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
    getSemicolonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 4), this.parser);
    }
}
export class ForRangeDeclarationAST extends DeclarationAST {
    accept(visitor, context) {
        return visitor.visitForRangeDeclaration(this, context);
    }
}
export class AliasDeclarationAST extends DeclarationAST {
    accept(visitor, context) {
        return visitor.visitAliasDeclaration(this, context);
    }
    getUsingToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getIdentifierToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    *getAttributeList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 2); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getEqualToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
    getTypeId() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 4), this.parser);
    }
    getSemicolonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 5), this.parser);
    }
}
export class SimpleDeclarationAST extends DeclarationAST {
    accept(visitor, context) {
        return visitor.visitSimpleDeclaration(this, context);
    }
    *getAttributeList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 0); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    *getDeclSpecifierList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 1); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    *getInitDeclaratorList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 2); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getRequiresClause() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
    getSemicolonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 4), this.parser);
    }
}
export class StaticAssertDeclarationAST extends DeclarationAST {
    accept(visitor, context) {
        return visitor.visitStaticAssertDeclaration(this, context);
    }
    getStaticAssertToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getLparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getCommaToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
    getRparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 5), this.parser);
    }
    getSemicolonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 6), this.parser);
    }
}
export class EmptyDeclarationAST extends DeclarationAST {
    accept(visitor, context) {
        return visitor.visitEmptyDeclaration(this, context);
    }
    getSemicolonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class AttributeDeclarationAST extends DeclarationAST {
    accept(visitor, context) {
        return visitor.visitAttributeDeclaration(this, context);
    }
    *getAttributeList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 0); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getSemicolonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
}
export class OpaqueEnumDeclarationAST extends DeclarationAST {
    accept(visitor, context) {
        return visitor.visitOpaqueEnumDeclaration(this, context);
    }
    getEnumToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getClassToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    *getAttributeList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 2); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getNestedNameSpecifier() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
    getName() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 4), this.parser);
    }
    getEnumBase() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 5), this.parser);
    }
    getEmicolonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 6), this.parser);
    }
}
export class UsingEnumDeclarationAST extends DeclarationAST {
    accept(visitor, context) {
        return visitor.visitUsingEnumDeclaration(this, context);
    }
}
export class NamespaceDefinitionAST extends DeclarationAST {
    accept(visitor, context) {
        return visitor.visitNamespaceDefinition(this, context);
    }
    getInlineToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getNamespaceToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    *getAttributeList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 2); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getNestedNameSpecifier() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
    getName() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 4), this.parser);
    }
    *getExtraAttributeList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 5); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getLbraceToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 6), this.parser);
    }
    *getDeclarationList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 7); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getRbraceToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 8), this.parser);
    }
}
export class NamespaceAliasDefinitionAST extends DeclarationAST {
    accept(visitor, context) {
        return visitor.visitNamespaceAliasDefinition(this, context);
    }
    getNamespaceToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getIdentifierToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getEqualToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getNestedNameSpecifier() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
    getName() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 4), this.parser);
    }
    getSemicolonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 5), this.parser);
    }
}
export class UsingDirectiveAST extends DeclarationAST {
    accept(visitor, context) {
        return visitor.visitUsingDirective(this, context);
    }
    *getAttributeList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 0); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getUsingToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getNamespaceToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getNestedNameSpecifier() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
    getName() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 4), this.parser);
    }
    getSemicolonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 5), this.parser);
    }
}
export class UsingDeclarationAST extends DeclarationAST {
    accept(visitor, context) {
        return visitor.visitUsingDeclaration(this, context);
    }
    getUsingToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    *getUsingDeclaratorList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 1); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getSemicolonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
}
export class AsmDeclarationAST extends DeclarationAST {
    accept(visitor, context) {
        return visitor.visitAsmDeclaration(this, context);
    }
    *getAttributeList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 0); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getAsmToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getLparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getRparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 4), this.parser);
    }
    getSemicolonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 5), this.parser);
    }
}
export class ExportDeclarationAST extends DeclarationAST {
    accept(visitor, context) {
        return visitor.visitExportDeclaration(this, context);
    }
    getExportToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getDeclaration() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
}
export class ExportCompoundDeclarationAST extends DeclarationAST {
    accept(visitor, context) {
        return visitor.visitExportCompoundDeclaration(this, context);
    }
    getExportToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getLbraceToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    *getDeclarationList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 2); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getRbraceToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
}
export class ModuleImportDeclarationAST extends DeclarationAST {
    accept(visitor, context) {
        return visitor.visitModuleImportDeclaration(this, context);
    }
    getImportToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getImportName() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    *getAttributeList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 2); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getSemicolonToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
}
export class TemplateDeclarationAST extends DeclarationAST {
    accept(visitor, context) {
        return visitor.visitTemplateDeclaration(this, context);
    }
    getTemplateToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getLessToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    *getTemplateParameterList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 2); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getGreaterToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
    getRequiresClause() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 4), this.parser);
    }
    getDeclaration() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 5), this.parser);
    }
}
export class TypenameTypeParameterAST extends DeclarationAST {
    accept(visitor, context) {
        return visitor.visitTypenameTypeParameter(this, context);
    }
    getClassKeyToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getIdentifierToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getEqualToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getTypeId() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
}
export class TypenamePackTypeParameterAST extends DeclarationAST {
    accept(visitor, context) {
        return visitor.visitTypenamePackTypeParameter(this, context);
    }
    getClassKeyToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getEllipsisToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getIdentifierToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
}
export class TemplateTypeParameterAST extends DeclarationAST {
    accept(visitor, context) {
        return visitor.visitTemplateTypeParameter(this, context);
    }
    getTemplateToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getLessToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    *getTemplateParameterList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 2); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getGreaterToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
    getRequiresClause() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 4), this.parser);
    }
    getClassKeyToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 5), this.parser);
    }
    getIdentifierToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 6), this.parser);
    }
    getEqualToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 7), this.parser);
    }
    getName() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 8), this.parser);
    }
}
export class TemplatePackTypeParameterAST extends DeclarationAST {
    accept(visitor, context) {
        return visitor.visitTemplatePackTypeParameter(this, context);
    }
    getTemplateToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getLessToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    *getTemplateParameterList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 2); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getGreaterToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
    getClassKeyToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 4), this.parser);
    }
    getEllipsisToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 5), this.parser);
    }
    getIdentifierToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 6), this.parser);
    }
}
export class DeductionGuideAST extends DeclarationAST {
    accept(visitor, context) {
        return visitor.visitDeductionGuide(this, context);
    }
}
export class ExplicitInstantiationAST extends DeclarationAST {
    accept(visitor, context) {
        return visitor.visitExplicitInstantiation(this, context);
    }
    getExternToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getTemplateToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getDeclaration() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
}
export class ParameterDeclarationAST extends DeclarationAST {
    accept(visitor, context) {
        return visitor.visitParameterDeclaration(this, context);
    }
    *getAttributeList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 0); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    *getTypeSpecifierList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 1); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getDeclarator() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getEqualToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
    getExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 4), this.parser);
    }
}
export class LinkageSpecificationAST extends DeclarationAST {
    accept(visitor, context) {
        return visitor.visitLinkageSpecification(this, context);
    }
    getExternToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getStringliteralToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getLbraceToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    *getDeclarationList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 3); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getRbraceToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 4), this.parser);
    }
}
export class SimpleNameAST extends NameAST {
    accept(visitor, context) {
        return visitor.visitSimpleName(this, context);
    }
    getIdentifierToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class DestructorNameAST extends NameAST {
    accept(visitor, context) {
        return visitor.visitDestructorName(this, context);
    }
    getTildeToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getId() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
}
export class DecltypeNameAST extends NameAST {
    accept(visitor, context) {
        return visitor.visitDecltypeName(this, context);
    }
    getDecltypeSpecifier() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class OperatorNameAST extends NameAST {
    accept(visitor, context) {
        return visitor.visitOperatorName(this, context);
    }
    getOperatorToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getOpToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getOpenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getCloseToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
}
export class ConversionNameAST extends NameAST {
    accept(visitor, context) {
        return visitor.visitConversionName(this, context);
    }
    getOperatorToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getTypeId() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
}
export class TemplateNameAST extends NameAST {
    accept(visitor, context) {
        return visitor.visitTemplateName(this, context);
    }
    getId() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getLessToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    *getTemplateArgumentList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 2); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getGreaterToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
}
export class QualifiedNameAST extends NameAST {
    accept(visitor, context) {
        return visitor.visitQualifiedName(this, context);
    }
    getNestedNameSpecifier() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getTemplateToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getId() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
}
export class TypedefSpecifierAST extends SpecifierAST {
    accept(visitor, context) {
        return visitor.visitTypedefSpecifier(this, context);
    }
    getTypedefToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class FriendSpecifierAST extends SpecifierAST {
    accept(visitor, context) {
        return visitor.visitFriendSpecifier(this, context);
    }
    getFriendToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class ConstevalSpecifierAST extends SpecifierAST {
    accept(visitor, context) {
        return visitor.visitConstevalSpecifier(this, context);
    }
    getConstevalToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class ConstinitSpecifierAST extends SpecifierAST {
    accept(visitor, context) {
        return visitor.visitConstinitSpecifier(this, context);
    }
    getConstinitToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class ConstexprSpecifierAST extends SpecifierAST {
    accept(visitor, context) {
        return visitor.visitConstexprSpecifier(this, context);
    }
    getConstexprToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class InlineSpecifierAST extends SpecifierAST {
    accept(visitor, context) {
        return visitor.visitInlineSpecifier(this, context);
    }
    getInlineToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class StaticSpecifierAST extends SpecifierAST {
    accept(visitor, context) {
        return visitor.visitStaticSpecifier(this, context);
    }
    getStaticToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class ExternSpecifierAST extends SpecifierAST {
    accept(visitor, context) {
        return visitor.visitExternSpecifier(this, context);
    }
    getExternToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class ThreadLocalSpecifierAST extends SpecifierAST {
    accept(visitor, context) {
        return visitor.visitThreadLocalSpecifier(this, context);
    }
    getThreadLocalToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class ThreadSpecifierAST extends SpecifierAST {
    accept(visitor, context) {
        return visitor.visitThreadSpecifier(this, context);
    }
    getThreadToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class MutableSpecifierAST extends SpecifierAST {
    accept(visitor, context) {
        return visitor.visitMutableSpecifier(this, context);
    }
    getMutableToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class VirtualSpecifierAST extends SpecifierAST {
    accept(visitor, context) {
        return visitor.visitVirtualSpecifier(this, context);
    }
    getVirtualToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class ExplicitSpecifierAST extends SpecifierAST {
    accept(visitor, context) {
        return visitor.visitExplicitSpecifier(this, context);
    }
    getExplicitToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getLparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getRparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
}
export class AutoTypeSpecifierAST extends SpecifierAST {
    accept(visitor, context) {
        return visitor.visitAutoTypeSpecifier(this, context);
    }
    getAutoToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class VoidTypeSpecifierAST extends SpecifierAST {
    accept(visitor, context) {
        return visitor.visitVoidTypeSpecifier(this, context);
    }
    getVoidToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class VaListTypeSpecifierAST extends SpecifierAST {
    accept(visitor, context) {
        return visitor.visitVaListTypeSpecifier(this, context);
    }
    getSpecifierToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class IntegralTypeSpecifierAST extends SpecifierAST {
    accept(visitor, context) {
        return visitor.visitIntegralTypeSpecifier(this, context);
    }
    getSpecifierToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class FloatingPointTypeSpecifierAST extends SpecifierAST {
    accept(visitor, context) {
        return visitor.visitFloatingPointTypeSpecifier(this, context);
    }
    getSpecifierToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class ComplexTypeSpecifierAST extends SpecifierAST {
    accept(visitor, context) {
        return visitor.visitComplexTypeSpecifier(this, context);
    }
    getComplexToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class NamedTypeSpecifierAST extends SpecifierAST {
    accept(visitor, context) {
        return visitor.visitNamedTypeSpecifier(this, context);
    }
    getName() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class AtomicTypeSpecifierAST extends SpecifierAST {
    accept(visitor, context) {
        return visitor.visitAtomicTypeSpecifier(this, context);
    }
    getAtomicToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getLparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getTypeId() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getRparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
}
export class UnderlyingTypeSpecifierAST extends SpecifierAST {
    accept(visitor, context) {
        return visitor.visitUnderlyingTypeSpecifier(this, context);
    }
}
export class ElaboratedTypeSpecifierAST extends SpecifierAST {
    accept(visitor, context) {
        return visitor.visitElaboratedTypeSpecifier(this, context);
    }
    getClassToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    *getAttributeList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 1); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getNestedNameSpecifier() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getName() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
}
export class DecltypeAutoSpecifierAST extends SpecifierAST {
    accept(visitor, context) {
        return visitor.visitDecltypeAutoSpecifier(this, context);
    }
    getDecltypeToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getLparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getAutoToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getRparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
}
export class DecltypeSpecifierAST extends SpecifierAST {
    accept(visitor, context) {
        return visitor.visitDecltypeSpecifier(this, context);
    }
    getDecltypeToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getLparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getRparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
}
export class PlaceholderTypeSpecifierAST extends SpecifierAST {
    accept(visitor, context) {
        return visitor.visitPlaceholderTypeSpecifier(this, context);
    }
    getTypeConstraint() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getSpecifier() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
}
export class ConstQualifierAST extends SpecifierAST {
    accept(visitor, context) {
        return visitor.visitConstQualifier(this, context);
    }
    getConstToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class VolatileQualifierAST extends SpecifierAST {
    accept(visitor, context) {
        return visitor.visitVolatileQualifier(this, context);
    }
    getVolatileToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class RestrictQualifierAST extends SpecifierAST {
    accept(visitor, context) {
        return visitor.visitRestrictQualifier(this, context);
    }
    getRestrictToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
}
export class EnumSpecifierAST extends SpecifierAST {
    accept(visitor, context) {
        return visitor.visitEnumSpecifier(this, context);
    }
    getEnumToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getClassToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    *getAttributeList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 2); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getNestedNameSpecifier() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
    getName() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 4), this.parser);
    }
    getEnumBase() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 5), this.parser);
    }
    getLbraceToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 6), this.parser);
    }
    getCommaToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 7), this.parser);
    }
    *getEnumeratorList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 8); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getRbraceToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 9), this.parser);
    }
}
export class ClassSpecifierAST extends SpecifierAST {
    accept(visitor, context) {
        return visitor.visitClassSpecifier(this, context);
    }
    getClassToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    *getAttributeList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 1); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getName() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    getBaseClause() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 3), this.parser);
    }
    getLbraceToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 4), this.parser);
    }
    *getDeclarationList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 5); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    getRbraceToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 6), this.parser);
    }
}
export class TypenameSpecifierAST extends SpecifierAST {
    accept(visitor, context) {
        return visitor.visitTypenameSpecifier(this, context);
    }
    getTypenameToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getNestedNameSpecifier() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getName() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
}
export class IdDeclaratorAST extends CoreDeclaratorAST {
    accept(visitor, context) {
        return visitor.visitIdDeclarator(this, context);
    }
    getEllipsisToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getName() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    *getAttributeList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 2); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
}
export class NestedDeclaratorAST extends CoreDeclaratorAST {
    accept(visitor, context) {
        return visitor.visitNestedDeclarator(this, context);
    }
    getLparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getDeclarator() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getRparenToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
}
export class PointerOperatorAST extends PtrOperatorAST {
    accept(visitor, context) {
        return visitor.visitPointerOperator(this, context);
    }
    getStarToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    *getAttributeList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 1); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    *getCvQualifierList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 2); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
}
export class ReferenceOperatorAST extends PtrOperatorAST {
    accept(visitor, context) {
        return visitor.visitReferenceOperator(this, context);
    }
    getRefToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    *getAttributeList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 1); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
}
export class PtrToMemberOperatorAST extends PtrOperatorAST {
    accept(visitor, context) {
        return visitor.visitPtrToMemberOperator(this, context);
    }
    getNestedNameSpecifier() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getStarToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    *getAttributeList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 2); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
    *getCvQualifierList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 3); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
}
export class FunctionDeclaratorAST extends DeclaratorModifierAST {
    accept(visitor, context) {
        return visitor.visitFunctionDeclarator(this, context);
    }
    getParametersAndQualifiers() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getTrailingReturnType() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
}
export class ArrayDeclaratorAST extends DeclaratorModifierAST {
    accept(visitor, context) {
        return visitor.visitArrayDeclarator(this, context);
    }
    getLbracketToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 0), this.parser);
    }
    getExpression() {
        return AST.from(cxx.getASTSlot(this.getHandle(), 1), this.parser);
    }
    getRbracketToken() {
        return Token.from(cxx.getASTSlot(this.getHandle(), 2), this.parser);
    }
    *getAttributeList() {
        for (let it = cxx.getASTSlot(this.getHandle(), 3); it; it = cxx.getListNext(it)) {
            yield AST.from(cxx.getListValue(it), this.parser);
        }
    }
}
const AST_CONSTRUCTORS = [
    TypeIdAST,
    NestedNameSpecifierAST,
    UsingDeclaratorAST,
    HandlerAST,
    EnumBaseAST,
    EnumeratorAST,
    DeclaratorAST,
    InitDeclaratorAST,
    BaseSpecifierAST,
    BaseClauseAST,
    NewTypeIdAST,
    RequiresClauseAST,
    ParameterDeclarationClauseAST,
    ParametersAndQualifiersAST,
    LambdaIntroducerAST,
    LambdaDeclaratorAST,
    TrailingReturnTypeAST,
    CtorInitializerAST,
    RequirementBodyAST,
    TypeConstraintAST,
    GlobalModuleFragmentAST,
    PrivateModuleFragmentAST,
    ModuleDeclarationAST,
    ModuleNameAST,
    ImportNameAST,
    ModulePartitionAST,
    SimpleRequirementAST,
    CompoundRequirementAST,
    TypeRequirementAST,
    NestedRequirementAST,
    TypeTemplateArgumentAST,
    ExpressionTemplateArgumentAST,
    ParenMemInitializerAST,
    BracedMemInitializerAST,
    ThisLambdaCaptureAST,
    DerefThisLambdaCaptureAST,
    SimpleLambdaCaptureAST,
    RefLambdaCaptureAST,
    RefInitLambdaCaptureAST,
    InitLambdaCaptureAST,
    EqualInitializerAST,
    BracedInitListAST,
    ParenInitializerAST,
    NewParenInitializerAST,
    NewBracedInitializerAST,
    EllipsisExceptionDeclarationAST,
    TypeExceptionDeclarationAST,
    DefaultFunctionBodyAST,
    CompoundStatementFunctionBodyAST,
    TryStatementFunctionBodyAST,
    DeleteFunctionBodyAST,
    TranslationUnitAST,
    ModuleUnitAST,
    ThisExpressionAST,
    CharLiteralExpressionAST,
    BoolLiteralExpressionAST,
    IntLiteralExpressionAST,
    FloatLiteralExpressionAST,
    NullptrLiteralExpressionAST,
    StringLiteralExpressionAST,
    UserDefinedStringLiteralExpressionAST,
    IdExpressionAST,
    RequiresExpressionAST,
    NestedExpressionAST,
    RightFoldExpressionAST,
    LeftFoldExpressionAST,
    FoldExpressionAST,
    LambdaExpressionAST,
    SizeofExpressionAST,
    SizeofTypeExpressionAST,
    SizeofPackExpressionAST,
    TypeidExpressionAST,
    TypeidOfTypeExpressionAST,
    AlignofExpressionAST,
    TypeTraitsExpressionAST,
    UnaryExpressionAST,
    BinaryExpressionAST,
    AssignmentExpressionAST,
    BracedTypeConstructionAST,
    TypeConstructionAST,
    CallExpressionAST,
    SubscriptExpressionAST,
    MemberExpressionAST,
    PostIncrExpressionAST,
    ConditionalExpressionAST,
    ImplicitCastExpressionAST,
    CastExpressionAST,
    CppCastExpressionAST,
    NewExpressionAST,
    DeleteExpressionAST,
    ThrowExpressionAST,
    NoexceptExpressionAST,
    LabeledStatementAST,
    CaseStatementAST,
    DefaultStatementAST,
    ExpressionStatementAST,
    CompoundStatementAST,
    IfStatementAST,
    SwitchStatementAST,
    WhileStatementAST,
    DoStatementAST,
    ForRangeStatementAST,
    ForStatementAST,
    BreakStatementAST,
    ContinueStatementAST,
    ReturnStatementAST,
    GotoStatementAST,
    CoroutineReturnStatementAST,
    DeclarationStatementAST,
    TryBlockStatementAST,
    AccessDeclarationAST,
    FunctionDefinitionAST,
    ConceptDefinitionAST,
    ForRangeDeclarationAST,
    AliasDeclarationAST,
    SimpleDeclarationAST,
    StaticAssertDeclarationAST,
    EmptyDeclarationAST,
    AttributeDeclarationAST,
    OpaqueEnumDeclarationAST,
    UsingEnumDeclarationAST,
    NamespaceDefinitionAST,
    NamespaceAliasDefinitionAST,
    UsingDirectiveAST,
    UsingDeclarationAST,
    AsmDeclarationAST,
    ExportDeclarationAST,
    ExportCompoundDeclarationAST,
    ModuleImportDeclarationAST,
    TemplateDeclarationAST,
    TypenameTypeParameterAST,
    TypenamePackTypeParameterAST,
    TemplateTypeParameterAST,
    TemplatePackTypeParameterAST,
    DeductionGuideAST,
    ExplicitInstantiationAST,
    ParameterDeclarationAST,
    LinkageSpecificationAST,
    SimpleNameAST,
    DestructorNameAST,
    DecltypeNameAST,
    OperatorNameAST,
    ConversionNameAST,
    TemplateNameAST,
    QualifiedNameAST,
    TypedefSpecifierAST,
    FriendSpecifierAST,
    ConstevalSpecifierAST,
    ConstinitSpecifierAST,
    ConstexprSpecifierAST,
    InlineSpecifierAST,
    StaticSpecifierAST,
    ExternSpecifierAST,
    ThreadLocalSpecifierAST,
    ThreadSpecifierAST,
    MutableSpecifierAST,
    VirtualSpecifierAST,
    ExplicitSpecifierAST,
    AutoTypeSpecifierAST,
    VoidTypeSpecifierAST,
    VaListTypeSpecifierAST,
    IntegralTypeSpecifierAST,
    FloatingPointTypeSpecifierAST,
    ComplexTypeSpecifierAST,
    NamedTypeSpecifierAST,
    AtomicTypeSpecifierAST,
    UnderlyingTypeSpecifierAST,
    ElaboratedTypeSpecifierAST,
    DecltypeAutoSpecifierAST,
    DecltypeSpecifierAST,
    PlaceholderTypeSpecifierAST,
    ConstQualifierAST,
    VolatileQualifierAST,
    RestrictQualifierAST,
    EnumSpecifierAST,
    ClassSpecifierAST,
    TypenameSpecifierAST,
    IdDeclaratorAST,
    NestedDeclaratorAST,
    PointerOperatorAST,
    ReferenceOperatorAST,
    PtrToMemberOperatorAST,
    FunctionDeclaratorAST,
    ArrayDeclaratorAST,
];
//# sourceMappingURL=AST.js.map